import abc
from collections.abc import Sequence
from typing import Any, Generic, Literal, Never, Protocol, Self, SupportsIndex, final, overload, type_check_only
from typing_extensions import TypeVar, override

import numpy as np
import optype.numpy as onp
import optype.numpy.compat as npc

from ._base import _spbase, sparray
from ._coo import coo_array, coo_matrix
from ._matrix import spmatrix
from ._sputils import _ScalarLike

__all__: list[str] = []

_ScalarT = TypeVar("_ScalarT", bound=npc.number | np.bool_)
_ScalarT_co = TypeVar("_ScalarT_co", bound=npc.number | np.bool_, default=Any, covariant=True)
_ShapeT_co = TypeVar("_ShapeT_co", bound=onp.AtLeast1D, default=onp.AtLeast0D[Any], covariant=True)

###

_DataMatrixT = TypeVar("_DataMatrixT", bound=_data_matrix[Any, Any])
_DataMatrixT_co = TypeVar("_DataMatrixT_co", bound=_data_matrix[Any, Any], covariant=True)

@final
@type_check_only
class _CanAsFloat32(Protocol[_DataMatrixT_co]):
    @property
    def dtype(self, /) -> np.dtype[npc.integer16]: ...
    @type_check_only
    def __assoc_as_float32__(self, /) -> _DataMatrixT_co: ...

@final
@type_check_only
class _CanAsFloat64(Protocol[_DataMatrixT_co]):
    @property
    def dtype(self, /) -> np.dtype[npc.integer32 | npc.integer64]: ...
    @type_check_only
    def __assoc_as_float64__(self, /) -> _DataMatrixT_co: ...

###

class _data_matrix(_spbase[_ScalarT_co, _ShapeT_co], Generic[_ScalarT_co, _ShapeT_co]):
    # NOTE: These two methods do not exist at runtime.
    @type_check_only
    def __assoc_as_float32__(self, /) -> _data_matrix[np.float32, _ShapeT_co]: ...
    @type_check_only
    def __assoc_as_float64__(self, /) -> _data_matrix[np.float64, _ShapeT_co]: ...

    #
    @property
    @abc.abstractmethod
    @override
    def format(self, /) -> Literal["bsr", "coo", "csc", "csr", "dia"]: ...

    #
    @property
    def dtype(self, /) -> np.dtype[_ScalarT_co]: ...
    @dtype.setter
    def dtype(self, newtype: Never, /) -> None: ...

    #
    @overload
    def __init__(
        self, /, arg1: _spbase[_ScalarT_co, _ShapeT_co] | onp.CanArrayND[_ScalarT_co, _ShapeT_co], *, maxprint: int | None = None
    ) -> None: ...
    @overload
    def __init__(
        self: _data_matrix[_ScalarT, tuple[int]], /, arg1: Sequence[_ScalarT], *, maxprint: int | None = None
    ) -> None: ...
    @overload
    def __init__(
        self: _data_matrix[_ScalarT, tuple[int, int]],
        /,
        arg1: Sequence[onp.CanArrayND[_ScalarT] | Sequence[_ScalarT]],
        *,
        maxprint: int | None = None,
    ) -> None: ...
    @overload
    def __init__(
        self: _data_matrix[np.float64, tuple[int]], /, arg1: tuple[SupportsIndex], *, maxprint: int | None = None
    ) -> None: ...
    @overload
    def __init__(
        self: _data_matrix[np.float64, tuple[int, int]],
        /,
        arg1: tuple[SupportsIndex, SupportsIndex],
        *,
        maxprint: int | None = None,
    ) -> None: ...
    @overload
    def __init__(self, /, arg1: onp.CanArrayND[_ScalarT_co], *, maxprint: int | None = None) -> None: ...

    #
    @override
    def __imul__(self, rhs: _ScalarLike, /) -> Self: ...  # type: ignore[override]
    @override
    def __itruediv__(self, rhs: _ScalarLike, /) -> Self: ...  # type: ignore[override]

    # NOTE: The following methods do not convert the scalar type
    def sign(self, /) -> Self: ...
    def ceil(self, /) -> Self: ...
    def floor(self, /) -> Self: ...
    def rint(self, /) -> Self: ...
    def trunc(self, /) -> Self: ...

    #
    @overload
    def sqrt(self: _CanAsFloat32[_DataMatrixT], /) -> _DataMatrixT: ...
    @overload
    def sqrt(self: _CanAsFloat64[_DataMatrixT], /) -> _DataMatrixT: ...
    @overload
    def sqrt(self, /) -> Self: ...

    #
    @overload
    def expm1(self: _CanAsFloat32[_DataMatrixT], /) -> _DataMatrixT: ...
    @overload
    def expm1(self: _CanAsFloat64[_DataMatrixT], /) -> _DataMatrixT: ...
    @overload
    def expm1(self, /) -> Self: ...
    #
    @overload
    def log1p(self: _CanAsFloat32[_DataMatrixT], /) -> _DataMatrixT: ...
    @overload
    def log1p(self: _CanAsFloat64[_DataMatrixT], /) -> _DataMatrixT: ...
    @overload
    def log1p(self, /) -> Self: ...

    #
    @overload
    def sin(self: _CanAsFloat32[_DataMatrixT], /) -> _DataMatrixT: ...
    @overload
    def sin(self: _CanAsFloat64[_DataMatrixT], /) -> _DataMatrixT: ...
    @overload
    def sin(self, /) -> Self: ...
    #
    @overload
    def arcsin(self: _CanAsFloat32[_DataMatrixT], /) -> _DataMatrixT: ...
    @overload
    def arcsin(self: _CanAsFloat64[_DataMatrixT], /) -> _DataMatrixT: ...
    @overload
    def arcsin(self, /) -> Self: ...
    #
    @overload
    def sinh(self: _CanAsFloat32[_DataMatrixT], /) -> _DataMatrixT: ...
    @overload
    def sinh(self: _CanAsFloat64[_DataMatrixT], /) -> _DataMatrixT: ...
    @overload
    def sinh(self, /) -> Self: ...
    #
    @overload
    def arcsinh(self: _CanAsFloat32[_DataMatrixT], /) -> _DataMatrixT: ...
    @overload
    def arcsinh(self: _CanAsFloat64[_DataMatrixT], /) -> _DataMatrixT: ...
    @overload
    def arcsinh(self, /) -> Self: ...
    #
    @overload
    def tan(self: _CanAsFloat32[_DataMatrixT], /) -> _DataMatrixT: ...
    @overload
    def tan(self: _CanAsFloat64[_DataMatrixT], /) -> _DataMatrixT: ...
    @overload
    def tan(self, /) -> Self: ...
    #
    @overload
    def arctan(self: _CanAsFloat32[_DataMatrixT], /) -> _DataMatrixT: ...
    @overload
    def arctan(self: _CanAsFloat64[_DataMatrixT], /) -> _DataMatrixT: ...
    @overload
    def arctan(self, /) -> Self: ...
    #
    @overload
    def tanh(self: _CanAsFloat32[_DataMatrixT], /) -> _DataMatrixT: ...
    @overload
    def tanh(self: _CanAsFloat64[_DataMatrixT], /) -> _DataMatrixT: ...
    @overload
    def tanh(self, /) -> Self: ...
    #
    @overload
    def arctanh(self: _CanAsFloat32[_DataMatrixT], /) -> _DataMatrixT: ...
    @overload
    def arctanh(self: _CanAsFloat64[_DataMatrixT], /) -> _DataMatrixT: ...
    @overload
    def arctanh(self, /) -> Self: ...

    #
    @overload
    def deg2rad(self: _CanAsFloat32[_DataMatrixT], /) -> _DataMatrixT: ...
    @overload
    def deg2rad(self: _CanAsFloat64[_DataMatrixT], /) -> _DataMatrixT: ...
    @overload
    def deg2rad(self, /) -> Self: ...
    #
    @overload
    def rad2deg(self: _CanAsFloat32[_DataMatrixT], /) -> _DataMatrixT: ...
    @overload
    def rad2deg(self: _CanAsFloat64[_DataMatrixT], /) -> _DataMatrixT: ...
    @overload
    def rad2deg(self, /) -> Self: ...

class _minmax_mixin(Generic[_ScalarT_co, _ShapeT_co]):
    # NOTE: The following 4 methods have identical signatures
    @overload  # axis: None = ..., out: None = ...
    def max(self, /, axis: None = None, out: None = None, *, explicit: bool = False) -> _ScalarT_co: ...
    @overload  # 1-d, axis: int, out: None = ...
    def max(
        self: _minmax_mixin[Any, tuple[int]], /, axis: onp.ToInt | None = None, out: None = None, *, explicit: bool = False
    ) -> _ScalarT_co: ...
    @overload  # sparray, axis: int, out: None = ...
    def max(  # type: ignore[misc]
        self: sparray, /, axis: onp.ToInt, out: None = None, *, explicit: bool = False
    ) -> coo_array[_ScalarT_co, tuple[int]]: ...
    @overload  # spmatrix, axis: int, out: None = ...
    def max(self: spmatrix, /, axis: onp.ToInt, out: None = None, *, explicit: bool = False) -> coo_matrix[_ScalarT_co]: ...  # type: ignore[misc]
    #
    @overload  # axis: None = ..., out: None = ...
    def nanmax(self, /, axis: None = None, out: None = None, *, explicit: bool = False) -> _ScalarT_co: ...
    @overload  # 1-d, axis: int, out: None = ...
    def nanmax(
        self: _minmax_mixin[Any, tuple[int]], /, axis: onp.ToInt | None = None, out: None = None, *, explicit: bool = False
    ) -> _ScalarT_co: ...
    @overload  # sparray, axis: int, out: None = ...
    def nanmax(  # type: ignore[misc]
        self: sparray, /, axis: onp.ToInt, out: None = None, *, explicit: bool = False
    ) -> coo_array[_ScalarT_co, tuple[int]]: ...
    @overload  # spmatrix, axis: int, out: None = ...
    def nanmax(self: spmatrix, /, axis: onp.ToInt, out: None = None, *, explicit: bool = False) -> coo_matrix[_ScalarT_co]: ...  # type: ignore[misc]
    #
    @overload  # axis: None = ..., out: None = ...
    def min(self, /, axis: None = None, out: None = None, *, explicit: bool = False) -> _ScalarT_co: ...
    @overload  # 1-d, axis: int, out: None = ...
    def min(
        self: _minmax_mixin[Any, tuple[int]], /, axis: onp.ToInt | None = None, out: None = None, *, explicit: bool = False
    ) -> _ScalarT_co: ...
    @overload  # sparray, axis: int, out: None = ...
    def min(  # type: ignore[misc]
        self: sparray, /, axis: onp.ToInt, out: None = None, *, explicit: bool = False
    ) -> coo_array[_ScalarT_co, tuple[int]]: ...
    @overload  # spmatrix, axis: int, out: None = ...
    def min(self: spmatrix, /, axis: onp.ToInt, out: None = None, *, explicit: bool = False) -> coo_matrix[_ScalarT_co]: ...  # type: ignore[misc]
    #
    @overload  # axis: None = ..., out: None = ...
    def nanmin(self, /, axis: None = None, out: None = None, *, explicit: bool = False) -> _ScalarT_co: ...
    @overload  # 1-d, axis: int, out: None = ...
    def nanmin(
        self: _minmax_mixin[Any, tuple[int]], /, axis: onp.ToInt | None = None, out: None = None, *, explicit: bool = False
    ) -> _ScalarT_co: ...
    @overload  # sparray, axis: int, out: None = ...
    def nanmin(  # type: ignore[misc]
        self: sparray, /, axis: onp.ToInt, out: None = None, *, explicit: bool = False
    ) -> coo_array[_ScalarT_co, tuple[int]]: ...
    @overload  # spmatrix, axis: int, out: None = ...
    def nanmin(self: spmatrix, /, axis: onp.ToInt, out: None = None, *, explicit: bool = False) -> coo_matrix[_ScalarT_co]: ...  # type: ignore[misc]

    # NOTE: The following 2 methods have identical signatures
    @overload  # axis: None = ..., out: None = ...
    def argmax(self, /, axis: None = None, out: None = None, *, explicit: bool = False) -> int: ...
    @overload  # 1-d, axis: int, out: None = ...
    def argmax(
        self: _minmax_mixin[Any, tuple[int]], /, axis: onp.ToInt | None = None, out: None = None, *, explicit: bool = False
    ) -> int: ...
    @overload  # sparray, axis: int, out: None = ...
    def argmax(self: sparray, /, axis: onp.ToInt, out: None = None, *, explicit: bool = False) -> onp.Array1D[np.intp]: ...  # type: ignore[misc]
    @overload  # spmatrix, axis: int, out: None = ...
    def argmax(self: spmatrix, /, axis: onp.ToInt, out: None = None, *, explicit: bool = False) -> onp.Matrix[np.intp]: ...  # type: ignore[misc]

    #
    @overload  # axis: None = ..., out: None = ...
    def argmin(self, /, axis: None = None, out: None = None, *, explicit: bool = False) -> int: ...
    @overload  # 1-d, axis: int, out: None = ...
    def argmin(
        self: _minmax_mixin[Any, tuple[int]], /, axis: onp.ToInt | None = None, out: None = None, *, explicit: bool = False
    ) -> int: ...
    @overload  # sparray, axis: int, out: None = ...
    def argmin(self: sparray, /, axis: onp.ToInt, out: None = None, *, explicit: bool = False) -> onp.Array1D[np.intp]: ...  # type: ignore[misc]
    @overload  # spmatrix, axis: int, out: None = ...
    def argmin(self: spmatrix, /, axis: onp.ToInt, out: None = None, *, explicit: bool = False) -> onp.Matrix[np.intp]: ...  # type: ignore[misc]
