import unittest
import os
from pathlib import Path
import shutil
import ast
from datetime import datetime
import pandas as pd
import pyarrow as pa
import pyarrow.compute as pc
import pyarrow.parquet as pq
from ds_capability.intent.feature_build_intent import FeatureBuildIntent
from ds_core.properties.property_manager import PropertyManager
from ds_capability import *
from ds_capability.components.commons import Commons
from ds_capability.intent.common_intent import AnalysisOptions

# Pandas setup
pd.set_option('max_colwidth', 320)
pd.set_option('display.max_rows', 100)
pd.set_option('display.max_columns', 99)
pd.set_option('expand_frame_repr', True)


class AbstractTest(unittest.TestCase):

    @classmethod
    def setUpClass(cls):
        pass

    @classmethod
    def tearDownClass(cls):
        pass

    def setUp(self):
        # clean out any old environments
        for key in os.environ.keys():
            if key.startswith('HADRON'):
                del os.environ[key]
        # Local Domain Contract
        os.environ['HADRON_PM_PATH'] = os.path.join('working', 'contracts')
        os.environ['HADRON_PM_TYPE'] = 'json'
        # Local Connectivity
        os.environ['HADRON_DEFAULT_PATH'] = Path('working/data').as_posix()
        # Specialist Component
        try:
            os.makedirs(os.environ['HADRON_PM_PATH'])
        except OSError:
            pass
        try:
            os.makedirs(os.environ['HADRON_DEFAULT_PATH'])
        except OSError:
            pass
        try:
            shutil.copytree('../_test_data', os.path.join(os.environ['PWD'], 'working/source'))
        except OSError:
            pass
        PropertyManager._remove_all()

    def tearDown(self):
        try:
            shutil.rmtree('working')
        except OSError:
            pass

    def test_get_mask(self):
        fb = FeatureBuild.from_memory()
        tools: FeatureBuildIntent = fb.tools
        tbl = pa.table([pa.array([1,2,1,4,None,-1,7]), pa.array(list('abbbcad'))], names=['A','B'])
        mask = tools._extract_mask(tbl.column('A'), condition=(1, 'equal', None))
        self.assertEqual(2, pc.sum(mask).as_py())
        self.assertEqual(1, mask.null_count)
        mask = tools._extract_mask(tbl.column('A'), condition=(1, 'equal', None), mask_null=True)
        self.assertEqual(2, pc.sum(mask).as_py())
        self.assertEqual(0, mask.null_count)
        # multiple
        condition = [(2, 'greater', 'or_'),(0, 'less', None)]
        mask = tools._extract_mask(tbl.column('A'), condition=condition)
        self.assertEqual(3, pc.sum(mask).as_py())
        self.assertEqual(1, mask.null_count)
        # wrong compare type
        with self.assertRaises(ValueError) as context:
            mask = tools._extract_mask(tbl.column('B'), condition=(1, 'equal', None))
        self.assertTrue("The operator 'equal' is not supported for data type 'string'." in str(context.exception))

    def test_analysis_options(self):
        opt = AnalysisOptions()
        opt.add_option(name='opt1', var=23, order=True, type='int')
        result = opt.get_option('opt1')
        print(result)
        result = opt.options
        print(result)


    def test_raise(self):
        startTime = datetime.now()
        with self.assertRaises(KeyError) as context:
            env = os.environ['NoEnvValueTest']
        self.assertTrue("'NoEnvValueTest'" in str(context.exception))
        print(f"Duration - {str(datetime.now() - startTime)}")


def pm_view(capability: str, task: str, section: str = None):
    uri = os.path.join(os.environ['HADRON_PM_PATH'], f"hadron_pm_{capability}_{task}.parquet")
    tbl = pq.read_table(uri)
    tbl = tbl.column(0).combine_chunks()
    result = ast.literal_eval(tbl.to_pylist()[0]).get(capability, {}).get(task, {})
    return result.get(section, {}) if isinstance(section, str) and section in result.keys() else result


def tprint(t: pa.table, headers: [str, list] = None, d_type: [str, list] = None, regex: [str, list] = None):
    _ = Commons.filter_columns(t.slice(0, 10), headers=headers, d_types=d_type, regex=regex)
    print(Commons.table_report(_).to_string())


if __name__ == '__main__':
    unittest.main()
