'''
Module: fastq.py
Author: Marc Zepeda
Created: 2024-08-05
Description: Fastq processing and analysis 

Usage:
[Supporting methods for sequences]
- fuzzy_substring_search: retuns a dataframe containing all substrings in 'text' that resemble 'pattern' within the Levenshtein 'max_distance'.

[Input/Output]
- revcom_fastqs(): write reverse complement of fastqs to a new directory
- unzip_fastqs(): Unzip gzipped fastqs and write to a new directory
- comb_fastqs(): Combines one or more (un)compressed fastqs files into a single (un)compressed fastq file

[Quantify epeg/ngRNA or Signature abundance]
- [Motif search: mU6, ..., & target sequence flanks]
    - count_motif(): returns a dataframe with the sequence motif location per read and abundance for every fastq file in a directory
    - plot_motif(): generate plots highlighting motif mismatches, locations, and sequences
- [Region/read alignments: spacer, ..., & ngRNA-epegRNA]
    - mismatch_alignments(): compute & save mismatch number and position per alignment; enables checkpoints
    - perform_alignments(): perform alignments on fastq reads using PairwiseAligner and compute mismatches using mismatch_alignments()
    - plot_alignments(): generate line & distribution plots from fastq alignments dictionary
    - count_region(): align read region from fastq directory to the annotated library with mismatches; plot and return fastq alignments dictionary
    - count_alignments(): align reads from fastq directory to annotated library with mismatches; plot and return fastq alignments dictionary
    - plot_paired(): generate stacked bar plots from paired_regions() dataframe
    - paired_regions(): quantify, plot, & return (un)paired regions that aligned to the annotated library
- [Signature]
    - count_signatures(): generate signatures from fastq read region alignments to WT sequence; count signatures, plot and return fastq signatures dataframe

[Quantify edit outcomes]
- trim_filter(): trim and filter fastq sequence based on quality scores
- get_fastqs(): get fastq files from directory and store records in dataframes in a dictionary
- region(): gets DNA and AA sequence for records within flanks
- [Supporting methods for genotype()]
    - find_AA_edits(): find amino acid edits compared to wildtype sequence
    - trim(): trim the sequence to a multiple of 3.
    - format_alignment(): formats two sequences for alignment display & return the middle.
    - find_indel(): aligns two sequences and returns the indel edit.
- genotype(): assign genotypes to sequence records
- outcomes(): returns edit count & fraction per sample (tidy format)
- genotyping(): quantify edit outcomes workflow
- abundances(): quantify desired edits count & fraction per sample
- editing_per_library(): Determine editing relative library abundance

[UMI methods]
- extract_umis(): extract UMIs using umi_tools
- trim_motifs(): trimming motifs with cutadapt
- make_sams(): generates alignments saved as a SAM files using bowtie2
- make_bams(): converts SAM files to BAM files using samtools
- bam_umi_tags(): copy UMI in read ID to RX tag in BAM files using fgbio
- group_umis(): group BAM files by UMI using fgbio
- consensus_umis(): generate consensus sequences from grouped UMIs using fgbio
- bam_to_fastq(): convert BAM files to fastq files using samtools
# Note: Consider adding fgbio FilterConsensusReads & fgbio ReviewConsensusVariants

[Supporting methods for DMS plots]
- aa_props: dictionary of AA properties with citations (Generated by ChatGPT)
- edit_1(): split edit column to before, after, and amino acid number
- dms_cond(): returns DMS grid data in tidy format grouped by condition
- dms_comp(): returns comparison DMS grid dataframe in tidy format split by condition
- subscript(): returns dataframe with subscripts to tick labels

[Plot methods]
- scat(): creates scatter plot related graphs
- cat: creates category dependent graphs
- stack(): creates stacked bar plot
- heat(): creates heatmap
- vol(): creates volcano plot
'''

# Import packages
from Bio.Seq import Seq
from Bio import SeqIO
from Bio.Align import PairwiseAligner
from Bio.SeqRecord import SeqRecord
import gzip
import os
import re
import datetime
import pandas as pd
import numpy as np
import matplotlib.pyplot as plt
import seaborn as sns
from adjustText import adjust_text
from scipy.stats import ttest_ind
import Levenshtein
from typing import Literal
import math
import subprocess

from ..bio.signature import parse_signature_literal, signature_from_alignment
from ..gen import io
from ..gen import tidy as t
from ..gen import plot as p
from ..utils import memory_timer
from .. import config

# Supporting methods for sequences
def fuzzy_substring_search(text: str, pattern: str, max_distance: int):
    """
    fuzzy_substring_search: retuns a dataframe containing all substrings in 'text' that resemble 'pattern' within the Levenshtein 'max_distance'.
    
    Parameters:
    text (str): text to search within
    pattern (str): text to search for
    max_distance (int): maximum Levenshtein distance of 'pattern' in text substring

    Dependencies: Levenshtein, pandas
    """
    # Initialize lists that will be stored in the output dataframe
    windows = []
    distances = []
    starts_i = []
    ends_i = []

    # Iterate through 'text' using windows with equal length to 'pattern'
    len_pat = len(pattern)
    for i in range(len(text) - len_pat + 1):
        window = text[i:i + len_pat]
        distance = Levenshtein.distance(window, pattern)
        if distance <= max_distance: # Save windows that resemble 'pattern' within the Levenshtein 'max_distance'
            windows.append(window)
            distances.append(distance)
            starts_i.append(int(i))
            ends_i.append(int(i + len_pat))

    # Return final dataframe
    return pd.DataFrame({'pattern': [pattern]*len(windows),
                         'window': windows,
                         'distance': distances,
                         'start_i': starts_i,
                         'end_i': ends_i})

# Input/Output
def revcom_fastqs(in_dir: str, out_dir: str):
    ''' 
    revcom_fastqs(): write reverse complement of fastqs to a new directory
    
    Parameters:
    in_dir (str): directory with fastq files
    out_dir (str): new directory with reverse complement fastq files
    
    Dependencies: Bio.SeqIO, gzip, os, & Bio.Seq.Seq
    '''
    io.mkdir(out_dir) # Ensure the output directory exists

    for filename in os.listdir(in_dir): # Find all .fastq.gz & .fastq files in the input directory

        if filename.endswith(".fastq.gz"):
            input_fastq_gz = os.path.join(in_dir, filename)
            output_fastq_gz = os.path.join(out_dir, filename)
            print(f"Processing {filename}...")
            with gzip.open(input_fastq_gz, "rt") as infile, gzip.open(output_fastq_gz, "wt") as outfile:
                for record in SeqIO.parse(infile, "fastq"):
                    reverse_complement_seq = record.seq.reverse_complement() # Compute the reverse complement of the sequence
                    reverse_complement_record = record[:] # Create a new record with the reverse complement sequence
                    reverse_complement_record.seq = reverse_complement_seq # Write the new record to the output file
                    SeqIO.write(reverse_complement_record, outfile, "fastq")
            print(f"Saved reverse complement to {output_fastq_gz}")
        
        elif filename.endswith(".fastq"):
            input_fastq = os.path.join(in_dir, filename)
            output_fastq = os.path.join(out_dir, filename)
            print(f"Processing {filename}...")
            with open(input_fastq, "r") as infile, open(output_fastq, "w") as outfile:
                for record in SeqIO.parse(infile, "fastq"):
                    reverse_complement_seq = record.seq.reverse_complement() # Compute the reverse complement of the sequence
                    reverse_complement_record = record[:] # Create a new record with the reverse complement sequence
                    reverse_complement_record.seq = reverse_complement_seq # Write the new record to the output file
                    SeqIO.write(reverse_complement_record, outfile, "fastq")
            print(f"Saved reverse complement to {output_fastq_gz}")

def unzip_fastqs(in_dir: str, out_dir: str):
    ''' 
    unzip_fastqs(): Unzip gzipped fastqs and write to a new directory

    Parameters:
    in_dir (str): directory with compresesd fastq files
    out_dir (str): new directory with uncompressed fastq files
    
    Dependencies: gzip & os
    '''
    io.mkdir(out_dir) # Ensure the output directory exists

    for in_file in os.listdir(in_dir): # Find all .fastq.gz & .fastq files in the input directory
        print(f"Processing {in_file}...")
        if in_file.endswith(".fastq.gz"):
            with gzip.open(os.path.join(in_dir,in_file), 'rt') as handle:
                with open(os.path.join(out_dir,in_file.split('.fastq.gz')[0]+'.fastq'), 'wt') as out:
                    for line in handle:
                        out.write(line)

def comb_fastqs(in_dir: str, out_dir: str, out_file: str):
    ''' 
    comb_fastqs(): Combines one or more (un)compressed fastqs files into a single (un)compressed fastq file

    Parameters:
    in_dir (str): directory with fastq files
    out_dir (str): new directory with combined fastq file
    out_file (str): Name of output fastq file (Needs .fastq or .fastq.gz suffix)
    
    Dependencies: gzip & os
    '''
    io.mkdir(out_dir) # Ensure the output directory exists

    if out_file.endswith(".fastq.gz"):
        with gzip.open(os.path.join(out_dir,out_file), 'wt') as out:
            for in_file in os.listdir(in_dir): # Find all .fastq.gz & .fastq files in the input directory
                print(f"Processing {in_file}...")
                if in_file.endswith(".fastq.gz"):
                    with gzip.open(os.path.join(in_dir,in_file), 'rt') as handle:
                        for line in handle:
                            out.write(line)
                
                elif in_file.endswith(".fastq"):
                    with open(os.path.join(in_dir,in_file), 'r') as handle:
                        for line in handle:
                            out.write(line)
    
    elif out_file.endswith(".fastq"):
        with open(os.path.join(out_dir,out_file), 'wt') as out:
            for in_file in os.listdir(in_dir): # Find all .fastq.gz & .fastq files in the input directory
                print(f"Processing {in_file}...")
                if in_file.endswith(".fastq.gz"):
                    with gzip.open(os.path.join(in_dir,in_file), 'rt') as handle:
                        for line in handle:
                            out.write(line)
                
                elif in_file.endswith(".fastq"):
                    with open(os.path.join(in_dir,in_file), 'r') as f:
                        for line in handle:
                            out.write(line)

    else: print('out_file needs .fastq or .fastq.gz suffix')

# Quantify epeg/ngRNA abundance
### Motif search: mU6,...
def count_motif(fastq_dir: str, pattern: str, out_dir: str, motif: str="motif", 
                max_distance:int=0, max_reads:int=0, meta: pd.DataFrame | str=None,
                return_df:bool=False) -> pd.DataFrame:
    ''' 
    count_motif(): returns a dataframe with the sequence motif location with mismatches per read for every fastq file in a directory

    Parameters:
    fastq_dir (str): path to fastq directory
    pattern (str): search for this sequence motif
    out_dir (str): path to save directory
    motif (str, optional): motif name (Default: 'motif')
    max_distance (int, optional): max Levenstein distance for seq in fastq read (Default: 0)
    meta (DataFrame | str, optional): meta dataframe (or file path) must have 'fastq_file' column (Default: None)
    return_df (bool, optional): return dataframe (Default: False)

    Dependencies: pandas, gzip, os, Bio, fuzzy_substring_search() & memory_timer()
    '''
    # Initialize timer; memory & stats reporting
    memory_timer(reset=True)
    memories = []
    stats = []

    # Create a dataframe to store motif abundance
    df = pd.DataFrame(columns=['fastq_file','read','motif','pattern','window','start_i','end_i','distance'])
    for fastq_file in os.listdir(fastq_dir): # Find all .fastq.gz & .fastq files in the fastq directory
        
        print(f"Processing {fastq_file}...") # Keep track of sequence motifs & reads
        has_motif = 0
        missing_motif = 0
        reads = pd.DataFrame() 

        if fastq_file.endswith(".fastq.gz"): # Compressed fastq
            fastq_name = fastq_file[:-len(".fastq.gz")] # Get fastq name
            with gzip.open(os.path.join(fastq_dir,fastq_file), 'rt') as handle:
                for r,record in enumerate(SeqIO.parse(handle, "fastq")): # Parse reads
                    
                    # Find all substrings that resemble 'pattern' within the Levenshtein 'max_distance'.
                    read = fuzzy_substring_search(text=str(record.seq),pattern=pattern,max_distance=max_distance)
                    
                    # Retain the substring with the smallest Levenshtein distance to 'pattern'. 
                    if len(read)==0: # No substring
                        missing_motif += 1
                        read.loc[0] = [pattern, "N"*len(pattern), -1, -1, -1]
                    else: # Found substring(s)
                        has_motif += 1
                        read = read[read['distance']==min(read['distance'])] # Smallest Levenshtein distance
                        read = read.iloc[:1] # Isolate first instance 
                    read['read'] = [r+1] # Add read index
                    reads = pd.concat([reads,read]).reset_index(drop=True)

                    # Processing status and down sample to reduce computation
                    if len(reads)%10000==0: 
                        print(f"Processed {len(reads)} reads")
                    if (len(reads)+1>max_reads) & (max_reads!=0): 
                        break
                 
        elif fastq_file.endswith(".fastq"): # Uncompressed fastq
            fastq_name = fastq_file[:-len(".fastq")] # Get fastq name
            with open(os.path.join(fastq_dir,fastq_file), 'r') as handle:
                for r,record in enumerate(SeqIO.parse(handle, "fastq")): # Parse reads 

                    # Find all substrings that resemble 'pattern' within the Levenshtein 'max_distance'.
                    read = fuzzy_substring_search(text=str(record.seq),pattern=pattern,max_distance=max_distance)
                    
                    # Retain the substring with the smallest Levenshtein distance to 'pattern'. 
                    if len(read)==0: # No substring
                        missing_motif += 1
                        read.loc[0] = [pattern, "N"*len(pattern), -1, -1, -1]
                    else: # Found substring(s)
                        has_motif += 1
                        read = read[read['distance']==min(read['distance'])] # Smallest Levenshtein distance
                        read = read.iloc[:1] # Isolate first instance 
                    read['read'] = [r+1] # Add read index
                    reads = pd.concat([reads,read]).reset_index(drop=True)

                    # Processing status and down sample to reduce computation
                    if len(reads)%10000==0: 
                        print(f"Processed {len(reads)} reads")
                    if (len(reads)+1>max_reads) & (max_reads!=0): 
                        break
        
        else: # Not a fastq file
            print("Not a fastq file")
            continue         
        
        # Update memories & stats: # of reads with(out) the region
        print(f'{fastq_name}:\t{len(reads)} reads\t=>\t{has_motif} has motif;\t{missing_motif} missing motif')
        stats.append((fastq_name,len(reads),has_motif,missing_motif))
        memories.append(memory_timer(task=f"{fastq_file} (motif)"))
        
        # Append metadata...
        reads['fastq_file'] = [fastq_file]*len(reads)
        reads['motif'] = [motif]*len(reads)
         
        if meta is not None: # ...and merge with meta
            if type(meta)==str: # Get from file path if needed
                meta = io.get(pt=meta)
            
            if 'fastq_file' not in list(meta.columns): # Check for 'fastq_file' column
                print(f"Warning: Did not merge with meta.\nmeta needs 'fastq_file' column.\nDetected columns: {list(meta.columns)}")
            else: # Merge on 'fastq_file' column
                reads = pd.merge(left=meta,right=reads,on='fastq_file')

        # Improve dataframe column formatting 
        reads = reads.astype({'start_i': int,'end_i': int,'read': int,'distance':int})
        reads['mismatches'] = [f">{max_distance}" if d==-1 else int(d) for d in reads['distance']]
        reads['location'] = [(start_i,end_i) if start_i!=-1 else "Absent" for (start_i,end_i) in t.zip_cols(df=reads,cols=['start_i','end_i'])]

        # Save & append fastq dataframe to final dataframe
        print('Save & append fastq dataframe to final dataframe')
        io.save(dir=os.path.join(out_dir,motif),file=f'{fastq_name}.csv',obj=reads) # Save checkpoint
        df = pd.concat([df,reads]).reset_index(drop=True) # save to final dataframe
    
    # Save & return
    memories.append(memory_timer(task='count_motif()'))
    io.save(dir=os.path.join(out_dir,f'.count_{motif}'),
            file=f'{datetime.datetime.now().strftime("%Y%m%d_%H%M%S")}_stats.csv',
            obj=pd.DataFrame(stats, columns=['file','reads','reads_w_motif','reads_wo_motif']))
    io.save(dir=os.path.join(out_dir,f'.count_{motif}'),
            file=f'{datetime.datetime.now().strftime("%Y%m%d_%H%M%S")}_memories.csv',
            obj=pd.DataFrame(memories, columns=['Task','Memory, MB','Time, s']))
    io.save(dir=out_dir,file=f'{motif}.csv',obj=df)
    if return_df: return df # Return dataframe (optional)

def plot_motif(df: pd.DataFrame | str, out_dir: str=None, plot_suf='.pdf',numeric: str='count',
               id_col: str='fastq_file', id_axis: str='fastq', stack_figsize: tuple=(7,3), heat_figsize: tuple=None,
               cutoff_frac:float=0.01, return_df:bool=False, show:bool=True) -> tuple[pd.DataFrame]:
    '''
    plot_motif(): generate plots highlighting motif mismatches, locations, and sequences
    
    Parameters:
    df (dataframe | str): count_motif() dataframe (or file path)
    out_dir (str, optional): output directory
    plot_suf (str, optional): plot type suffix with '.' (Default: '.pdf')
    numeric (str, optional): 'count' or 'fraction' can be the numeric column for plots (Default: 'count')
    id_col (str, optional): id column name (Default: 'fastq_file')
    id_axis (str, optional): replace id column name on plots (Default: 'fastq')
    stack_figsize (tuple, optional): stacked bar plot figure size (Default: (7,3))
    heat_figsize (tuple, optional): heatmap figure size (Default: None)
    cutoff_frac (float, optional): y-axis values needs be greater than (e.g. 0.01) fraction
    return_df (bool, optional): return dataframe (Default: False)
    show (bool, optional): show plots (Default: True)

    Dependencies: count_motifs(), plot
    '''
    # Get dataframe from file path if needed
    if type(df)==str:
        df = io.get(pt=df)
    
    # Check numeric column
    if numeric not in ['count','fraction']:
        raise(ValueError(f"numeric can only be 'count' or 'fraction'; not {numeric}"))

    # Get value_counts() dataframes...
    # ...remove unwanted columns
    mismatches_cols = [c for c in list(df.columns) if c not in ['read','window','start_i','end_i','location']]
    locations_cols = [c for c in list(df.columns) if c not in ['read','window','distance','mismatches']]
    windows_cols = [c for c in list(df.columns) if c not in ['read','start_i','end_i','distance','mismatches','location']]

    # ...generate dataframes
    df_mismatches = df[mismatches_cols].value_counts().reset_index()
    df_locations = df[locations_cols].value_counts().reset_index()
    df_windows = df[windows_cols].value_counts().reset_index()
    
    # ...save (optional)
    if out_dir is not None:
        io.save(dir=out_dir,file=f"mismatches.csv",obj=df_mismatches)
        io.save(dir=out_dir,file=f"locations.csv",obj=df_locations)
        io.save(dir=out_dir,file=f"windows.csv",obj=df_windows)

    # ...define cut() based on cutoff_frac
    def cut(df_vc: pd.DataFrame, col: str, off: bool=True):
        '''
        cut(): apply cutoff_fraction to value_counts() dataframe grouped by id

        Parameters:
        df_vc (dataframe): value_counts() dataframe
        col (str): column name that will be overwritten with cutoff_frac
        off (bool, optional): apply cutoff (Default: True)
        '''
        df_vc_cutoff = pd.DataFrame() # Initialize output dataframe
        for id in list(df_vc[id_col].value_counts().keys()): # Iterate through ids
            # Group by id and calculate fraction
            df_vc_id = df_vc[df_vc[id_col]==id]
            df_vc_id['fraction'] = df_vc_id['count']/sum(df_vc_id['count'])
            if off: # Apply cutoff_fract
                # Append greater than cutoff_frac
                df_vc_cutoff = pd.concat([df_vc_cutoff,df_vc_id[df_vc_id['fraction']>=cutoff_frac]]).reset_index(drop=True)
                # Group less than cutoff_frac and append
                df_vc_id_other = df_vc_id[df_vc_id['fraction']<cutoff_frac].reset_index(drop=True)
                if df_vc_id_other.empty==False:
                    df_vc_id_other['count']=sum(df_vc_id_other['count'])
                    df_vc_id_other['fraction']=sum(df_vc_id_other['fraction'])
                    df_vc_id_other[col]=f'<{cutoff_frac*100}%'
                    df_vc_cutoff = pd.concat([df_vc_cutoff,df_vc_id_other.iloc[:1]]).reset_index(drop=True)
            else: # Do not apply cutoff_fract
                df_vc_cutoff = pd.concat([df_vc_cutoff,df_vc_id]).reset_index(drop=True)  

        # Return output dataframe
        return df_vc_cutoff
    
    #...apply cut()
    df_mismatches = cut(df_vc=df_mismatches,col='mismatches',off=False)
    df_locations = cut(df_vc=df_locations,col='location')
    df_windows = cut(df_vc=df_windows,col='window')

    # Adjust heat_figsize if not specified
    if heat_figsize is None:
        heat_figsize = (len(df_windows[id_col].unique()),len(df_windows['window'].unique()))

    # Plots
    if numeric=='count':
        p.stack(df=df_mismatches,x=id_col,y=numeric,cols='mismatches', y_axis='Reads',
                title=f"{df.iloc[0]['motif']}: {df.iloc[0]['pattern']}", x_axis=id_axis,
                vertical=False,figsize=stack_figsize,palette_or_cmap='tab20',
                dir=out_dir,file=f"mismatches{plot_suf}",show=show)
        
        p.stack(df=df_locations,x=id_col,y=numeric,cols='location', y_axis='Reads',
                title=f"{df.iloc[0]['motif']}: {df.iloc[0]['pattern']}", x_axis=id_axis,
                vertical=False,figsize=stack_figsize,palette_or_cmap='tab20',
                dir=out_dir,file=f"locations{plot_suf}",show=show)
        
        p.heat(df=df_windows,x=id_col,y='window',vars='motif',vals=numeric,x_axis=id_axis,y_ticks_font='Courier New',
               figsize=heat_figsize,x_ticks_rot=45,title=f"{df.iloc[0]['motif']}: {df.iloc[0]['pattern']}",sq=False,
               dir=out_dir,file=f"windows{plot_suf}",show=show)
    
    else: # fraction
        p.stack(df=df_mismatches,x=id_col,y=numeric,cols='mismatches', y_axis='Reads fraction',
                title=f"{df.iloc[0]['motif']}: {df.iloc[0]['pattern']}", x_axis=id_axis,
                vertical=False,figsize=stack_figsize,palette_or_cmap='tab20',
                dir=out_dir,file=f"mismatches{plot_suf}",show=show)
        
        p.stack(df=df_locations,x=id_col,y=numeric,cols='location', y_axis='Reads fraction',
                title=f"{df.iloc[0]['motif']}: {df.iloc[0]['pattern']}", x_axis=id_axis,
                vertical=False,figsize=stack_figsize,palette_or_cmap='tab20',
                dir=out_dir,file=f"locations{plot_suf}",show=show)
        
        p.heat(df=df_windows,x=id_col,y='window',vars='motif',vals=numeric,x_axis=id_axis,y_ticks_font='Courier New',
               figsize=heat_figsize,x_ticks_rot=45,title=f"{df.iloc[0]['motif']}: {df.iloc[0]['pattern']}",sq=False,
               dir=out_dir,file=f"windows{plot_suf}",vals_dims=(0,1),show=show)
    
    # Return value_counts() dataframes (optional)
    if return_df: return (df_mismatches,df_locations,df_windows)

### Region/read alignments: spacer,..., & ngRNA-epegRNA
def mismatch_alignments(align_col: str, out_dir: str, fastq_name: str,
                        fastq_df_ref: pd.DataFrame, dc_alignments: dict, dc_aligned_reads: dict,
                        dc_alignments_mismatch_pos: dict, dc_alignments_mismatch_num: dict,
                        return_df: bool=False) -> pd.DataFrame:
    '''
    mismatch_alignments(): Compute & save mismatch number and position per alignment; enables checkpoints

    Parameters:
    align_col (str): align column name in the annotated library reference file
    out_dir (str): directory for output files
    fastq_name (str): name of fastq file
    fastq_df_ref (dataframe): annotated reference library dataframe
    dc_alignments (dict): dictionary of alignments
    dc_aligned_reads (dict): dictionary of aligned reads
    dc_alignments_mismatch_pos (dict): dictionary of mismatch positions
    dc_alignments_mismatch_num (dict): dictionary of mismatch numbers
    return_df (bool, optional): return dataframe (Default: False)
    
    Dependencies: count_region(), count_alignments(), perform_alignments(), pandas, tidy
    '''
    # Merge alignment dictionaries into a fastq dataframe
    print('Merge alignment dictionaries into a fastq dataframe')
    df_alignments = pd.DataFrame(dc_alignments.items(),columns=[align_col,'alignments'])
    df_aligned_reads = pd.DataFrame(dc_aligned_reads.items(),columns=[align_col,'reads_aligned'])
    df_alignments_mismatch_num = pd.DataFrame(dc_alignments_mismatch_num.items(),columns=[align_col,'mismatch_num'])
    df_alignments_mismatch_pos = pd.DataFrame(dc_alignments_mismatch_pos.items(),columns=[align_col,'mismatch_pos']) 
    df_fastq = pd.merge(left=fastq_df_ref,right=df_alignments,on=align_col)
    df_fastq['alignments_fraction'] = [alignments/reads_processed for (alignments,reads_processed) in t.zip_cols(df=df_fastq,cols=['alignments','reads_processed'])]
    df_fastq = pd.merge(left=df_fastq,right=df_aligned_reads,on=align_col)
    df_fastq = pd.merge(left=df_fastq,right=df_alignments_mismatch_num,on=align_col)
    df_fastq = pd.merge(left=df_fastq,right=df_alignments_mismatch_pos,on=align_col)
    
    # Calculate mismatch num & position per alignment
    print('Calculate mismatch num & position per alignment')
    mismatch_num_per_alignment_ls = []
    mismatch_pos_per_alignment_ls = []
    for (ref,mismatch_pos,mismatch_num,alignments) in t.zip_cols(df=df_fastq,cols=[align_col,'mismatch_pos','mismatch_num','alignments']):
        if alignments==0:
            mismatch_num_per_alignment_ls.append(0)
            mismatch_pos_per_alignment_ls.append({pos:0 for pos in range(1,len(ref)+1)})
        else:
            mismatch_num_per_alignment_ls.append(mismatch_num/alignments)
            mismatch_pos_per_alignment_ls.append({pos:mismatch_pos.count(pos)/alignments for pos in range(1,len(ref)+1)})
    df_fastq['mismatch_num_per_alignment'] = mismatch_num_per_alignment_ls
    df_fastq['mismatch_pos_per_alignment'] = mismatch_pos_per_alignment_ls
    
    # Save & return fastq dataframe
    print('Save & return fastq dataframe')
    io.save(dir=out_dir,file=f'{fastq_name}.csv',obj=df_fastq)
    if return_df: return df_fastq

def perform_alignments(align_col: str, out_dir: str, fastq_name: str, fastq_df_ref: pd.DataFrame,
                       aligner: PairwiseAligner, seqs: list, memories: list, align_ckpt: int) -> pd.DataFrame:
    '''
    perform_alignments(): perform alignments on fastq reads using PairwiseAligner and compute mismatches using mismatch_alignments()

    Parameters:
    align_col (str): align column name in the annotated library reference file
    out_dir (str): directory for output files
    fastq_name (str): name of fastq file
    fastq_df_ref (dataframe): annotated reference library dataframe
    aligner (PairwiseAligner): pairwise aligner object
    seqs (list): list of sequences to align
    memories (list): list of memory timers
    align_ckpt (int): alignment checkpoint interval
    
    Dependencies: Bio.SeqIO, gzip, os, pandas, Bio.Seq.Seq, Bio.PairwiseAligner, count_region(), count_alignments(), perform_alignments(), io, tidy
    '''
    print('Perform alignments')
    dc_alignments = {ref:0 for ref in fastq_df_ref[align_col]}
    dc_aligned_reads = {ref:[] for ref in fastq_df_ref[align_col]}
    dc_alignments_mismatch_num = {ref:0 for ref in fastq_df_ref[align_col]}
    dc_alignments_mismatch_pos = {ref:[] for ref in fastq_df_ref[align_col]}
    
    for s,seq in enumerate(seqs): # Iterate though sequences
        if s==0: # Initial alignment status
            print(f'{s+1} out of {len(seqs)}') 
        elif s%align_ckpt==0: # Alignment status; save checkpoint
            print(f'{s+1} out of {len(seqs)}')
            mismatch_alignments(align_col=align_col, out_dir=out_dir, fastq_name=fastq_name, fastq_df_ref=fastq_df_ref,
                                dc_alignments=dc_alignments, dc_aligned_reads=dc_aligned_reads,
                                dc_alignments_mismatch_pos=dc_alignments_mismatch_pos, dc_alignments_mismatch_num=dc_alignments_mismatch_num)
            memories.append(memory_timer(task=f"{fastq_name} (align {s+1} out of {len(seqs)})"))

        if seq is None: # Missing region (not applicable to count_alignments())
            continue

        seq_alignments_scores = []
        seq_alignments_aligned = []
        for ref in fastq_df_ref[align_col]: # Iterate though reference sequences
            seq_alignment = aligner.align(ref, seq[0:len(ref)]) # trim ngs sequence to reference sequence & align
            seq_alignments_scores.append(seq_alignment[0].score) # Save highest alignment score
            seq_alignments_aligned.append(seq_alignment[0].aligned[0]) # Save alignment matches

        # Isolate maximum score alignment
        i = seq_alignments_scores.index(max(seq_alignments_scores))
        ref_i = fastq_df_ref.iloc[i][align_col]
        aligned_i = seq_alignments_aligned[i]
        dc_alignments[fastq_df_ref.iloc[i][align_col]] = dc_alignments[ref_i]+1
        dc_aligned_reads[fastq_df_ref.iloc[i][align_col]].append(s)

        # Find & quantify mismatches (Change zero-indexed to one-indexed)
        mismatch_pos = []
        if len(aligned_i) == 1: 
            (a1,b1) = aligned_i[0]
            if (a1==0)&(b1==len(ref_i)-1): mismatch_pos.extend([])
            elif a1==0: mismatch_pos.extend([k+1 for k in range(b1+1,len(ref_i))])
            elif b1==len(ref_i)-1: mismatch_pos.extend([k+1 for k in range(0,a1-1)])
            else: mismatch_pos.extend([j+1 for j in range(0,a1-1)] + [k+1 for k in range(b1+1,len(ref_i))])
        else:
            for j in range(len(aligned_i)-1):
                (a1,b1) = aligned_i[j]
                (a2,b2) = aligned_i[j+1]
                if (j==0)&(a1!=0): mismatch_pos.extend([k+1 for k in range(0,a1-1)])
                if (j==len(aligned_i)-2)&(b2!=len(ref_i)-1): mismatch_pos.extend([k+1 for k in range(b2+1,len(ref_i))])
                mismatch_pos.extend([k+1 for k in range(b1+1,a2-1)])
        dc_alignments_mismatch_num[ref_i] = dc_alignments_mismatch_num[ref_i] + len(mismatch_pos)
        dc_alignments_mismatch_pos[ref_i] = dc_alignments_mismatch_pos[ref_i] + mismatch_pos

    # Perform mismatch alignments and return fastq dataframe
    return mismatch_alignments(align_col=align_col, out_dir=out_dir, fastq_name=fastq_name, fastq_df_ref=fastq_df_ref,
                               dc_alignments=dc_alignments, dc_aligned_reads=dc_aligned_reads,
                               dc_alignments_mismatch_pos=dc_alignments_mismatch_pos, dc_alignments_mismatch_num=dc_alignments_mismatch_num,
                               return_df=True)
    
def plot_alignments(fastq_alignments: dict | str, align_col: str, id_col: str,
                    out_dir: str, plot_suf:str='.pdf', show:bool=False, **plot_kwargs):
    ''' 
    plot_alignments(): generate line & distribution plots from fastq alignments dictionary
    
    Parameters:
    fastq_alignments (dict | str): fastq alignments dictionary from count_region() or count_alignments()
    align_col (str): align column name in the annotated library reference file
    id_col (str): id column name in the annotated library reference file
    fastq_dir (str): directory with fastq files
    out_dir (str): directory for output files
    plot_suf (str, optional): plot type suffix with '.' (Default: '.pdf')
    show (bool, optional): show plots (Default: False)
    **plot_kwargs (optional): plot key word arguments

    Dependencies: pandas & plot
    '''
    # Get dictionary from directory path if needed
    if type(fastq_alignments)==str: 
        fastq_alignments = io.get_dir(fastq_alignments)
    
    for fastq_name,df_fastq in fastq_alignments.items(): # Iterate through dictionary
        
        # Plot mismatch position per alignment
        print('Plot mismatch position per alignment')
        
        out_dir_fastq_name = os.path.join(out_dir,fastq_name)
        df_fastq_plot = pd.DataFrame()
        for align,id,mismatch_pos_per_alignment in t.zip_cols(df=df_fastq,cols=[align_col,id_col,'mismatch_pos_per_alignment']):
            df_fastq_plot_align = pd.DataFrame({align_col:[align]*len(mismatch_pos_per_alignment), # Obtain individual alignments
                                                id_col:[id]*len(mismatch_pos_per_alignment),
                                                'mismatch_pos':list(mismatch_pos_per_alignment.keys()),
                                                'mismatch_pos_per_alignment':list(mismatch_pos_per_alignment.values())})
            
            p.scat(typ='line',df=df_fastq_plot_align,x='mismatch_pos',y='mismatch_pos_per_alignment', # Plot mismatches for each alignment
                   title=f'{fastq_name} {id}',x_axis='Alignment Position',y_axis='Mismatches/Alignment',y_axis_dims=(0,1),
                   dir=out_dir_fastq_name,file=f'{id.replace(".","_")}{plot_suf}',
                   show=show,**plot_kwargs)
            
            df_fastq_plot = pd.concat(objs=[df_fastq_plot,df_fastq_plot_align]).reset_index(drop=True) # Group alignment mismatches

        p.scat(typ='line',df=df_fastq_plot,x='mismatch_pos',y='mismatch_pos_per_alignment',cols=id_col, # Plot mismatches for each alignment
               title=f'{fastq_name}',x_axis='Alignment Position',y_axis='Mismatches/Alignment',y_axis_dims=(0,1),
               dir=out_dir_fastq_name,file=f'alignment_mismatches{plot_suf}',legend_ncol=int(round(len(df_fastq_plot[id_col].value_counts())/20)),
               show=show,**plot_kwargs)

        p.dist(typ='hist',df=df_fastq,x='alignments',x_axis_dims=(0,max(df_fastq['alignments'])),
               title=f'{fastq_name}',dir=out_dir_fastq_name,file=f'alignments{plot_suf}',
               show=show,**plot_kwargs)

def count_region(df_ref: pd.DataFrame | str, align_col: str, id_col: str,
                 fastq_dir: str, df_motif5: pd.DataFrame | str, df_motif3: pd.DataFrame | str,
                 out_dir: str, fastq_col: str=None, match_score: float = 2, mismatch_score: float = -1, 
                 open_gap_score: float = -10, extend_gap_score: float = -0.1, align_dims: tuple=(0,0),
                 align_ckpt: int=10000, plot_suf: str=None, show: bool=False, return_dc: bool=False,
                 **plot_kwargs) -> dict[pd.DataFrame]:
    ''' 
    count_region(): align read region from fastq directory to the annotated library with mismatches; plot and return fastq alignments dictionary

    Parameters:
    df_ref (dataframe | str): annotated reference library dataframe (or file path)
    align_col (str): align column name in the annotated reference library
    id_col (str): id column name in the annotated reference library
    fastq_dir (str): directory with fastq files
    df_motif5 (dataframe | str): 5' motif dataframe (or file path)
    df_motif3 (dataframe | str): 3' motif dataframe (or file path)
    out_dir (str): directory for output files
    fastq_col (str, optional): fastq column name in the annotated reference library (Default: None)
    match_score (float, optional): match score for pairwise alignment (Default: 2)
    mismatch_score (float, optional): mismatch score for pairwise alignment (Default: -1)
    open_gap_score (float, optional): open gap score for pairwise alignment (Default: -10)
    extend_gap_score (float, optional): extend gap score for pairwise alignment (Default: -0.1)
    align_dims (tuple, optional): (start_i, end_i) alignments per fastq file to save compute (Default: None)
    align_ckpt (int, optional): save checkpoints for alignments (Default: 10000)
    plot_suf (str, optional): plot type suffix with '.' (Default: None)
    show (bool, optional): show plots (Default: False)
    return_dc (bool, optional): return fastqs dictionary (Default: False)
    **plot_kwargs (optional): plot key word arguments

    Dependencies: Bio.SeqIO, gzip, os, pandas, Bio.Seq.Seq, Bio.PairwiseAligner, mismatch_alignments(), perform_alignments(), plot_alignments(), memory_timer(), io, tidy
    '''
    # Initialize timer
    memory_timer(reset=True)

    # Intialize the aligner
    aligner = PairwiseAligner()
    aligner.mode = 'global'  # Use 'local' for local alignment
    aligner.match_score = match_score  # Score for a match
    aligner.mismatch_score = mismatch_score  # Penalty for a mismatch; applied to both strands
    aligner.open_gap_score = open_gap_score  # Penalty for opening a gap; applied to both strands
    aligner.extend_gap_score = extend_gap_score  # Penalty for extending a gap; applied to both strands

    # Get dataframes from file path if needed
    if type(df_ref)==str: 
        df_ref = io.get(df_ref)
    if type(df_motif5)==str: 
        df_motif5 = io.get(df_motif5)
    if type(df_motif3)==str: 
        df_motif3 = io.get(df_motif3)
    
    # Check dataframe for alignment, id, & fastq columns
    if align_col not in df_ref.columns.tolist():
        raise Exception(f'Missing alignment column: {align_col}') 
    if id_col not in df_ref.columns.tolist():
        raise Exception(f'Missing id column: {id_col}')
    if fastq_col is not None:
        if fastq_col not in df_ref.columns.tolist():
            raise Exception(f'Missing fastq column: {fastq_col}')
    df_ref[align_col] = df_ref[align_col].str.upper() 

    # Check for fastq_file, start_i and end_i columns
    if 'fastq_file' not in df_motif5.columns.tolist():
        raise Exception('Missing column in df_motif5: fastq_file') 
    if 'end_i' not in df_motif5.columns.tolist():
        raise Exception('Missing column in df_motif5: end_i') 
    if 'fastq_file' not in df_motif3.columns.tolist():
        raise Exception('Missing column in df_motif3: fastq_file') 
    if 'start_i' not in df_motif3.columns.tolist():
        raise Exception('Missing column in df_motif3: start_i') 

    # Check if align_dims is a tuple of length 2 with start_i greater than end_i
    if align_dims is None:
        align_dims=(0,0)
    elif not isinstance(align_dims, tuple) or len(align_dims) != 2:
        raise ValueError(f"align_dims={align_dims} was not a tuple of length 2")
    else:
        if align_dims[0]<0 or align_dims[1]<0:
            raise ValueError(f"align_dims={align_dims} needs to be greater than 0")
        if align_dims[1]<align_dims[0]:
            raise ValueError(f"align_dims={align_dims} needs to be in the form (start_i, end_i)")

    # Check if align_ckpt is a positive integer
    if not isinstance(align_ckpt, int) or align_ckpt <= 0:
        raise ValueError(f"align_ckpt={align_ckpt} needs to be a positive integer")
    
    # Memory & stats reporting
    memories = []
    stats = []

    # Make fastqs dictionary
    if return_dc: fastqs = dict()
    for fastq_file in os.listdir(fastq_dir): # Iterate through fastq files
        
        print(f"Processing {fastq_file}...") # Get reads
        regions = 0 # Keep track of # of reads with(out) regions
        missing5 = 0
        missing3 = 0
        overlap53 = 0
        seqs = [] # Store region sequences from reads with motifs
        if fastq_file.endswith(".fastq.gz"): # Compressed fastq
            fastq_name = fastq_file[:-len(".fastq.gz")] # Get fastq name
            fastq_motif5 = df_motif5[df_motif5['fastq_file']==fastq_file].reset_index(drop=True) # Isolate fastq motif5 info
            fastq_motif3 = df_motif3[df_motif3['fastq_file']==fastq_file].reset_index(drop=True) # Isolate fastq motif3 info
            if fastq_col is not None: fastq_df_ref = df_ref[df_ref[fastq_col]==fastq_file].reset_index(drop=True) # Isolate fastq reference library info (if needed)
            else: fastq_df_ref = df_ref.copy()
            with gzip.open(os.path.join(fastq_dir,fastq_file), "rt") as handle:
                for i,record in enumerate(SeqIO.parse(handle, "fastq")): # Parse reads & isolate region between motifs
                    
                    # Get # of reads in fastq file; skip reads that are not in the alignment range
                    reads = i+1
                    if (align_dims[0]!=0)&(align_dims[0]>i):
                        continue
                    elif (align_dims[1]!=0)&(align_dims[1]<=i):
                        continue

                    # Obtain motif boundaries that define region
                    start_i = fastq_motif5.iloc[i]["end_i"]
                    end_i = fastq_motif3.iloc[i]["start_i"] 
                    if (start_i!=-1) & (end_i!=-1): # Both motifs are present
                        if start_i<end_i: # Motifs do not overlap
                            regions += 1
                            seqs.append(record.seq[start_i:end_i]) 
                        else: # Motifs overlap
                            overlap53 += 1
                            seqs.append(None)
                    elif (start_i==-1) & (end_i==-1): # Both motifs are missing
                        missing5 += 1
                        missing3 += 1
                        seqs.append(None)
                    elif start_i==0: # 5' motif is missing
                        missing5 += 1 
                        seqs.append(None)
                    else: # 3' motif is missing
                        missing3 += 1 
                        seqs.append(None)

                    # Processing status
                    if len(seqs)%align_ckpt==0: 
                        print(f"Processed {len(seqs)} reads")
        
        elif fastq_file.endswith(".fastq"): # Uncompressed fastq
            fastq_name = fastq_file[:-len(".fastq")] # Get fastq name
            fastq_motif5 = df_motif5[df_motif5['fastq_file']==fastq_file].reset_index(drop=True) # Isolate fastq motif5 info
            fastq_motif3 = df_motif3[df_motif3['fastq_file']==fastq_file].reset_index(drop=True) # Isolate fastq motif3 info
            if fastq_col is not None: fastq_df_ref = df_ref[df_ref[fastq_col]==fastq_file].reset_index(drop=True) # Isolate fastq reference library info (if needed)
            else: fastq_df_ref = df_ref.copy()
            with open(os.path.join(fastq_dir,fastq_file), "r") as handle:    
                for i,record in enumerate(SeqIO.parse(handle, "fastq")): # Parse reads & isolate region between motifs
                    
                    # Get # of reads in fastq file; skip reads that are not in the alignment range
                    reads = i+1
                    if (align_dims[0]!=0)&(align_dims[0]>i):
                        continue
                    elif (align_dims[1]!=0)&(align_dims[1]<=i):
                        continue

                    # Obtain motif boundaries that define region
                    start_i = fastq_motif5.iloc[i]["end_i"]
                    end_i = fastq_motif3.iloc[i]["start_i"] 
                    if (start_i!=-1) & (end_i!=-1): # Both motifs are present
                        if start_i<end_i: # Motifs do not overlap
                            regions += 1
                            seqs.append(record.seq[start_i:end_i]) 
                        else: # Motifs overlap
                            overlap53 += 1
                            seqs.append(None)
                    elif (start_i==-1) & (end_i==-1): # Both motifs are missing
                        missing5 += 1
                        missing3 += 1
                        seqs.append(None)
                    elif start_i==0: # 5' motif is missing
                        missing5 += 1 
                        seqs.append(None)
                    else: # 3' motif is missing
                        missing3 += 1 
                        seqs.append(None)

                    # Processing status
                    if len(seqs)%align_ckpt==0: 
                        print(f"Processed {len(seqs)} reads")
        
        else: # Not a fastq file
            print("Not a fastq file")
            continue
        
        # Update memories & stats: # of reads with(out) the region
        print(f'{fastq_name}:\t{len(seqs)} reads\t=>\t{regions} reads;\t{missing5} missing motif5;\t{missing3} missing motif3;\t{overlap53} motif overlaps')
        stats.append((fastq_name,reads,len(seqs),regions,missing5,missing3,overlap53))
        memories.append(memory_timer(task=f"{fastq_file} (region)"))

        # Append # of reads & alignment range to fastq_df_ref
        fastq_df_ref['reads_total']= [reads]*len(fastq_df_ref)
        fastq_df_ref['reads_processed']= [len(seqs)]*len(fastq_df_ref)
        if align_dims==(0,0): 
            fastq_df_ref['align_dims']= [(0,reads)]*len(fastq_df_ref)
        else:
            fastq_df_ref['align_dims']= [(align_dims[0],align_dims[1])] * len(fastq_df_ref)

        # Perform alignments, compute mismatches, & append to fastq dataframe to dictionary
        df_fastq = perform_alignments(align_col=align_col, out_dir=out_dir, fastq_name=fastq_name, fastq_df_ref=fastq_df_ref,
                                      aligner=aligner, seqs=seqs, memories=memories, align_ckpt=align_ckpt)
        if return_dc: fastqs[fastq_name]=df_fastq
        memories.append(memory_timer(task=f"{fastq_name} (aligned)"))

        # Plot mismatch position per alignment
        if plot_suf is not None: 
            plot_alignments(fastq_alignments={fastq_name:df_fastq}, align_col=align_col, id_col=id_col,
                            out_dir=out_dir, plot_suf=plot_suf, show=show, **plot_kwargs)
        
    # Save and return
    memories.append(memory_timer(task='count_region()'))
    io.save(dir=os.path.join(out_dir,'.count_region'),
            file=f'{datetime.datetime.now().strftime("%Y%m%d_%H%M%S")}_stats.csv',
            obj=pd.DataFrame(stats, columns=['file','reads_total','reads_processed','reads_w_region','reads_wo_motif5','reads_wo_motif3','reads_w_motif_overlap']))
    io.save(dir=os.path.join(out_dir,'.count_region'),
            file=f'{datetime.datetime.now().strftime("%Y%m%d_%H%M%S")}_memories.csv',
            obj=pd.DataFrame(memories, columns=['Task','Memory, MB','Time, s']))
    if return_dc: return fastqs

def count_alignments(df_ref: pd.DataFrame | str, align_col: str, id_col: str,
                     fastq_dir: str, out_dir: str, fastq_col: str=None, match_score: float = 2, mismatch_score: float = -1, 
                     open_gap_score: float = -10, extend_gap_score: float = -0.1, align_dims: tuple=(0,0),
                     align_ckpt: int=10000, plot_suf: str=None, show: bool=False, return_dc: bool=False,
                     **plot_kwargs) -> dict[pd.DataFrame]:
    ''' 
    count_alignments(): align reads from fastq directory to annotated library with mismatches; plot and return fastq alignments dictionary
    
    Parameters:
    df_ref (dataframe | str): annotated reference library dataframe (or file path)
    align_col (str): align column name in the annotated reference library
    id_col (str): id column name in the annotated reference library
    fastq_dir (str): directory with fastq files
    out_dir (str): directory for output files
    fastq_col (str, optional): fastq column name in the annotated reference library (Default: None)
    match_score (float, optional): match score for pairwise alignment (Default: 2)
    mismatch_score (float, optional): mismatch score for pairwise alignment (Default: -1)
    open_gap_score (float, optional): open gap score for pairwise alignment (Default: -10)
    extend_gap_score (float, optional): extend gap score for pairwise alignment (Default: -0.1)
    align_dims (tuple, optional): (start_i, end_i) alignments per fastq file to save compute (Default: None)
    align_ckpt (int, optional): save checkpoints for alignments (Default: 10000)    plot_suf (str, optional): plot type suffix with '.' (Default: '.pdf')
    plot_suf (str, optional): plot type suffix with '.' (Default: None)
    show (bool, optional): show plots (Default: False)
    return_dc (bool, optional): return fastqs dictionary (Default: False)
    **plot_kwargs (optional): plot key word arguments

    Dependencies: Bio.SeqIO, gzip, os, pandas, Bio.Seq.Seq, Bio.PairwiseAligner, mismatch_alignments(), perform_alignments(), plot_alignments(), memory_timer(), io, tidy
    '''
    # Initialize timer
    memory_timer(reset=True)

    # Intialize the aligner
    aligner = PairwiseAligner()
    aligner.mode = 'global'  # Use 'local' for local alignment
    aligner.match_score = match_score  # Score for a match
    aligner.mismatch_score = mismatch_score  # Penalty for a mismatch; applied to both strands
    aligner.open_gap_score = open_gap_score  # Penalty for opening a gap; applied to both strands
    aligner.extend_gap_score = extend_gap_score  # Penalty for extending a gap; applied to both strands

    # Get dataframe from file path if needed
    if type(df_ref)==str: 
        df_ref = io.get(df_ref)
    
    # Check dataframe for alignment and id columns
    if align_col not in df_ref.columns.tolist():
        raise Exception(f'Missing alignment column: {align_col}') 
    if id_col not in df_ref.columns.tolist():
        raise Exception(f'Missing id column: {id_col}')
    if fastq_col is not None:
        if fastq_col not in df_ref.columns.tolist():
            raise Exception(f'Missing fastq column: {fastq_col}')
    df_ref[align_col] = df_ref[align_col].str.upper() 
    
    # Memory & stats reporting
    memories = []
    stats = []

    # Make fastqs dictionary
    if return_dc: fastqs = dict()
    for fastq_file in os.listdir(fastq_dir): # Iterate through fastq files
        
        print(f"Processing {fastq_file}...") # Get reads
        seqs = [] # Store alignment sequences
        if fastq_file.endswith(".fastq.gz"): # Compressed fastq
            fastq_name = fastq_file[:-len(".fastq.gz")] # Get fastq name
            if fastq_col is not None: fastq_df_ref = df_ref[df_ref[fastq_col]==fastq_file].reset_index(drop=True) # Isolate fastq reference library info (if needed)
            else: fastq_df_ref = df_ref.copy()
            with gzip.open(os.path.join(fastq_dir,fastq_file), "rt") as handle:
                for i,record in enumerate(SeqIO.parse(handle, "fastq")): # Parse reads
                    
                    # Get # of reads in fastq file; skip reads that are not in the alignment range
                    reads = i + 1
                    if (align_dims[0]!=0)&(align_dims[0]>i):
                        continue
                    elif (align_dims[1]!=0)&(align_dims[1]<=i):
                        continue
                    
                    # Append sequence
                    seqs.append(record.seq)

                    # Processing status
                    if len(seqs)%align_ckpt==0: 
                        print(f"Processed {len(seqs)} reads")
                
        elif fastq_file.endswith(".fastq"): # Uncompressed fastq
            fastq_name = fastq_file[:-len(".fastq")] # Get fastq name
            if fastq_col is not None: fastq_df_ref = df_ref[df_ref[fastq_col]==fastq_file].reset_index(drop=True) # Isolate fastq reference library info (if needed)
            else: fastq_df_ref = df_ref.copy()
            with open(os.path.join(fastq_dir,fastq_file), "r") as handle:
                for i,record in enumerate(SeqIO.parse(handle, "fastq")): # Parse reads
                    
                    # Get # of reads in fastq file; skip reads that are not in the alignment range
                    reads = i + 1
                    if (align_dims[0]!=0)&(align_dims[0]>i):
                        continue
                    elif (align_dims[1]!=0)&(align_dims[1]<=i):
                        continue
                    
                    # Append sequence
                    seqs.append(record.seq)

                    # Processing status
                    if len(seqs)%align_ckpt==0: 
                        print(f"Processed {len(seqs)} reads")
        
        else: # Not a fastq file
            print("Not a fastq file")
            continue
        
        # Update memories & stats file with # of reads
        print(f'{fastq_name}:\t{len(seqs)} reads')
        stats.append((fastq_name,reads,len(seqs)))
        memories.append(memory_timer(task=f"{fastq_file} (reads)"))
        
        # Append # of reads & alignment range to fastq_df_ref
        fastq_df_ref['reads_total']= [reads]*len(fastq_df_ref)
        fastq_df_ref['reads_processed']= [len(seqs)]*len(fastq_df_ref)
        if align_dims==(0,0): 
            fastq_df_ref['align_dims']= [(0,reads)]*len(fastq_df_ref)
        else:
            fastq_df_ref['align_dims']= [(align_dims[0],align_dims[1])] * len(fastq_df_ref)

        # Perform alignments, compute mismatches, & append to fastq dataframe to dictionary
        df_fastq = perform_alignments(align_col=align_col, out_dir=out_dir, fastq_name=fastq_name, fastq_df_ref=fastq_df_ref,
                                      aligner=aligner, seqs=seqs, memories=memories, align_ckpt=align_ckpt)
        if return_dc: fastqs[fastq_name]=df_fastq
        memories.append(memory_timer(task=f"{fastq_name} (aligned)"))

        # Plot mismatch position per alignment
        if plot_suf is not None: 
            plot_alignments(fastq_alignments={fastq_name:df_fastq}, align_col=align_col, id_col=id_col,
                            out_dir=out_dir, plot_suf=plot_suf, show=show, **plot_kwargs)

    # Save and return
    memories.append(memory_timer(task='count_alignments()'))
    io.save(dir=os.path.join(out_dir,'.count_alignments'),
            file=f'{datetime.datetime.now().strftime("%Y%m%d_%H%M%S")}_stats.csv',
            obj=pd.DataFrame(stats, columns=['file','total_reads','processed_reads']))
    io.save(dir=os.path.join(out_dir,'.count_alignments'),
            file=f'{datetime.datetime.now().strftime("%Y%m%d_%H%M%S")}_memories.csv',
            obj=pd.DataFrame(memories, columns=['Task','Memory, MB','Time, s']))
    if return_dc: return fastqs

def plot_paired(df: pd.DataFrame | str, title: str, out_dir: str,  
                id_col: str='ID', desired_col: str='desired', y: Literal['count','fraction']='count',
                plot_suf: str='.pdf', show: bool=False, **plot_kwargs):
    ''' 
    plot_paired(): generate stacked bar plots from paired_regions() dataframe
    
    Parameters:
    df (dataframe | str): dataframe from paired_regions() or file path
    title (str): plot title and file name
    out_dir (str): directory for output files
    id_col (str, optional): id column name in the paired region file (Default: 'ID')
    desired_col (str, optional): desired column name in the paired region file (Default: 'desired')
    y (str, optional): y axis for plots (Default: 'count', Options: 'count' & 'fraction')
    plot_suf (str, optional): plot type suffix with '.' (Default: '.pdf')
    show (bool, optional): show plots (Default: False)
    **plot_kwargs (optional): plot key word arguments

    Dependencies: pandas & plot
    '''
    # Get dataframe from file path if needed
    if type(df)==str: 
        df = io.get(df)

    # Create, save & plot alignment status
    paired_regions_alignment_status_df = df[[desired_col,'alignment_status']].value_counts().reset_index()
    paired_regions_alignment_status_df_sum = sum(paired_regions_alignment_status_df['count'])
    paired_regions_alignment_status_df['fraction'] = [count/paired_regions_alignment_status_df_sum for count in paired_regions_alignment_status_df['count']]
    
    io.save(dir=os.path.join(out_dir, title),file='alignment_status.csv',obj=paired_regions_alignment_status_df)
    
    p.stack(df=paired_regions_alignment_status_df,x='alignment_status',y=y,
            cols=desired_col,cols_ord=[True,False],vertical=False,figsize=(6,2),
            title=title,dir=os.path.join(out_dir, title),file=f'alignment_status{plot_suf}',show=show,**plot_kwargs)

    # Create, save & plot alignment distribution
    paired_regions_alignment_distribution_df = df[df['alignment_status']=='region 1 & 2']
    desired_ID = []
    for id,desired in t.zip_cols(df=paired_regions_alignment_distribution_df,cols=[id_col,desired_col]):
        if desired==True or id=='chimera': desired_ID.append(id)
        else: desired_ID.append('not chimera')
    paired_regions_alignment_distribution_df[id_col] = desired_ID

    io.save(dir=os.path.join(out_dir, title),file='alignment_distribution_per_read.csv',obj=paired_regions_alignment_distribution_df)

    paired_regions_alignment_distribution_df = paired_regions_alignment_distribution_df[[id_col,desired_col]].value_counts().reset_index()
    paired_regions_alignment_distribution_df_sum = sum(paired_regions_alignment_distribution_df['count'])
    paired_regions_alignment_distribution_df['fraction'] = [count/paired_regions_alignment_distribution_df_sum for count in paired_regions_alignment_distribution_df['count']]
    
    io.save(dir=os.path.join(out_dir, title),file='alignment_distribution.csv',obj=paired_regions_alignment_distribution_df)

    p.stack(df=paired_regions_alignment_distribution_df,
            x=desired_col,y=y,cols=id_col,palette_or_cmap='Spectral',x_ord=[True,False],vertical=False,
            cols_ord=list(paired_regions_alignment_distribution_df[id_col]),
            legend_ncol=4,legend_bbox_to_anchor=(0,-.3),figsize=(10,2),
            title=title,dir=os.path.join(out_dir, title),file=f'alignment_distribution{plot_suf}',show=show,**plot_kwargs)

def paired_regions(meta_dir: str, region1_dir: str, region2_dir: str, out_dir: str, 
                   id_col: str='ID', desired_col: str='desired', 
                   region1_alignment_col: str='r1_alignment', region2_alignment_col: str='r2_alignment', 
                   reads_aligned_col: str='reads_aligned', reads_processed_col: str='reads_processed',
                   y: Literal['count','fraction']='count', plot_suf: str='.pdf', show: bool=False, return_dc: bool=False,
                   **plot_kwargs) -> dict[pd.DataFrame]:
    '''
    paired_regions(): quantify, plot, & return (un)paired regions that aligned to the annotated library

    Parameters:
    meta_dir (str): directory with meta files
    region1_dir (str): directory with region 1 files
    region2_dir (str): directory with region 2 files
    out_dir (str): directory for output files
    id_col (str): id column name in the region & meta files
    desired_col (str): desired column name in the meta files
    region1_alignment_col (str): region 1 alignment column name in the region & meta files
    region2_alignment_col (str): region 2 alignment column name in the region & meta files
    reads_aligned_col (str, optional): reads_aligned column name in the region files (Default: 'reads_aligned')
    reads_processed_col (str, optional): reads_processed column name in the region files (Default: 'reads_processed')
    y (str, optional): y axis for plots (Default: 'count'; Options: 'count' & 'fraction')
    plot_suf (str, optional): plot type suffix with '.' (Default: '.pdf')
    show (bool, optional): show plots (Default: False)
    return_dc (bool, optional): return (un)paired regions dataframe (Default: False)
    **plot_kwargs (optional): plot key word arguments

    Dependencies: os, pandas, io, plot, & plot_paired()
    '''
    # Initialize timer; memory reporting
    memory_timer(reset=True)
    memories = []

    # Get meta, region 1 and 2 file names
    meta_file_names = io.sorted_file_names(dir=meta_dir)
    region1_file_names = io.sorted_file_names(dir=region1_dir)
    region2_file_names = io.sorted_file_names(dir=region2_dir)

    # Check that files are correctly paired...
    if (len(region1_file_names)!=len(region2_file_names)) & (len(meta_file_names)!=len(region2_file_names)): # Equal # of files
        raise Exception(f'Unequal # of files in meta & region directories:\nmeta_dir: {meta_dir}\nregion1_dir: {region1_dir}\nregion2_dir: {region2_dir}')
    
    # ...and obtain mismatch indices (if applicable)
    file_name_mismatch_ls = [] # Allowed mismatch R"1" & R"2"

    # Compare region file names
    for region1_file_name,region2_file_name in zip(region1_file_names,region2_file_names):
        if len(region1_file_name)!=len(region2_file_name):
            raise ValueError(f"Mispaired files in region directories:\nregion1_dir: {region1_file_name}\nregion2_dir: {region2_file_name}\nFile names are different lengths")
        
        file_name_mismatch = [i for i,(a,b) in enumerate(zip(region1_file_name, region2_file_name)) if a!=b]
        if len(file_name_mismatch)>1:
            raise Exception(f'Mispaired files in region directories:\nregion1_dir: {region1_file_name}\nregion2_dir: {region2_file_name}\nFile names contain more than 1 mismatch')
        elif len(file_name_mismatch)==1:
            file_name_mismatch_ls.extend(file_name_mismatch)
        else:
            file_name_mismatch_ls.append(len(region1_file_name))
    
    # Compare with meta file names
    for i,(meta_file_name,region1_file_name,region2_file_name) in enumerate(zip(meta_file_names,region1_file_names,region2_file_names)):
        if meta_file_name[0:file_name_mismatch_ls[i]]!=region1_file_name[0:file_name_mismatch_ls[i]]:
            raise ValueError(f"Mispaired files in meta & region 1 directories:\nmeta_dir: {meta_file_name}\nregion1_dir: {region1_file_name}\nFile names are different lengths")
        elif meta_file_name[0:file_name_mismatch_ls[i]]!=region2_file_name[0:file_name_mismatch_ls[i]]:
            raise ValueError(f"Mispaired files in meta & region 2 directories:\nrmeta_dir: {meta_file_name}\nregion2_dir: {region2_file_name}\nFile names are different lengths")
    
    # Parse paired regions
    paired_regions_dc = dict()
    for i,(meta_file_name,region1_file_name,region2_file_name) in enumerate(zip(meta_file_names,region1_file_names,region2_file_names)):
        
        # Get regions dataframe
        region1_file_df = io.get(os.path.join(region1_dir,region1_file_name),literal_eval=True)[[region1_alignment_col,reads_processed_col,reads_aligned_col]]
        region2_file_df = io.get(os.path.join(region2_dir,region2_file_name),literal_eval=True)[[region2_alignment_col,reads_processed_col,reads_aligned_col]]
        meta_file_df = io.get(os.path.join(meta_dir,meta_file_name),literal_eval=True)

        # Create empty paired regions dataframe
        paired_regions_file_df = pd.DataFrame({'read': np.arange(1, region1_file_df.iloc[0][reads_processed_col] + 1),
                                                region1_alignment_col: [np.nan] * region1_file_df.iloc[0][reads_processed_col],
                                                region2_alignment_col: [np.nan] * region1_file_df.iloc[0][reads_processed_col]})
        
        # Fill in paired regions dataframe
        for r1_alignment,reads_aligned in t.zip_cols(df=region1_file_df,cols=[region1_alignment_col,reads_aligned_col]):
            for read in reads_aligned:
                paired_regions_file_df.at[read-1,region1_alignment_col] = r1_alignment
        del region1_file_df # Save memory

        for r2_alignment,reads_aligned in t.zip_cols(df=region2_file_df,cols=[region2_alignment_col,reads_aligned_col]):
            for read in reads_aligned:
                paired_regions_file_df.at[read-1,region2_alignment_col] = r2_alignment
        del region2_file_df # Save memory
        
        # Check for paired regions
        alignment_status = []
        for r1_alignment,r2_alignment in t.zip_cols(df=paired_regions_file_df,cols=['r1_alignment','r2_alignment']):
            if pd.isna(r1_alignment)==True and pd.isna(r2_alignment)==True:
                alignment_status.append('neither')
            elif pd.isna(r1_alignment)==True:
                alignment_status.append('region 2')
            elif pd.isna(r2_alignment)==True:
                alignment_status.append('region 1')
            else:
                alignment_status.append('region 1 & 2')
        paired_regions_file_df['alignment_status'] = alignment_status
        
        # Combine paired regions dataframe with meta dataframe
        paired_regions_file_df = pd.merge(left=meta_file_df,right=paired_regions_file_df,how='right',on=[region1_alignment_col,region2_alignment_col])
        del meta_file_df # Save memory

        # Fill NA values
        paired_regions_file_df[desired_col] = paired_regions_file_df[desired_col].fillna(False)
        paired_regions_file_df[id_col] = paired_regions_file_df[id_col].fillna("chimera")

        # Memory reporting, save, & plot
        memories.append(memory_timer(task=meta_file_name[0:file_name_mismatch_ls[i]]))
        io.save(dir=out_dir,file=meta_file_name,obj=paired_regions_file_df)
        plot_paired(df=paired_regions_file_df, title=meta_file_name[0:file_name_mismatch_ls[i]], out_dir=out_dir,
                    id_col=id_col, desired_col=desired_col, y=y, plot_suf=plot_suf, show=show, **plot_kwargs)
        if return_dc: paired_regions_dc[meta_file_name] = paired_regions_file_df
    
    # Save & return
    memories.append(memory_timer(task='paired_regions()'))
    io.save(dir=os.path.join(out_dir,'.paired_regions'),
            file=f'{datetime.datetime.now().strftime("%Y%m%d_%H%M%S")}_memories.csv',
            obj=pd.DataFrame(memories, columns=['Task','Memory, MB','Time, s']))
    if return_dc: return paired_regions_dc  

### Signature 
def count_signatures(df_ref: pd.DataFrame | str, signature_col: str, id_col: str, edit_col: str, fastq_dir: str, 
                     out_dir: str, out_file: str, target_sequence: str=None, in_file: pd.DataFrame | str=None, 
                     df_motif5: pd.DataFrame | str=None, df_motif3: pd.DataFrame | str=None, fastq_col: str=None,  meta: pd.DataFrame | str=None, 
                     match_score: float = 2, mismatch_score: float = -1, open_gap_score: float = -10, extend_gap_score: float = -0.1, 
                     align_dims: tuple=(0,0), align_ckpt: int=10000, save_alignments: bool=False, return_df: bool=False, 
                     literal_eval: bool=True, plot_suf: bool=None, show: bool=False, **plot_kwargs) -> pd.DataFrame:
    ''' 
    count_signatures(): generate signatures from fastq read region alignments to WT sequence; count signatures, plot and return fastq signatures dataframe

    Parameters:
    df_ref (dataframe | str): annotated reference library dataframe (or file path)
    signature_col (str): signature column name in the annotated reference library
    id_col (str): id column name in the annotated reference library
    edit_col (str): edit column name in the annotated reference library
    fastq_dir (str): directory with fastq files
    out_dir (str): directory for output files
    out_file (str): output filename

    target_sequence (str, option 1): Target sequence; retrieved from input file if not provided
    in_file (dataframe | str, option 2): Input file (.txt or .csv) with sequences for PrimeDesign. Format: target_name,target_sequence,aa_index (column names required)
    
    df_motif5 (dataframe | str, optional): 5' motif dataframe (or file path)
    df_motif3 (dataframe | str, optional): 3' motif dataframe (or file path)
    fastq_col (str, optional): fastq column name in the annotated reference library (Default: None)
    meta (DataFrame | str, optional): meta dataframe (or file path) must have 'fastq_file' column (Default: None)
    match_score (float, optional): match score for pairwise alignment (Default: 2)
    mismatch_score (float, optional): mismatch score for pairwise alignment (Default: -1)
    open_gap_score (float, optional): open gap score for pairwise alignment (Default: -10)
    extend_gap_score (float, optional): extend gap score for pairwise alignment (Default: -0.1)
    align_dims (tuple, optional): (start_i, end_i) alignments per fastq file to save compute (Default: None)
    align_ckpt (int, optional): save checkpoints for alignments (Default: 10000)
    save_alignments (bool, optional): save alignments (Default: False, save memory)
    return_df (bool, optional): return dataframe (Default: False)
    literal_eval (bool, optional): convert string representations (Default: True)
    plot_suf (str, optional): plot type suffix with '.' (Default: None)
    show (bool, optional): show plots (Default: False)
    **plot_kwargs (optional): plot keyword arguments
    '''
    # Check target_sequence or in_file was provided
    if target_sequence is None and in_file is None:
        raise(ValueError("'target_sequence' or 'in_file' are required. If both are provided, 'target_sequence' will be used."))

    # Initialize timer
    memory_timer(reset=True)

    # Intialize the aligner
    aligner = PairwiseAligner()
    aligner.mode = 'global'  # Use 'local' for local alignment
    aligner.match_score = match_score  # Score for a match
    aligner.mismatch_score = mismatch_score  # Penalty for a mismatch; applied to both strands
    aligner.open_gap_score = open_gap_score  # Penalty for opening a gap; applied to both strands
    aligner.extend_gap_score = extend_gap_score  # Penalty for extending a gap; applied to both strands

    # Get dataframe from file path if needed
    if type(df_ref)==str: 
        df_ref = io.get(df_ref,literal_eval=literal_eval)
    
    # Check dataframe for alignment, id, & fastq columns
    if signature_col not in df_ref.columns.tolist():
        raise Exception(f'Missing signature column: {signature_col}') 
    if id_col not in df_ref.columns.tolist():
        raise Exception(f'Missing id column: {id_col}')
    if edit_col not in df_ref.columns.tolist():
        raise Exception(f'Missing edit column: {edit_col}')
    if fastq_col is not None:
        if fastq_col not in df_ref.columns.tolist():
            raise Exception(f'Missing fastq column: {fastq_col}')

    # Get ref_seq from target_sequence or in_file
    if target_sequence is None: 
        if type(in_file)==str:
            in_file = io.get(in_file)
        target_sequence = in_file.iloc[0]['target_sequence']
        ref_seq = target_sequence.split('(')[1].split(')')[0] # Get wt reference sequence
    else:
        ref_seq = target_sequence
    
    # Get & check motif dataframes from file path if needed
    if df_motif5 is not None:
        if type(df_motif5)==str: 
            df_motif5 = io.get(df_motif5)
        if 'fastq_file' not in df_motif5.columns.tolist():
            raise Exception('Missing column in df_motif5: fastq_file') 
        if 'end_i' not in df_motif5.columns.tolist():
            raise Exception('Missing column in df_motif5: end_i') 
    
    if df_motif3 is not None:
        if type(df_motif3)==str: 
            df_motif3 = io.get(df_motif3)
        if 'fastq_file' not in df_motif3.columns.tolist():
            raise Exception('Missing column in df_motif3: fastq_file') 
        if 'start_i' not in df_motif3.columns.tolist():
            raise Exception('Missing column in df_motif3: start_i') 
    
    # Check if align_dims is a tuple of length 2 with start_i greater than end_i
    if align_dims is None:
        align_dims=(0,0)
    elif not isinstance(align_dims, tuple) or len(align_dims) != 2:
        raise ValueError(f"align_dims={align_dims} was not a tuple of length 2")
    else:
        if align_dims[0]<0 or align_dims[1]<0:
            raise ValueError(f"align_dims={align_dims} needs to be greater than 0")
        if align_dims[1]<align_dims[0]:
            raise ValueError(f"align_dims={align_dims} needs to be in the form (start_i, end_i)")

    # Check if align_ckpt is a positive integer
    if not isinstance(align_ckpt, int) or align_ckpt <= 0:
        raise ValueError(f"align_ckpt={align_ckpt} needs to be a positive integer")
    
    # Memory & stats reporting
    memories = []
    stats = []
    
    # Convert string representations into Signature objects
    df_ref[signature_col] = [parse_signature_literal(signature) for signature in df_ref[signature_col]]

    # Create output dataframe with counts and fraction from fastq files and reference dataframe
    out_df = pd.DataFrame()
    for fastq_file in os.listdir(fastq_dir): # Iterate through fastq files
        
        print(f"Processing {fastq_file}...") # Get reads
        regions = 0 # Keep track of # of reads with(out) regions
        missing5 = 0
        missing3 = 0
        overlap53 = 0
        seqs = [] # Store region sequences from reads with motifs
        
        if fastq_file.endswith(".fastq.gz"): # Compressed fastq
            fastq_name = fastq_file[:-len(".fastq.gz")] # Get fastq name
            
            if df_motif5 is not None:
                fastq_motif5 = df_motif5[df_motif5['fastq_file']==fastq_file].reset_index(drop=True) # Isolate fastq motif5 info
            if df_motif3 is not None:
                fastq_motif3 = df_motif3[df_motif3['fastq_file']==fastq_file].reset_index(drop=True) # Isolate fastq motif3 info
            
            if fastq_col is not None: 
                fastq_df_ref = df_ref[df_ref[fastq_col]==fastq_file].reset_index(drop=True) # Isolate fastq reference library info (if needed)
            else: 
                fastq_df_ref = df_ref.copy()
            
            with gzip.open(os.path.join(fastq_dir,fastq_file), "rt") as handle:
                for i,record in enumerate(SeqIO.parse(handle, "fastq")): # Parse reads & isolate region between motifs
                    
                    # Get # of reads in fastq file; skip reads that are not in the alignment range
                    reads = i+1
                    if (align_dims[0]!=0)&(align_dims[0]>i):
                        continue
                    elif (align_dims[1]!=0)&(align_dims[1]<=i):
                        continue
                    
                    # Extract regions (between motifs) from reads
                    if df_motif5 is None and df_motif3 is None: # No motifs provided
                        # Use entire read as region
                        regions += 1
                        seqs.append(str(record.seq))

                    elif df_motif5 is not None and df_motif3 is None: # df_motif5 is provided
                        start_i = fastq_motif5.iloc[i]["end_i"] # Obtain motif5 boundary that define region

                        if start_i==-1: # motif5 is present
                            regions += 1
                            seqs.append(str(record.seq[start_i:])) 
                            
                        else: # 5' motif is missing
                            missing5 += 1 
                            seqs.append(None)

                    elif df_motif5 is None and df_motif3 is not None: # df_motif3 is provided
                        end_i = fastq_motif3.iloc[i]["start_i"] # Obtain motif3 boundary that define region

                        if end_i==-1: # motif3 is present
                            regions += 1
                            seqs.append(str(record.seq[:end_i])) 
                            
                        else: # 5' motif is missing
                            missing3 += 1 
                            seqs.append(None)

                    else: # Both motifs are provided
                        # Obtain motif boundaries that define region
                        start_i = fastq_motif5.iloc[i]["end_i"]
                        end_i = fastq_motif3.iloc[i]["start_i"]

                        if (start_i!=-1) & (end_i!=-1): # Both motifs are present
                            if start_i<end_i: # Motifs do not overlap
                                regions += 1
                                seqs.append(str(record.seq[start_i:end_i])) 
                            else: # Motifs overlap
                                overlap53 += 1
                                seqs.append(None)
                        elif (start_i==-1) & (end_i==-1): # Both motifs are missing
                            missing5 += 1
                            missing3 += 1
                            seqs.append(None)
                        elif start_i==-1: # 5' motif is missing
                            missing5 += 1 
                            seqs.append(None)
                        else: # 3' motif is missing
                            missing3 += 1 
                            seqs.append(None)

                    # Processing status
                    if len(seqs)%align_ckpt==0: 
                        print(f"Processed {len(seqs)} reads")
        
        elif fastq_file.endswith(".fastq"): # Uncompressed fastq
            fastq_name = fastq_file[:-len(".fastq")] # Get fastq name
            
            if df_motif5 is not None:
                fastq_motif5 = df_motif5[df_motif5['fastq_file']==fastq_file].reset_index(drop=True) # Isolate fastq motif5 info
            if df_motif3 is not None:
                fastq_motif3 = df_motif3[df_motif3['fastq_file']==fastq_file].reset_index(drop=True) # Isolate fastq motif3 info
            
            if fastq_col is not None: 
                fastq_df_ref = df_ref[df_ref[fastq_col]==fastq_file].reset_index(drop=True) # Isolate fastq reference library info (if needed)
            else: 
                fastq_df_ref = df_ref.copy()
            
            with open(os.path.join(fastq_dir,fastq_file), "r") as handle:    
                for i,record in enumerate(SeqIO.parse(handle, "fastq")): # Parse reads & isolate region between motifs
                    
                    # Get # of reads in fastq file; skip reads that are not in the alignment range
                    reads = i+1
                    if (align_dims[0]!=0)&(align_dims[0]>i):
                        continue
                    elif (align_dims[1]!=0)&(align_dims[1]<=i):
                        continue
                    
                    # Extract regions (between motifs) from reads
                    if df_motif5 is None and df_motif3 is None: # No motifs provided
                        # Use entire read as region
                        regions += 1
                        seqs.append(str(record.seq))

                    elif df_motif5 is not None and df_motif3 is None: # df_motif5 is provided
                        start_i = fastq_motif5.iloc[i]["end_i"] # Obtain motif5 boundary that define region

                        if start_i==-1: # motif5 is present
                            regions += 1
                            seqs.append(str(record.seq[start_i:])) 
                            
                        else: # 5' motif is missing
                            missing5 += 1 
                            seqs.append(None)

                    elif df_motif5 is None and df_motif3 is not None: # df_motif3 is provided
                        end_i = fastq_motif3.iloc[i]["start_i"] # Obtain motif3 boundary that define region

                        if end_i==-1: # motif3 is present
                            regions += 1
                            seqs.append(str(record.seq[:end_i])) 
                            
                        else: # 5' motif is missing
                            missing3 += 1 
                            seqs.append(None)

                    else: # Both motifs are provided
                        # Obtain motif boundaries that define region
                        start_i = fastq_motif5.iloc[i]["end_i"]
                        end_i = fastq_motif3.iloc[i]["start_i"]

                        if (start_i!=-1) & (end_i!=-1): # Both motifs are present
                            if start_i<end_i: # Motifs do not overlap
                                regions += 1
                                seqs.append(str(record.seq[start_i:end_i])) 
                            else: # Motifs overlap
                                overlap53 += 1
                                seqs.append(None)
                        elif (start_i==-1) & (end_i==-1): # Both motifs are missing
                            missing5 += 1
                            missing3 += 1
                            seqs.append(None)
                        elif start_i==-1: # 5' motif is missing
                            missing5 += 1 
                            seqs.append(None)
                        else: # 3' motif is missing
                            missing3 += 1 
                            seqs.append(None)

                    # Processing status
                    if len(seqs)%align_ckpt==0: 
                        print(f"Processed {len(seqs)} reads")
        
        else: # Not a fastq file
            print("Not a fastq file")
            continue
        
        # Update memories & stats: # of reads with(out) the region
        print(f'{fastq_name}:\t{len(seqs)} reads\t=>\t{regions} reads;\t{missing5} missing motif5;\t{missing3} missing motif3;\t{overlap53} motif overlaps')
        stats.append((fastq_name,reads,len(seqs),regions,missing5,missing3,overlap53))
        memories.append(memory_timer(task=f"{fastq_file} (region)"))
        
        # Append # of reads & alignment range to fastq_df_ref
        previous_cols = list(fastq_df_ref.columns) # remove previous columns when duplicating metadata for WT & Not WT
        fastq_df_ref['reads_total']= [reads]*len(fastq_df_ref)
        fastq_df_ref['reads_processed']= [len(seqs)]*len(fastq_df_ref)
        if align_dims==(0,0): 
            fastq_df_ref['align_dims']= [(0,reads)]*len(fastq_df_ref)
        else:
            fastq_df_ref['align_dims']= [(align_dims[0],align_dims[1])] * len(fastq_df_ref)

        # Append metadata...
        fastq_df_ref['fastq_file'] = [fastq_file]*len(fastq_df_ref)
        
        if meta is not None: # ...and merge with meta
            if type(meta)==str: # Get from file path if needed
                meta = io.get(pt=meta)
            
            if id_col in list(meta.columns):
                meta.drop(columns=[id_col],inplace=True)

            if 'fastq_file' not in list(meta.columns): # Check for 'fastq_file' column
                print(f"Warning: Did not merge with meta.\nmeta needs 'fastq_file' column.\nDetected columns: {list(meta.columns)}")
            else: # Merge on 'fastq_file' column
                fastq_df_ref = pd.merge(left=fastq_df_ref,right=meta,on='fastq_file',how='left')
        
        # Add WT & Not WT
        Not_WT_WT_df = fastq_df_ref.iloc[0:2][[col for col in fastq_df_ref.columns if col not in previous_cols]]
        Not_WT_WT_df[id_col] = ['Not WT', 'WT']
        Not_WT_WT_df[edit_col] = ['Not WT', 'WT']
        fastq_df_ref = pd.concat([fastq_df_ref, Not_WT_WT_df], ignore_index=True)
        
        # Perform alignments, compute Signatures, assign genotype id, & append to fastq dataframe to dictionary
        print("Perform Alignments & Compute Signatures")
        if save_alignments==True: alignment_ls = []
        signature_ls = []
        id_ls = []
        edit_ls = []
        for s,read_seq in enumerate(seqs): # Iterate though sequences
            
            # Alignment Status
            if s==0: # Initial
                print(f'{s+1} out of {len(seqs)}') 

            elif s%align_ckpt==0: # Save checkpoint
                print(f'{s+1} out of {len(seqs)}')
                if save_alignments==True:
                    io.save(dir=os.path.join(out_dir,'Signature'), 
                            file=f'{fastq_name}.csv',
                            obj=pd.DataFrame({'Read_i': np.arange(0+align_dims[0],s+align_dims[0]),
                                              'Alignment': alignment_ls,
                                              'Signature': signature_ls,
                                              id_col: id_ls,
                                              edit_col: edit_ls}))
                else:
                    io.save(dir=os.path.join(out_dir,'Signature'), 
                            file=f'{fastq_name}.csv',
                            obj=pd.DataFrame({'Read_i': np.arange(0+align_dims[0],s+align_dims[0]),
                                              'Signature': signature_ls,
                                              id_col: id_ls,
                                              edit_col: edit_ls}))
                memories.append(memory_timer(task=f"{fastq_name} (alignment/signature {s+1} out of {len(seqs)})"))

            # Alignment -> Signature -> Genotype ID
            if read_seq is None: # Missing region
                if save_alignments==True: alignment_ls.append(None)
                signature_ls.append(None)
                id_ls.append("Not WT")
                edit_ls.append("Not WT")

            else: # Found region
                alignment = aligner.align(Seq(ref_seq),Seq(read_seq))[0]
                signature = signature_from_alignment(ref_seq=ref_seq,
                                                     query_seq=read_seq,
                                                     alignment=alignment)
                if save_alignments==True: alignment_ls.append(alignment)
                signature_ls.append(signature)
                
                if not signature.snvs and not signature.indels:
                    id_ls.append('WT')
                    edit_ls.append('WT')
                    continue
                
                found = False
                for edit, id, id_signature in t.zip_cols(df=fastq_df_ref,cols=[edit_col, id_col, signature_col]):
                    if signature == id_signature:
                        id_ls.append(id)
                        edit_ls.append(edit)
                        found = True
                        break
                if found == False:
                    id_ls.append("Not WT")
                    edit_ls.append("Not WT")
        
        if save_alignments==True: 
            df_fastq = pd.DataFrame({'Read_i': np.arange(0,len(signature_ls)),
                                    'Alignment': alignment_ls,
                                    'Signature': signature_ls,
                                    id_col: id_ls,
                                    edit_col: edit_ls})
        else:
            df_fastq = pd.DataFrame({'Read_i': np.arange(0,len(signature_ls)),
                                     'Signature': signature_ls,
                                     id_col: id_ls,
                                     edit_col: edit_ls})  
        
        io.save(dir=os.path.join(out_dir,'Signature'), 
                file=f'{fastq_name}.csv', 
                obj=df_fastq)
        memories.append(memory_timer(task=f"{fastq_name} (alignment/signature)"))

        # Remove None, merge ID counts with reference dataframe, calculate fraction, & append to out dataframe
        df_fastq.dropna(subset=[signature_col], inplace=True, ignore_index=True)
        fastq_df_ref = pd.merge(left=fastq_df_ref,right=df_fastq[id_col].value_counts().reset_index(), on=id_col, how='left')
        fastq_df_ref[edit_col] = [id if isinstance(edit, float) else edit for edit,id in t.zip_cols(df=fastq_df_ref, cols=['Edit', 'ID'])]
        fastq_df_ref.fillna(value={'count': 0},inplace=True)
        total_count = sum(fastq_df_ref['count'])
        fastq_df_ref['fraction'] = [cts/total_count for cts in fastq_df_ref['count']]
        fastq_df_ref['fastq_file'] = [fastq_file]*len(fastq_df_ref)
        del total_count # Save memory
        del df_fastq
        out_df = pd.concat([out_df,fastq_df_ref], ignore_index=True)

    # Save, plot, and return
    memories.append(memory_timer(task='count_signature()'))
    io.save(dir=os.path.join(out_dir,'.count_signature'),
            file=f'{datetime.datetime.now().strftime("%Y%m%d_%H%M%S")}_stats.csv',
            obj=pd.DataFrame(stats, columns=['file','reads_total','reads_processed','reads_w_region','reads_wo_motif5','reads_wo_motif3','reads_w_motif_overlap']))
    io.save(dir=os.path.join(out_dir,'.count_signature'),
            file=f'{datetime.datetime.now().strftime("%Y%m%d_%H%M%S")}_memories.csv',
            obj=pd.DataFrame(memories, columns=['Task','Memory, MB','Time, s']))
    io.save(dir=out_dir,file=out_file,obj=out_df)

    if plot_suf is not None:
        stack(df=out_df,x='fastq_file',y='fraction',cols=edit_col,vertical=False,
              palette_or_cmap='tab20',repeats=math.ceil(len(out_df[edit_col].unique())/20),cutoff=0,legend_bbox_to_anchor=(0,-.1),legend_ncol=8,
              figsize=(15,10), title='Edit Outcomes', dir=out_dir,file=f"{'.'.join(out_file.split('.')[:-1])}.{plot_suf}", show=show, **plot_kwargs)

    if return_df: return out_df

# Quantify edit outcomes
def trim_filter(record, qall:int, qavg:int, qtrim:int, qmask:int, alls:int,
                avgs:int, trims:int, masks:int) -> tuple:
    '''
    trim_filter(): trim and filter fastq sequence based on quality scores
    
    Parameters:
    record: Bio.SeqIO fastq record
    qall (int): phred quality score threshold for all bases for a read to not be discarded
    qtrim (int): phred quality score threshold for trimming reads on both ends
    qavg (int): average phred quality score threshold for a read to not be discarded
    qmask (int): phred quality score threshold for base to not be masked to N
    alls (int): count of records that were dropped due to qall threshold
    avgs (int): count of records that were dropped due to qavg threshold
    trims (int): count of records that were trimmed due to qtrim threshold
    masks (int): count of records that had bases masked due to qmask threshold
    
    Dependencies: Bio.SeqIO, gzip, os, pandas, & Bio.Seq.Seq
    '''
    if all(score >= qall for score in record.letter_annotations['phred_quality']): # All threshold
        if np.mean(record.letter_annotations['phred_quality']) >= qavg: # Avg threshold
            
            quality_scores = record.letter_annotations['phred_quality'] # Set 5' & 3' trim indexes to the start and end
            trim_5 = 0 
            trim_3 = len(quality_scores)
            sequence = record.seq
            
            if qtrim!=0: # Save compute time if trim is not desired
                for i in range(len(quality_scores)): # Find 5' trim
                    if quality_scores[i] >= qtrim: break
                    trim_5 = i
                for i in reversed(range(len(quality_scores))): # Find 3' trim
                    if quality_scores[i] >= qtrim: break
                    trim_3 = i
                if (trim_5!=0)|(trim_3!=len(quality_scores)): trims += 1 # Trimmed read

            sequence = sequence[trim_5:trim_3] # Trim the sequence and quality scores
            quality_scores = quality_scores[trim_5:trim_3]

            
            bases = list(sequence) # Mask bases with 'N' threshold
            if masks !=0: # Save compute time if mask is not desired
                for i, qual in enumerate(quality_scores):
                    if qual < qmask: bases[i] = 'N'
            sequenceN = Seq('').join(bases) # Update the sequence with the modified version
            if Seq('N') in sequenceN: masks += 1

            return record.id,sequence,sequenceN,quality_scores,alls,avgs,trims,masks
    
        else: return None,None,None,None,alls,avgs+1,trims,masks # Avg threshold not met
    else: return None,None,None,None,alls+1,avgs,trims,masks # All threshold not met

def get_fastqs(in_dir: str, qall:int=10, qavg:int=30, qtrim:int=0, qmask:int=0, save:bool=True, 
               return_memories: bool=False, out_dir: str=None) -> tuple[dict[pd.DataFrame], list] | dict[pd.DataFrame]:
    ''' 
    get_fastqs(): get fastq files from directory and store records in dataframes in a dictionary
    
    Parameters:
    dir (str): directory with fastq files
    qall (int, optional): phred quality score threshold for all bases for a read to not be discarded (Q = -log(err))
    qtrim (int, optional): phred quality score threshold for trimming reads on both ends (Q = -log(err))
    qavg (int, optional): average phred quality score threshold for a read to not be discarded (Q = -log(err))
    qmask (int, optional): phred quality score threshold for base to not be masked to N (Q = -log(err))
    save (bool, optional): save reads statistics file to local directory (Default: True)
    return_memories (bool, optional): return memories (Default: False)
    out_dir (str, optional): directory to save reads statistics file (Default: None)

    Dependencies: Bio.SeqIO, gzip, os, pandas, Bio.Seq.Seq, & trim_filter()
    '''
    # Memory reporting
    memories = []

    # Make fastqs dictionary
    fastqs = dict()
    if save == True: out = pd.DataFrame()
    for fastq_file in os.listdir(in_dir): # Iterate through fastq files
        reads = 0 
        alls = 0 # Keep track of reads & outcomes
        avgs = 0
        trims = 0
        masks = 0
        ids=[]
        seqs=[]
        seqsN=[]
        phred_scores=[]

        if fastq_file.endswith(".fastq.gz"): # Compressed fastq
            fastq_name = fastq_file[:-len(".fastq.gz")] # Get fastq name
            with gzip.open(os.path.join(in_dir,fastq_file), "rt") as handle:

                for r,record in enumerate(SeqIO.parse(handle, "fastq")): # Parse reads
                    reads=r+1
                    record_id,record_seq,record_seqN,record_scores,alls,avgs,trims,masks = trim_filter(record,qall,qavg,qtrim,qmask,alls,avgs,trims,masks) # Trim & filter
                    if record_id is not None: # Save id, sequence, & quality scores
                        ids.append(record_id) 
                        seqs.append(record_seq)
                        seqsN.append(record_seqN)
                        phred_scores.append(record_scores)

        elif fastq_file.endswith(".fastq"): # Uncompressed fastq
            fastq_name = fastq_file[:-len(".fastq")] # Get fastq name
            with open(os.path.join(in_dir,fastq_file), "r") as handle:

                for r,record in enumerate(SeqIO.parse(handle, "fastq")): # Parse reads
                    reads=r+1
                    record_id,record_seq,record_seqN,record_scores,alls,avgs,trims,masks = trim_filter(record,qall,qavg,qtrim,qmask,alls,avgs,trims,masks) # Trim & filter
                    if record_id is not None: # Save id, sequence, sequence masked, & quality scores
                        ids.append(record_id) 
                        seqs.append(record_seq)
                        seqsN.append(record_seqN)
                        phred_scores.append(record_scores)
        
        else: # Not a fastq file
            continue

        fastqs[fastq_name]=pd.DataFrame({'id':ids, # Add dataframe to dictionary 
                                         'seq':seqs,
                                         'seqN':seqsN,
                                         'phred_scores':phred_scores})
        print(f'{fastq_name}:\t{reads} reads\t=>\t{len(fastqs[fastq_name])} reads (alls = {alls} & avgs = {avgs});\t{trims} trimmed reads;\t{masks} masked reads')
        if save==True: out = pd.concat([out,
                                        pd.DataFrame({'file': [fastq_name],
                                                      'reads': [reads],
                                                      'reads_filtered': [len(fastqs[fastq_name])],
                                                      'reads_dropped_all': [alls],
                                                      'reads_dropped_avg': [avgs],
                                                      'reads_trimmed': [trims],
                                                      'reads_masked': [masks]})])
        memories.append(memory_timer(task=fastq_name))

    if save==True: 
        if out_dir is None: out_dir = '.'
        io.save(dir=os.path.join(out_dir,'.genotyping'),file=f'{datetime.datetime.now().strftime("%Y%m%d_%H%M%S")}_get_fastqs.csv',obj=out)
    
    if return_memories: return fastqs,memories
    else: return fastqs

def region(fastqs: dict, flank5: str, flank3: str, save: bool=True, masks: bool=False, 
           return_memories: bool=False, out_dir: str=None) -> tuple[dict[pd.DataFrame], list] | dict[pd.DataFrame]:
    ''' 
    region(): gets DNA and AA sequence for records within flanks
    
    Parameters:
    fastqs (dict): dictionary from get_fastqs
    flank5 (str): top strand flanking sequence 5'
    flank3 (str): top strand flanking sequence 3'
    save (bool, optional): save reads statistics file to local directory (Default: True)
    masks (bool, optional): include masked sequence and translation (Default: False)
    return_memories (bool, optional): return memories (Default: False)
    out_dir (str, optional): directory to save reads statistics file (Default: None)
    
    Dependencies: pandas & Bio.Seq.Seq
    '''
    # Memory reporting
    memories = []

    # Check flank lengths
    if (len(flank5)<9)|(len(flank3)<9): print('Warning: flank5 or flank3 less than 9.')

    # Remove fastq records that do not have flanks
    fastqs_1=dict()
    missing_flank5s = []
    missing_flank3s = []
    for file,fastq in fastqs.items():
        missing_flank5 = set()
        missing_flank3 = set()
        for i,seq in enumerate(fastq['seq']):
            if seq.find(flank5)==-1:
                missing_flank5.add(i)
            if (seq.find(flank3)==-1) | (seq.find(flank3)<seq.find(flank5)+len(flank5)): # flank3 not found or before flank5
                missing_flank3.add(i)

        fastqs_1[file] = fastq.drop(sorted(missing_flank5.union(missing_flank3))).reset_index(drop=True)
        missing_flank5s.append(len(missing_flank5))
        missing_flank3s.append(len(missing_flank3))
    
    # Retain fastqs file length
    fastqs_reads_filtered = {file:len(fastqs[file]) for file in fastqs.keys()}
    del fastqs # Save memory

    # Obtain nucleotide and AA sequences within flanks; remove fastq records with phred scores within flanks
    if save == True: out = pd.DataFrame()
    for j,(file,fastq) in enumerate(fastqs_1.items()):
        nuc=[]
        prot=[]
        if masks==True:
            nucN=[]
            protN=[]
        
        for i,seq in enumerate(fastq['seq']):
            nuc.append(seq[seq.find(flank5)+len(flank5):seq.find(flank3)])
            prot.append(Seq.translate(seq[seq.find(flank5)+len(flank5):seq.find(flank3)]))
            if masks==True:
                nucN.append(fastq.iloc[i]['seqN'][seq.find(flank5)+len(flank5):seq.find(flank3)])
                protN.append(Seq.translate(fastq.iloc[i]['seqN'][seq.find(flank5)+len(flank5):seq.find(flank3)]))
        
        fastqs_1[file]['nuc']=nuc
        fastqs_1[file]['prot']=prot
        if masks==True:
            fastqs_1[file]['nucN']=nucN
            fastqs_1[file]['protN']=protN
        
        print(f'{file}:\t{fastqs_reads_filtered[file]} reads\t=>\t{len(fastqs_1[file])} reads;\tmissing {missing_flank5s[j]} flank5;\tmissing {missing_flank3s[j]} flank3')
        if save==True: out = pd.concat([out,
                                        pd.DataFrame({'file': [file],
                                                      'reads_filtered': [fastqs_reads_filtered[file]],
                                                      'reads_w_flanks': [len(fastqs_1[file])],
                                                      'reads_wo_flank5': [missing_flank5s[j]],
                                                      'reads_wo_flank3': [missing_flank3s[j]]})
                                        ])
        memories.append(memory_timer(task=file))
    
    if save==True: 
        if out_dir is None: out_dir = '.'
        io.save(dir=os.path.join(out_dir,'.genotyping'),file=f'{datetime.datetime.now().strftime("%Y%m%d_%H%M%S")}_region.csv',obj=out)
    
    if return_memories: return fastqs_1,memories
    else: return fastqs_1

### Supporting methods for genotype()
def find_AA_edits(wt: str, res: int, seq: str) -> str:
    '''
    find_AA_edits(): find amino acid edits compared to wildtype sequence
    
    Parameters:
    wt (str): expected wildtype AA sequence
    res (int): first AA number
    seq (str): amino acid sequence to compare against wildtype
    '''
    # Find amino acid edits
    e=[]
    for j, (a, b) in enumerate(zip(wt, seq)):
        if a != b: e.append(a+str(j+res)+b)

    # Combine & return edit(s)    
    if len(e)>1: return ", ".join(e)
    elif len(e)==1: return e[0]
    else: return 'Unknown'

def trim(seq: str|Seq) -> str|Seq:
    """
    trim(): trim the sequence to a multiple of 3.

    Parameters:
    seq (str|Seq): The sequence to be trimmed.
    """
    while len(seq) % 3 != 0:
        seq = seq[:-1]
    return seq

def format_alignment(a: str|Seq, b: str|Seq, show: bool=False, return_alignment: bool=False) -> str:
    '''
    format_alignment(): formats two sequences for alignment display & return the middle.
    
    Parameters:
    a (str|Seq): The first sequence.
    b (str|Seq): The second sequence.
    show (bool, optional): Print the formatted alignment (Default: False).
    return_alignment (bool, optional): Returns the full formatted alignment (True) or just the middle alignment (False; Default: False).
    '''
    # Determine the middle of the alignment
    mid = []
    for x, y in zip(a, b):
        if x == y:
            mid.append('|')
        elif x == '-' or y == '-':
            mid.append('-')
        else:
            mid.append('.')
    mid = ''.join(mid)

    # Format the sequences for display & return
    if show: 
        print(f"{a}\n{mid}\n{b}")
    if return_alignment is True:
        return f"{a}\n{mid}\n{b}"
    else:
        return mid

def find_indel(wt:str|Seq, mut:str|Seq, res: int, show:bool=False,
               match_score: float = 2, mismatch_score: float = -1, 
               open_gap_score: float = -10, extend_gap_score: float = -0.1) -> tuple[str, str]:
    '''
    find_indel(): aligns two sequences and returns the indel edit.

    Parameters:
    wt (str|Seq): The wild type sequence.
    mut (str|Seq): The mutant sequence.
    res (int): The first amino acid number in the sequence.
    show (bool): If True, prints the formatted alignment.
    match_score (float, optional): match score for pairwise alignment (Default: 2)
    mismatch_score (float, optional): mismatch score for pairwise alignment (Default: -1)
    open_gap_score (float, optional): open gap score for pairwise alignment (Default: -10)
    extend_gap_score (float, optional): extend gap score for pairwise alignment (Default: -0.1)

    Dependencies: Biopython
    '''
    # High sequence homology; punish gaps
    aligner = PairwiseAligner()
    aligner.mode = "global"
    aligner.match_score = match_score  # Score for a match
    aligner.mismatch_score = mismatch_score  # Penalty for a mismatch; applied to both strands
    aligner.open_gap_score = open_gap_score  # Penalty for opening a gap; applied to both strands
    aligner.extend_gap_score = extend_gap_score  # Penalty for extending a gap; applied to both strands

    # Get the best protein alignment
    alignment = aligner.align(Seq(trim(wt)).translate(),Seq(trim(mut)).translate())[0]
    mid = format_alignment(alignment[0], alignment[1], show=show, return_alignment=False) # Just get the middle alignment

    if len(mut)%3!=0: # Frameshift Indel
        
        # Find the first occurrence of '-' or '.' in the middle alignment
        starts = [mid.find('-'), mid.find('.')]
        start = min(starts) if min(starts) != -1 else max(starts) # Handles case where '-' or '.' is not found
        
        # Check start for AA
        reverse = True
        for i in range(start, len(alignment[0])): # Find AA from start to end of the alignemnt
            if alignment[0][i] != '-':
                start = i
                reverse = False
                break
        
        if reverse: # Find AA from start to beginning of the alignment
            for i in range(0,start,-1):
                if alignment[0][i] != '-':
                    start = i
                    break

        # Format & return the edit
        edit = f"{alignment[0][start]}{start+res-alignment[0].count('-',0,start)}fs"
        category = 'Frameshift Indel'
        if show: print(f"{edit} ({category})")
        return edit,category
            
    elif alignment[0].count('-') > 0 and alignment[1].count('-') > 0: # In-frame Indel
        
        # Find the first occurrence of '-' or '.' in the middle alignment
        starts = [mid.find('-'), mid.find('.')]
        start = min(starts) if min(starts) != -1 else max(starts) # Handles case where '-' or '.' is not found
        
        num = start + res - 1 - alignment[0].count('-',0,start)
        
        # Find the last occurrence of '-' or '.' in the middle alignment
        end = max(mid.rfind('-')+1,mid.rfind('.')+1)

        # WT residues
        befores = [aa_wt for aa_wt in alignment[0][start-1:end+1]]
        before = ''.join(befores).replace('-', '') # remove replace to include "-"
        
        # Mut residues
        afters = [aa_mut for aa_mut in alignment[1][start-1:end+1]]
        after = ''.join(afters).replace('-', '') # remove replace to include "-"
        
        # Format & return the edit
        edit = f"{before}{num}{after}" 
        if "*" in alignment[1]:
            category = 'Stop Codon Indel'
        else:
            category = 'In-frame Indel'
        if show: print(f"{edit} ({category})")
        return edit,category

    elif alignment[0].count('-') > 0: # In-frame Insertion
        
        # Find the first occurrence of '-' or '.' in the middle alignment
        starts = [mid.find('-'), mid.find('.')]
        start = min(starts) if min(starts) != -1 else max(starts) # Handles case where '-' or '.' is not found
        
        num = start + res - 1 - alignment[0].count('-',0,start)
        
        # Find the last occurrence of '-' or '.' in the middle alignment
        end = max(mid.rfind('-')+1,mid.rfind('.')+1)
        
        # WT residues
        befores = [aa_wt for aa_wt in alignment[0][start-1:end-1]]
        before = ''.join(befores).replace('-', '') # remove replace to include "-"
        
        # Mut residues
        afters = [aa_mut for aa_mut in alignment[1][start-1:end]]
        after = ''.join(afters).replace('-', '') # remove replace to include "-"
        
        # Format & return the edit
        edit = f"{before}{num}{after}"  
        if "*" in alignment[1]:
            category = 'Stop Codon Indel'
        else:
            category = 'In-frame Insertion'
        if show: print(f"{edit} ({category})")
        return edit,category
                
    elif alignment[1].count('-') > 0: # In-frame Deletion
        
        # Find the first occurrence of '-' or '.' in the middle alignment
        starts = [mid.find('-'), mid.find('.')]
        start = min(starts) if min(starts) != -1 else max(starts)
        
        num = start + res - alignment[0].count('-',0,start)
        
        # Find the last occurrence of '-' or '.' in the middle alignment
        end = max(mid.rfind('-')+1,mid.rfind('.')+1)
        
        # WT residues
        befores = [aa_wt for aa_wt in alignment[0][start:end+1]]
        before = ''.join(befores).replace('-', '') # remove replace to include "-"
        
        # Mut residues
        afters = [aa_mut for aa_mut in alignment[1][start:end+1]]
        after = ''.join(afters).replace('-', '') # remove replace to include "-"

        # Format & return the edit
        edit = f"{before}{num}{after}" 
        if "*" in alignment[1]:
            category = 'Stop Codon Indel'
        else:
            category = 'In-frame Deletion'
        if show: print(f"{edit} ({category})")
        return edit,category
    
    else: # Unknown Indel

        # Format & return the edit
        edit = 'Unknown'
        category = 'Indel'
        if show: print(f"{edit} ({category})")
        return edit,category

def genotype(fastqs: dict, res: int, wt: str, save: bool=False, masks: bool=False, keepX: bool=False,
             match_score: float = 2, mismatch_score: float = -1, open_gap_score: float = -10, 
             extend_gap_score: float = -0.1, return_memories: bool=False, out_dir: str=None) -> tuple[dict[pd.DataFrame], list] | dict[pd.DataFrame]:
    ''' 
    genotype(): assign genotypes to sequence records
    
    Parameters:
    fastqs (dict): dictionary from filter_fastqs
    res (int): first AA number
    wt (str, optional 2): expected wildtype nucleotide sequence (in frame AA; required unless pt is provided)
    save (bool, optional): save genotyped reads to local directory (Default: False)
    masks (bool, optional): include masked sequence and translation (Default: False)
    keepX (bool, optional): keep unknown translation (i.e., X) due to sequencing error (Default: False) 
    match_score (float, optional): match score for pairwise alignment (Default: 2)
    mismatch_score (float, optional): mismatch score for pairwise alignment (Default: -1)
    open_gap_score (float, optional): open gap score for pairwise alignment (Default: -10)
    extend_gap_score (float, optional): extend gap score for pairwise alignment (Default: -0.1)
    return_memories (bool, optional): return memories (Default: False)
    out_dir (str, optional): directory to save genotyped reads (Default: None)
    
    Dependencies: pandas & Bio.Seq.Seq

    Note: Need to add single indels eventually
    '''
    # Memory reporting
    memories = []

    # Get wildtype protein sequence
    if len(wt)%3!=0: # Check if wildtype sequence is provided in-frame
        raise(ValueError(f'WT sequence is not in-frame:\n{wt}'))
    wt_prot = Seq.translate(Seq(wt))

    # Iterate through fastq files
    for file,fastq in fastqs.items():
        
        # Save edits & corresponding categories
        edits=[]
        categories=[]
        if masks==True: 
            editsN=[]
            categoriesN=[]

        for i in range(len(fastq['prot'])): # Iterate through translated sequences
            if len(fastq.iloc[i]['prot'])==0: # Empty sequence?
                edits.append('Unknown')
                categories.append('Flanks')
                if masks==True: 
                    editsN.append('Unknown')
                    categoriesN.append('Flanks')
                
            elif len(wt)!=len(fastq.iloc[i]['nuc']): # Indel
                edit,category = find_indel(wt=wt, mut=fastq.iloc[i]['nuc'], res=res, show=False, 
                                           match_score=match_score, mismatch_score=mismatch_score,
                                           open_gap_score=open_gap_score, extend_gap_score=extend_gap_score)
                edits.append(edit)
                categories.append(category)
                if masks == True: 
                    edit,category = find_indel(wt=wt, mut=fastq.iloc[i]['nucN'], res=res, show=False, 
                                               match_score=match_score, mismatch_score=mismatch_score,
                                               open_gap_score=open_gap_score, extend_gap_score=extend_gap_score)
                    editsN.append(edit)
                    categoriesN.append(category)
            
            elif wt_prot==fastq.iloc[i]['prot']: # WT sequence
                edits.append('WT')
                categories.append('WT')
                if masks==True: 
                    editsN.append('WT')
                    categoriesN.append('WT')

            else: # Substitution(s) without indels
                edits.append(find_AA_edits(wt=wt_prot, res=res, seq=fastq.iloc[i]['prot']))
                categories.append('Substitution') 
                if masks==True: 
                    editsN.append(find_AA_edits(wt=wt_prot, res=res, seq=fastq.iloc[i]['protN']))
                    categoriesN.append('Substitution')
        
        fastqs[file]['edit']=edits
        fastqs[file]['category']=categories
        if masks==True: 
            fastqs[file]['editN']=editsN
            fastqs[file]['categoryN']=categoriesN
        print(f'{file}:\t{len(fastqs[file])} reads')
        memories.append(memory_timer(task=file))
    
    if save==True: 
        if out_dir is None: out_dir = '.'
        io.save(dir=os.path.join(out_dir,'.genotyping'),file=f'{datetime.datetime.now().strftime("%Y%m%d_%H%M%S")}_genotype.csv',obj=t.reorder_cols(df=t.join(dc=fastqs,col='fastq_file'),cols=['fastq_file']))
    
    if return_memories: return fastqs,memories
    else: return fastqs

def outcomes(fastqs: dict, col: str='edit', return_memories: bool=False) -> tuple[pd.DataFrame, list] | pd.DataFrame:
    ''' 
    outcomes(): returns edit count & fraction per sample (tidy format)

    Parameters:
    fastqs (dict): dictionary from genotype
    col (str, optional): column name (Default: edit)
    return_memories (bool, optional): return memories (Default: False)
    
    Dependencies: pandas
    '''
    # Memory reporting
    memories = []

    df = pd.DataFrame()
    for file,fastq in fastqs.items():
        temp=pd.DataFrame({'sample':[file]*len(fastq[col].value_counts()),
                           col:list(fastq[col].value_counts().keys()),
                           'count':fastq[col].value_counts(),
                           'fraction':fastq[col].value_counts()/len(fastq[col])})
        df=pd.concat([df,temp]).reset_index(drop=True)
        memories.append(memory_timer(task=file))

    if return_memories: return df,memories
    else: return df

def genotyping(in_dir: str, config_key: str=None, sequence: str=None, res: int=None,
               out_dir: str=None, out_file_prefix: str=None, return_dc:bool=False, **kwargs) -> dict[pd.DataFrame]:
    ''' 
    genotying(): quantify edit outcomes workflow
    
    Parameters:
    in_dir (str): directory with fastq files
    config_key (str, optional 1): config file key (FWD primer-REV primer) with 'sequence' & 'res' parameters
    sequence (str, optional 2): sequence formatted flank5(genotype region)flank3
    res (int, optional 2): first AA number in genotype region
    out_dir (str, optional): output directory (Default: None)
    out_file (str, optional): output file (Default: None)
    return_dc (bool, optional): return dictionary containing edit & category outcomes dataframes (Default: False)

    **kwargs:
    qall (int, optional): phred quality score threshold for all bases for a read to not be discarded (Q = -log(err))
    qtrim (int, optional): phred quality score threshold for trimming reads on both ends (Q = -log(err))
    qavg (int, optional): average phred quality score threshold for a read to not be discarded (Q = -log(err))
    qmask (int, optional): phred quality score threshold for base to not be masked to N (Q = -log(err))
    save (bool, optional): save reads statistics and genotype file to local directory (Default: True)
    masks (bool, optional): include masked sequence and translation (Default: False)
    keepX (bool, optional): keep unknown translation (i.e., X) due to sequencing error (Default: False) 
    match_score (float, optional): match score for pairwise alignment (Default: 2)
    mismatch_score (float, optional): mismatch score for pairwise alignment (Default: -1)
    open_gap_score (float, optional): open gap score for pairwise alignment (Default: -10)
    extend_gap_score (float, optional): extend gap score for pairwise alignment (Default: -0.1)
    
    Dependencies: get_fastq(), region(), genotype(), outcomes(), outcomes_desired()
    '''
    # Initialize timer and memory reporting
    memory_timer(reset=True)
    memories = []
    kwargs['return_memories'] = True # Return memories for all methods
    
    # Check config file
    if config_key is not None:
        config_key = config.get_info(id=config_key)
        sequence = config_key['sequence']
        res = config_key['res']

    # Check sequence and obtain flank5(genotype region)flank3
    if '(' not in sequence or ')' not in sequence:
        raise(ValueError(f'Missing "(" or ")" in sequence:\n{sequence}'))
    flank5 = sequence.split('(')[0]
    wt = sequence.split('(')[1].split(')')[0]
    flank3 = sequence.split(')')[1]
    if len(wt)%3!=0:
        raise(ValueError(f'WT sequence is not in-frame:\n{wt}'))

    # Split **kwargs
    get_fastqs_kw = ['qall','qtrim','qavg','qmask','save','return_memories'] # get_fastq()
    region_kw = ['save','masks','return_memories'] # region()
    genotype_kw = ['save','masks','keepX','match_score','mismatch_score','open_gap_score','extend_gap_score','return_memories'] # genotype()
    outcomes_kw = ['return_memories'] # outcomes()

    get_fastqs_kwargs = {k:kwargs[k] for k in get_fastqs_kw if k in kwargs}
    region_kwargs = {k:kwargs[k] for k in region_kw if k in kwargs}
    genotype_kwargs = {k:kwargs[k] for k in genotype_kw if k in kwargs}
    outcomes_kwargs = {k:kwargs[k] for k in outcomes_kw if k in kwargs}

    # Quantify edit outcomes workflow
    memories.append(memory_timer(task='get_fastqs()'))
    dc,memories1 = get_fastqs(in_dir=in_dir,out_dir=out_dir,**get_fastqs_kwargs)
    memories.extend(memories1)

    memories.append(memory_timer(task='region()'))
    dc,memories1 = region(fastqs=dc,flank5=flank5,flank3=flank3,out_dir=out_dir,**region_kwargs)
    memories.extend(memories1)

    memories.append(memory_timer(task='genotype()'))
    dc,memories1 = genotype(fastqs=dc,res=res,wt=wt,out_dir=out_dir,**genotype_kwargs)
    memories.extend(memories1)

    memories.append(memory_timer(task='outcomes()'))
    df_edits,memories1 = outcomes(fastqs=dc, **outcomes_kwargs)
    memories.extend(memories1)
    df_categories,memories1  = outcomes(fastqs=dc, col='category', **outcomes_kwargs)
    memories.extend(memories1)
    del dc # Remove dc to save memory

    # Save and return edit outcomes dataframe
    memories.append(memory_timer(task='genotyping()'))
    if out_dir is not None and out_file_prefix is not None: # Save dataframes (optional)
        io.save(dir=os.path.join(out_dir,'.genotyping'), # memory reporting
                file=f'{datetime.datetime.now().strftime("%Y%m%d_%H%M%S")}_memories.csv',
                obj=pd.DataFrame(memories, columns=['Task','Memory, MB','Time, s']))

        io.save(dir=out_dir,file=f'{out_file_prefix}_edit_outcomes.csv',obj=df_edits) # Edit outcomes
        io.save(dir=out_dir,file=f'{out_file_prefix}_category_outcomes.csv',obj=df_categories) # Edit categoy outcomes
    
    if return_dc: 
        return {'edit': df_edits,
                'category': df_categories}

def abundances(df: pd.DataFrame | str, desired_edits: list, sample_col: str='sample',
               edit_col: str='edit', count_col: str='count', fraction_col: str='fraction',
               combinations: int=1, return_memories: bool=False) -> pd.DataFrame | tuple[pd.DataFrame, list]:
    ''' 
    abundances(): quantify desired edits count & fraction per sample

    Parameters:
    df (DataFrame | str): dataframe with edit count & fraction per sample (tidy format)
    desired_edits (list): list of desired edits to isolate
    sample_col (str, optional): sample column name (Default: sample)
    edit_col (str, optional): edit column name (Default: edit)
    count_col (str, optional): count column name (Default: count)
    fraction_col (str, optional): fraction column name (Default: fraction)
    combinations (int, optional): maximum # of desired edit combinations to search for (Default: 1 => single edits)
    return_memories (bool, optional): return memories (Default: False)

    Dependencies: pandas
    '''
    # Get dataframe from file path if needed
    if type(df)==str:
        df = io.get(pt=df)

    # Memory reporting
    memories = []

    # Split data to increase performance
    dc = t.split(df=df, key=sample_col)
    
    # Search for multple edits
    if combinations > 1:

        desired_edits_combos = []

        for r in range(1, combinations + 1):
            for combo in combinations(desired_edits, r):

                sorted_combo = sorted(combo, key=lambda x: float(''.join(filter(str.isdigit, x)))) # Sort each combination numerically
                combo_str = ", ".join(sorted_combo) # Join as a string with ", "
                desired_edits_combos.append(combo_str)

        desired_edits = desired_edits_combos

    # Store desired edits abundances for each sample
    df_desired = pd.DataFrame(columns=df.columns) # Initialize empty DataFrame with same columns as df
    
    for key,df_sample in dc.items(): # Iterate through samples
        for desired_edit in desired_edits: # Iterate through desired edits
        
            i_desired = [] # Store desired edit & corresponding counts & fractions
            count_desired = []
            fraction_desired = []

            for i,(edit,count,fraction) in enumerate(t.zip_cols(df=df_sample,cols=[edit_col,count_col,fraction_col])):
                if edit == desired_edit: # Check if edit matches desired edit

                    i_desired.append(i)
                    count_desired.append(count)
                    fraction_desired.append(fraction)

            other_cols = [col for col in df_sample.columns if col not in [edit_col,count_col,fraction_col]]
            df_sample_desired = df_sample.iloc[0][other_cols].to_frame().T.reset_index(drop=True)
            df_sample_desired[edit_col] = [desired_edit]
            df_sample_desired[count_col] = [sum(count_desired)]
            df_sample_desired[fraction_col] = [sum(fraction_desired)]
            df_desired = pd.concat(objs=[df_desired,df_sample_desired]).reset_index(drop=True)

        memories.append(memory_timer(task=key))

    if return_memories: return df_desired,memories
    else: return df_desired

def editing_per_library(edit_dc: dict | str, paired_regions_dc: dict | str, fastq_ids: pd.DataFrame | str, 
                        out_dir: str=None, count: str='count', psuedocount: int=1, return_df: bool=True) -> pd.DataFrame:
    '''
    editing_per_library(): Determine editing relative library abundance

    Parameters:
    edit_dc (dict | str): Dictionary of edit outcomes dataframes or path to directory containing them.
    paired_regions_dc (dict | str): Dictionary of paired regions dataframes or path to directory containing them.
    fastq_ids (pd.DataFrame | str): DataFrame containing fastq IDs for 'genotyping' and 'paired_regions'.
    out_dir (str, optional): Directory to save the output dataframes (Default = None; no save).
    count (str, optional): Column name in paired regions dataframe to use for counting epeg-ngRNAs (Default = 'count').
    psuedocount (int, optional): Pseudocount to add to counts for calculations (Default = 1).
    return_df (bool, optional): Whether to return the final dataframe (Default = True).
    '''
    # Get edit and paired regions dictionaries; fastq_ids dataframe if needed
    if isinstance(edit_dc, str):
        edit_dc = io.get_dir(edit_dc)
    if isinstance(paired_regions_dc, str):
        paired_regions_dc = io.get_dir(paired_regions_dc)
    if isinstance(fastq_ids, str):
        fastq_ids = io.get(fastq_ids)

    # Check fastq_ids dataframe has 'genotyping' and 'paired_regions' columns
    if not all(col in fastq_ids.columns for col in ['genotyping', 'paired_regions']):
        raise ValueError("fastq_ids DataFrame must contain 'genotyping' and 'paired_regions' columns.")

    # Get editing per library abundance
    out_dc = dict()
    for edit_fq,edit_df in edit_dc.items(): # Iterate through edit outcomes dataframes
        paired_regions_fq = fastq_ids[fastq_ids['genotyping']==edit_fq]['paired_regions'].values[0]

        # Check if paired regions file column has 'edit' column
        if 'edit' not in paired_regions_dc[paired_regions_fq].columns:
            if 'Edit' in paired_regions_dc[paired_regions_fq].columns: # Rename 'Edit' column to 'edit' for consistency
                paired_regions_dc[paired_regions_fq].rename(columns={'Edit':'edit'}, inplace=True) 
            else:
                raise(TypeError(f"Error: 'edit' column not found in {paired_regions_fq}. Please check the file."))
            
        # Get counts of epeg-ngRNAs
        paired_regions_df = paired_regions_dc[paired_regions_fq][['edit','ID','desired']].value_counts().reset_index() 
        paired_regions_df = paired_regions_df[paired_regions_df['desired']==True] # Discard chimeras

        # Add missing edits
        for edit in edit_df['edit']:
            if edit not in paired_regions_df['edit'].values:
                paired_regions_df = pd.concat([paired_regions_df,
                                            pd.DataFrame({'edit':[edit], 'ID':[None], 'desired':[True], 'count':[0]})
                                            ]).reset_index(drop=True)
        
        # Calculate fraction of epeg-ngRNAs with psuedocount...
        paired_regions_df[f'{count}+{psuedocount}'] = paired_regions_df['count']+psuedocount
        total_count = paired_regions_df[count].sum()
        total_count_psuedocount = paired_regions_df[f'{count}+{psuedocount}'].sum()
        paired_regions_df['fraction'] = [count/total_count for count in paired_regions_df[count]]
        paired_regions_df[f'fraction+{psuedocount}'] = [count_psuedocount/total_count_psuedocount for count_psuedocount in paired_regions_df[f'{count}+{psuedocount}']]
        
        # ...to determine fraction of total epeg-ngRNAs for each edit and the editing per library
        library_count_psuedocount = []
        library_fraction_psuedocount = []
        for edit in edit_df['edit']:
            library_count_psuedocount.append(paired_regions_df[paired_regions_df['edit']==edit][f'count+{psuedocount}'].values[0])
            library_fraction_psuedocount.append(paired_regions_df[paired_regions_df['edit']==edit][f'fraction+{psuedocount}'].values[0])
        edit_df[f'library_count+{psuedocount}'] = library_count_psuedocount
        edit_df[f'library_fraction+{psuedocount}'] = library_fraction_psuedocount
        edit_df['editing_per_library'] = edit_df['fraction'] / edit_df[f'library_fraction+{psuedocount}']
        
        # Save the edited dataframe to the output dictionary
        out_dc[edit_fq] = edit_df
        if out_dir is not None:
            io.save(dir=os.path.join(out_dir,'split'), file=f"{edit_fq}.csv", obj=edit_df)

    # Save the output dictionary as a single dataframe
    out_df = t.join(dc=out_dc, col='sample')
    if out_dir is not None:
        io.save(dir=out_dir, file='editing_per_library.csv', obj=out_df)
    if return_df:
        return out_df

# UMI methods
def extract_umis(fastq_dir: str, out_dir: str='./extract_umis', 
                 bc_pattern: str='NNNNNNNNNNNNNNNN', env: str='umi_tools'):
    ''' 
    extract_umis(): extract UMIs using umi_tools

    Parameters:
    fastq_dir (str): directory with FASTQ files
    out_dir (str): output directory (Default: ./extract_umis)
    bc_pattern (str, optional): UMI barcode pattern (Default: NNNNNNNNNNNNNNNN)
    env (str, optional): conda environment with umi_tools installed (Default: umi_tools)
    '''
    # Memory reporting
    memory_timer(reset=True)
    memories = []

    # Create output directory and .extract_umi subdirectory for logs
    io.mkdir(os.path.join(out_dir,'.extract_umi'))

    # Iterate through fastq files in the directory
    for file in os.listdir(path=fastq_dir):
        if file.endswith('.fastq') or file.endswith('.fastq.gz'):
            # Extract UMIs using umi_tools
            command = f'conda run -n {env} umi_tools extract --bc-pattern={bc_pattern} --stdin={os.path.join(fastq_dir,file)} --stdout={os.path.join(out_dir,file.replace(".gz",""))} --log={os.path.join(out_dir,".extract_umi",file)}.log'
            print(f"{command}")
            result = subprocess.run(f"{command}", shell=True, cwd='.', capture_output=True, text=True)
            
            # Print output/errors
            if result.stdout: print(f"output:\n{result.stdout}")
            if result.stderr: print(f"errors:\n{result.stderr}")

            # Memory reporting
            memories.append(memory_timer(task=f'umi_tools extract: {file}'))

    # Memory reporting
    memories.append(memory_timer(task='extract_umis()'))
    io.save(dir=os.path.join(out_dir,'.extract_umis'),
            file=f'{datetime.datetime.now().strftime("%Y%m%d_%H%M%S")}_memories.csv',
            obj=pd.DataFrame(memories, columns=['Task','Memory, MB','Time, s']))     
    
def trim_motifs(fastq_dir: str, out_dir: str='./trim_motifs', 
                in_file: pd.DataFrame | str = None, motif5: str=None, motif3: str=None, 
                motif_length: int=21, error_rate: float=0.1,
                max_distance: int=2, env: str='umi_tools'):
    ''' 
    trim_motifs(): trimming motifs with cutadapt

    Parameters:
    fastq_dir (str): directory with FASTQ files (with UMIs extracted)
    out_dir (str): output directory (Default: ./trim_motifs)

    in_file (dataframe | str, option 1): Input file (.txt or .csv) with sequences for PrimeDesign. Format: target_name,target_sequence (column names required)
    motif5 (str, option 2): 5' motif pattern (Default: None)
    motif3 (str, option 2): 3' motif pattern (Default: None)

    motif_length (int, optional): trim 'in_file' motifs to this length (Default: 21)
    error_rate (float, optional): maximum error rate allowed in each motif (Default: 0.1)
    max_distance (int, optional): maximum number of errors allowed in each motif (Default: 2)
    cutadapt (str, optional): Conda environment with cutadapt installed (Default: umi_tools)
    '''
    # Memory reporting
    memory_timer(reset=True)
    memories = []

    # Create output directory and .trim_motifs subdirectory for logs
    io.mkdir(os.path.join(out_dir,'.trim_motifs'))
    
    # Get motifs from in_file if needed
    if motif5 is None and motif3 is None:
        if in_file is None:
            raise(ValueError("Error: Either 'in_file' or 'motif5' & 'motif3' must be provided."))
        
        if type(in_file)==str: # Get PrimeDesign input DataFrame from file path if needed
            in_file = io.get(pt=in_file)
        if 'target_sequence' not in in_file.columns:
            raise(ValueError(f"Error: 'target_sequence' column not found in {in_file}. Please check the file."))
        target_sequence = in_file.iloc[0]['target_sequence']
        motif5 = target_sequence.split('(')[0][-motif_length:]
        motif3 = target_sequence.split(')')[1][:motif_length]
        
    # Iterate through fastq files in the directory
    for file in os.listdir(path=fastq_dir):
        if file.endswith('.fastq') or file.endswith('.fastq.gz'):
            # Trim motifs using cutadapt (only keep reads with both motifs (up to 2 errors or 10% error rate))
            command = f'conda run -n {env} cutadapt -g {motif5} -a {motif3} -e {error_rate} --max-ee {max_distance} --trimmed-only -o {os.path.join(out_dir,file.replace(".gz",""))} {os.path.join(fastq_dir,file)} > {os.path.join(out_dir,".trim_motifs",file)}.log'
            print(f"{command}")
            result = subprocess.run(f"{command}", shell=True, cwd='.', capture_output=True, text=True)
            
            # Print output/errors
            if result.stdout: print(f"output:\n{result.stdout}")
            if result.stderr: print(f"errors:\n{result.stderr}")

            # Memory reporting
            memories.append(memory_timer(task=f'cutadapt: {file}'))

    # Memory reporting
    memories.append(memory_timer(task='trim_motifs()'))
    io.save(dir=os.path.join(out_dir,'.trim_motifs'),
            file=f'{datetime.datetime.now().strftime("%Y%m%d_%H%M%S")}_memories.csv',
            obj=pd.DataFrame(memories, columns=['Task','Memory, MB','Time, s']))  

def make_sams(fastq_dir: str, out_dir: str='./make_sams', 
              in_file: pd.DataFrame | str = None, fasta: str=None,
              sensitivity: Literal['very-fast','fast','sensitive','very-sensitive','very-fast-local','fast-local','sensitive-local','very-sensitive-local']='very-sensitive', 
              env: str='umi_tools'):
    ''' 
    make_sams(): generates alignments saved as a SAM files using bowtie2

    Parameters:
    fastq_dir (str): directory with FASTQ files (with UMIs extracted and motifs trimmed)
    out_dir (str): output directory (Default: ./make_sams)

    in_file (dataframe | str, option 1): Input file (.txt or .csv) with sequences for PrimeDesign. Format: target_name,target_sequence (column names required)
    fasta (str, option 2): reference fasta file (Default: None)
    
    sensitivity (str, optional): bowtie2 sensitivity preset (Default: very-sensitive). Options: very-fast, fast, sensitive, very-sensitive, very-fast-local, fast-local, sensitive-local, very-sensitive-local.
        For --end-to-end:
        --very-fast            -D 5 -R 1 -N 0 -L 22 -i S,0,2.50
        --fast                 -D 10 -R 2 -N 0 -L 22 -i S,0,2.50
        --sensitive            -D 15 -R 2 -N 0 -L 22 -i S,1,1.15 (default)
        --very-sensitive       -D 20 -R 3 -N 0 -L 20 -i S,1,0.50

        For --local:
        --very-fast-local      -D 5 -R 1 -N 0 -L 25 -i S,1,2.00
        --fast-local           -D 10 -R 2 -N 0 -L 22 -i S,1,1.75
        --sensitive-local      -D 15 -R 2 -N 0 -L 20 -i S,1,0.75 (default)
        --very-sensitive-local -D 20 -R 3 -N 0 -L 20 -i S,1,0.50
    env (str, optional): Conda environment with bowtie2 installed (Default: umi_tools)
    '''
    # Memory reporting
    memory_timer(reset=True)
    memories = []

    # Create output directory and .make_sams subdirectory for logs
    io.mkdir(os.path.join(out_dir,'.make_sams'))

    # Get fasta from in_file if needed
    if fasta is None:
        if in_file is None:
            raise(ValueError("Error: Either 'in_file' or 'fasta' must be provided."))
        
        if type(in_file)==str: # Get PrimeDesign input DataFrame from file path if needed
            in_file = io.get(pt=in_file)
        if 'target_sequence' not in in_file.columns:
            raise(ValueError(f"Error: 'target_sequence' column not found in {in_file}. Please check the file."))
        elif 'target_name' not in in_file.columns:
            raise(ValueError(f"Error: 'target_name' column not found in {in_file}. Please check the file."))
        
        # Extract target name and sequence
        target_name = in_file.iloc[0]['target_name']
        target_sequence = in_file.iloc[0]['target_sequence']
        seq = Seq(target_sequence.split('(')[1].split(')')[0])
        
        # Save fasta file
        fasta = os.path.join(out_dir,'.make_sams',f'{target_name}.fasta')
        SeqIO.write(SeqRecord(seq, id=target_name, description=''), fasta, 'fasta')
        
    # Create bowtie2 index
    command = f'conda run -n {env} bowtie2-build {fasta} {".".join(fasta.split(".")[:-1])}'
    print(f"{command}")
    result = subprocess.run(f"{command}", shell=True, cwd='.', capture_output=True, text=True)
    
    # Print output/errors
    if result.stdout: print(f"output:\n{result.stdout}")
    if result.stderr: print(f"errors:\n{result.stderr}")

    # Memory reporting
    memories.append(memory_timer(task=f'bowtie2-build: {fasta}'))

    # Iterate through fastq files in the directory
    for file in os.listdir(path=fastq_dir):
        if file.endswith('.fastq') or file.endswith('.fastq.gz'):
            # Generate SAM files using bowtie2
            command = f'conda run -n {env} bowtie2 -x {".".join(fasta.split(".")[:-1])} -U {os.path.join(fastq_dir,file)} -S {os.path.join(out_dir,file)}.sam --{sensitivity} > {os.path.join(out_dir,".make_sams",file)}.log'
            print(f"{command}")
            result = subprocess.run(f"{command}", shell=True, cwd='.', capture_output=True, text=True)
            
            # Print output/errors
            if result.stdout: print(f"output:\n{result.stdout}")
            if result.stderr: print(f"errors:\n{result.stderr}")

            # Memory reporting
            memories.append(memory_timer(task=f'bowtie2: {file}'))

    # Memory reporting
    memories.append(memory_timer(task='make_sams()'))

    io.save(dir=os.path.join(out_dir,'.make_sams'),
            file=f'{datetime.datetime.now().strftime("%Y%m%d_%H%M%S")}_memories.csv',
            obj=pd.DataFrame(memories, columns=['Task','Memory, MB','Time, s']))

def make_bams(sam_dir: str, out_dir: str='./make_bams', env: str='umi_tools'):
    '''
    make_bams(): converts SAM files to BAM files using samtools
    
    Parameters:
    sam_dir (str): directory with SAM files
    out_dir (str): output directory (Default: ./make_bams)
    env (str, optional): Conda environment with samtools installed (Default: umi_tools)
    '''
    # Memory reporting
    memory_timer(reset=True)
    memories = []

    # Create output directory and .make_bams subdirectory for logs
    io.mkdir(os.path.join(out_dir,'.make_bams'))

    # Iterate through SAM files in the directory
    for file in os.listdir(path=sam_dir):
        if file.endswith('.sam'):
            # Convert SAM to sorted BAM using samtools
            command = f"conda run -n {env} bash -lc \
                        'set -euo pipefail; \
                        samtools view -b {os.path.join(sam_dir,file)} | samtools sort -o {os.path.join(out_dir,file.replace('.sam','.sorted.bam'))}' \
                        > {os.path.join(out_dir,'.make_bams',file)}.log 2>&1"
            print(f"{command}")
            result = subprocess.run(f"{command}", shell=True, cwd='.', capture_output=True, text=True)
            
            # Print output/errors
            if result.stdout: print(f"output:\n{result.stdout}")
            if result.stderr: print(f"errors:\n{result.stderr}")

            # Memory reporting
            memories.append(memory_timer(task=f'samtools view | samtools sort: {file}'))

            # Make BAM index using samtools
            command = f'conda run -n {env} samtools index {os.path.join(out_dir,file.replace(".sam",".sorted.bam"))} > {os.path.join(out_dir,".make_bams",file)}.index.log'
            print(f"{command}")
            result = subprocess.run(f"{command}", shell=True, cwd='.', capture_output=True, text=True)

            # Print output/errors
            if result.stdout: print(f"output:\n{result.stdout}")
            if result.stderr: print(f"errors:\n{result.stderr}")

            # Memory reporting
            memories.append(memory_timer(task=f'samtools index: {file}'))

    # Memory reporting
    memories.append(memory_timer(task='make_bams()'))
    io.save(dir=os.path.join(out_dir,'.make_bams'),
            file=f'{datetime.datetime.now().strftime("%Y%m%d_%H%M%S")}_memories.csv',
            obj=pd.DataFrame(memories, columns=['Task','Memory, MB','Time, s']))

def bam_umi_tags(bam_dir: str, out_dir: str='./bam_umi_tags',
                 env: str='umi_tools'):
    '''
    bam_umi_tags(): copy UMI in read ID to RX tag in BAM files using fgbio
    
    Parameters:
    bam_dir (str): directory with BAM files
    out_dir (str): output directory (Default: ./bam_umi_tags)
    env (str, optional): Conda environment with umi_tools installed (Default: umi_tools)
    '''
    # Memory reporting
    memory_timer(reset=True)
    memories = []

    # Create output directory and .bam_umi_tags subdirectory for logs
    io.mkdir(os.path.join(out_dir,'.bam_umi_tags'))

    # Iterate through BAM files in the directory
    for file in os.listdir(path=bam_dir):
        if file.endswith('.bam'):
            # Group BAM files by UMI using fgbio
            command = f'conda run -n {env} fgbio CopyUmiFromReadName -i {os.path.join(bam_dir,file)} -o {os.path.join(out_dir,file.replace(".bam",".withRX.bam"))} --field-delimiter _'
            print(f"{command}")
            result = subprocess.run(f"{command}", shell=True, cwd='.', capture_output=True, text=True)
            
            # Print output/errors
            if result.stdout: print(f"output:\n{result.stdout}")
            if result.stderr: print(f"errors:\n{result.stderr}")

            # Memory reporting
            memories.append(memory_timer(task=f'fgbio GroupReadsByUmi: {file}'))
    
    # Memory reporting
    memories.append(memory_timer(task='bam_umi_tags()'))
    io.save(dir=os.path.join(out_dir,'.bam_umi_tags'),
            file=f'{datetime.datetime.now().strftime("%Y%m%d_%H%M%S")}_memories.csv',
            obj=pd.DataFrame(memories, columns=['Task','Memory, MB','Time, s']))

def group_umis(bam_dir: str, out_dir: str='./group_umis', 
               strategy: Literal['Identical','Edit','Adjacency','Paired']='Adjacency',
               edits: int=1,
               env: str='umi_tools'):
    '''
    group_umis(): group BAM files by UMI using fgbio
    
    Parameters:
    bam_dir (str): directory with BAM files
    out_dir (str): output directory (Default: ./group_umis)
    strategy (str, optional): UMI grouping strategy (Default: adjacency). Options: Identical, Edit, Adjacency, Paired
        1. Identity: only reads with identical UMI sequences are grouped together. This strategy may be useful for evaluating
           data, but should generally be avoided as it will generate multiple UMI groups per original molecule in the presence
           of errors.
        2. Edit: reads are clustered into groups such that each read within a group has at least one other read in the group
           with <= edits differences and there are inter-group pairings with <= edits differences. Effective when there are
           small numbers of reads per UMI, but breaks down at very high coverage of UMIs.
        3. Adjacency: a version of the directed adjacency method described in umi_tools (http://dx.doi.org/10.1101/051755)
           that allows for errors between UMIs but only when there is a count gradient.
        4. Paired: similar to adjacency but for methods that produce template such that a read with A-B is related to but not
           identical to a read with B-A. Expects the UMI sequences to be stored in a single SAM tag separated by a hyphen (e.g.
           'ACGT-CCGG') and allows for one of the two UMIs to be absent (e.g. 'ACGT-' or '-ACGT'). The molecular IDs produced
           have more structure than for single UMI strategies and are of the form '{base}/{A|B}'. E.g. two UMI pairs would be
           mapped as follows AAAA-GGGG -> 1/A, GGGG-AAAA -> 1/B.
    env (str, optional): Conda environment with fgbio installed (Default: umi_tools)
    '''
    # Memory reporting
    memory_timer(reset=True)
    memories = []

    # Create output directory and .group_umis, family_hist, & grouping_metrics subdirectories for logs
    io.mkdir(os.path.join(out_dir,'.group_umis'))
    io.mkdir(os.path.join(out_dir,'family_hist'))
    io.mkdir(os.path.join(out_dir,'grouping_metrics'))

    # Iterate through BAM files in the directory
    for file in os.listdir(path=bam_dir):
        if file.endswith('.bam'):
            # Group BAM files by UMI using fgbio
            command = f'conda run -n {env} fgbio GroupReadsByUmi -i {os.path.join(bam_dir,file)} -o {os.path.join(out_dir,file.replace(".bam",".grouped.bam"))} -s {strategy} -e {edits} -f {os.path.join(out_dir,"family_hist",file.replace(".bam",".family_hist.txt"))} -g {os.path.join(out_dir,"grouping_metrics",file.replace(".bam",".grouping_metrics.txt"))}'
            print(f"{command}")
            result = subprocess.run(f"{command}", shell=True, cwd='.', capture_output=True, text=True)
            
            # Print output/errors
            if result.stdout: print(f"output:\n{result.stdout}")
            if result.stderr: print(f"errors:\n{result.stderr}")

            # Memory reporting
            memories.append(memory_timer(task=f'fgbio GroupReadsByUmi: {file}'))
    
    # Memory reporting
    memories.append(memory_timer(task='group_umis()'))
    io.save(dir=os.path.join(out_dir,'.group_umis'),
            file=f'{datetime.datetime.now().strftime("%Y%m%d_%H%M%S")}_memories.csv',
            obj=pd.DataFrame(memories, columns=['Task','Memory, MB','Time, s']))

def consensus_umis(bam_dir: str, out_dir: str='./consensus_umis', 
                   min_reads: int=1,
                   env: str='umi_tools'):
    '''
    consensus_umis(): generate consensus reads from grouped BAM files using fgbio
    
    Parameters:
    bam_dir (str): directory with grouped BAM files
    out_dir (str): output directory (Default: ./consensus_umis)
    min_reads (int, optional): minimum reads per UMI group to generate consensus (Default: 1)
    env (str, optional): Conda environment with fgbio installed (Default: umi_tools)
    '''
    # Memory reporting
    memory_timer(reset=True)
    memories = []

    # Create output directory and .consensus_umis subdirectory for logs
    io.mkdir(os.path.join(out_dir,'.consensus_umis'))

    # Iterate through grouped BAM files in the directory
    for file in os.listdir(path=bam_dir):
        if file.endswith('.grouped.bam'):
            # Generate consensus reads from grouped BAM files using fgbio
            command = f'conda run -n {env} fgbio CallMolecularConsensusReads -i {os.path.join(bam_dir,file)} -o {os.path.join(out_dir,file.replace(".grouped.bam",".consensus.bam"))} -M {min_reads}'
            print(f"{command}")
            result = subprocess.run(f"{command}", shell=True, cwd='.', capture_output=True, text=True)
            
            # Print output/errors
            if result.stdout: print(f"output:\n{result.stdout}")
            if result.stderr: print(f"errors:\n{result.stderr}")

            # Memory reporting
            memories.append(memory_timer(task=f'fgbio CallMolecularConsensusReads: {file}'))

    # Memory reporting
    memories.append(memory_timer(task='consensus_umis()'))
    io.save(dir=os.path.join(out_dir,'.consensus_umis'),
            file=f'{datetime.datetime.now().strftime("%Y%m%d_%H%M%S")}_memories.csv',
            obj=pd.DataFrame(memories, columns=['Task','Memory, MB','Time, s']))

def bam_to_fastq(bam_dir: str, out_dir: str='./bam_to_fastq', env: str='umi_tools'):
    '''
    bam_to_fastq(): convert BAM files to FASTQ files using samtools

    Parameters:
    bam_dir (str): directory with BAM files
    out_dir (str): output directory (Default: ./bam_to_fastq)
    env (str, optional): Conda environment with samtools installed (Default: umi_tools)
    '''
    # Memory reporting
    memory_timer(reset=True)
    memories = []

    # Create output directory and .bam_to_fastq subdirectory for logs
    io.mkdir(os.path.join(out_dir,'.bam_to_fastq'))

    # Iterate through BAM files in the directory
    for file in os.listdir(path=bam_dir):
        if file.endswith('.bam'):
            # Convert BAM to FASTQ using samtools
            command = f'conda run -n {env} samtools fastq -n {os.path.join(bam_dir,file)} > {os.path.join(out_dir,file.replace(".bam",".fastq"))}'
            print(f"{command}")
            result = subprocess.run(f"{command}", shell=True, cwd='.', capture_output=True, text=True)
            
            # Print output/errors
            if result.stdout: print(f"output:\n{result.stdout}")
            if result.stderr: print(f"errors:\n{result.stderr}")

            # Memory reporting
            memories.append(memory_timer(task=f'samtools fastq: {file}'))
    
    # Memory reporting
    memories.append(memory_timer(task='bam_to_fastq()'))
    io.save(dir=os.path.join(out_dir,'.bam_to_fastq'),
            file=f'{datetime.datetime.now().strftime("%Y%m%d_%H%M%S")}_memories.csv',
            obj=pd.DataFrame(memories, columns=['Task','Memory, MB','Time, s']))

# Supporting methods for DMS plots
''' aa_props: dictionary of AA properties with citations (Generated by ChatGPT)
    
    Sources:
    Hydrophobicity: https://resources.qiagenbioinformatics.com/manuals/clcgenomicsworkbench/650/Hydrophobicity_scales.html
    Weight: from Bio.Data import IUPACData (protein_weights)
    Polarity: https://web.expasy.org/protscale/pscale/PolarityGrantham.html
'''
aa_props = {
    'E': {'name': 'Glutamic acid', 'hydrophobicity': -3.5, 'weight': 147.1, 'polarity': 12.3, 'charge': 'negative'},
    'D': {'name': 'Aspartic acid', 'hydrophobicity': -3.5, 'weight': 133.1, 'polarity': 13.0, 'charge': 'negative'},
    'R': {'name': 'Arginine', 'hydrophobicity': -4.5, 'weight': 174.2, 'polarity': 10.5, 'charge': 'positive'},
    'H': {'name': 'Histidine', 'hydrophobicity': -3.2, 'weight': 155.2, 'polarity': 10.4, 'charge': 'positive'},
    'K': {'name': 'Lysine', 'hydrophobicity': -3.9, 'weight': 146.2, 'polarity': 11.3, 'charge': 'positive'},
    'F': {'name': 'Phenylalanine', 'hydrophobicity': 2.8, 'weight': 165.2, 'polarity': 5.2, 'charge': 'neutral'},
    'Y': {'name': 'Tyrosine', 'hydrophobicity': -1.3, 'weight': 181.2, 'polarity': 6.2, 'charge': 'neutral'},
    'W': {'name': 'Tryptophan', 'hydrophobicity': -0.9, 'weight': 204.2, 'polarity': 5.4, 'charge': 'neutral'},
    'S': {'name': 'Serine', 'hydrophobicity': -0.8, 'weight': 105.1, 'polarity': 9.2, 'charge': 'neutral'},
    'Q': {'name': 'Glutamine', 'hydrophobicity': -3.5, 'weight': 146.2, 'polarity': 10.5, 'charge': 'neutral'},
    'T': {'name': 'Threonine', 'hydrophobicity': -0.7, 'weight': 119.1, 'polarity': 8.6, 'charge': 'neutral'},
    'N': {'name': 'Asparagine', 'hydrophobicity': -3.5, 'weight': 132.1, 'polarity': 11.6, 'charge': 'neutral'},
    'C': {'name': 'Cysteine', 'hydrophobicity': 2.5, 'weight': 121.2, 'polarity': 5.5, 'charge': 'neutral'},
    'P': {'name': 'Proline', 'hydrophobicity': -1.6, 'weight': 115.1, 'polarity': 8.0, 'charge': 'neutral'},
    'A': {'name': 'Alanine', 'hydrophobicity': 1.8, 'weight': 89.1, 'polarity': 8.1, 'charge': 'neutral'},
    'G': {'name': 'Glycine', 'hydrophobicity': -0.4, 'weight': 75.1, 'polarity': 9.0, 'charge': 'neutral'},
    'M': {'name': 'Methionine', 'hydrophobicity': 1.9, 'weight': 149.2, 'polarity': 5.7, 'charge': 'neutral'},
    'V': {'name': 'Valine', 'hydrophobicity': 4.2, 'weight': 117.1, 'polarity': 5.9, 'charge': 'neutral'},
    'I': {'name': 'Isoleucine', 'hydrophobicity': 4.5, 'weight': 131.2, 'polarity': 5.2, 'charge': 'neutral'},
    'L': {'name': 'Leucine', 'hydrophobicity': 3.8, 'weight': 131.2, 'polarity': 4.9, 'charge': 'neutral'},
    '*': {'name': 'Stop', 'hydrophobicity': None, 'weight': None, 'polarity': None, 'charge': None}
}

def edit_1(df: pd.DataFrame, col: str='edit') -> pd.DataFrame:
    ''' 
    edit_1(): split edit column to before, after, and amino acid number
    
    Parameters:
    df (dataframe): fastq outcomes dataframe
    col (str, optional): edit column name
    
    Dependencies: pandas
    '''
    df_1 = df[(df[col].str.contains(',')==False)&(df[col]!='WT')&(df[col]!='Indel')] # Isolate single AA changes
    df_1['before']=df_1[col].str[0] # Split edit information
    df_1['after']=df_1[col].str[-1]
    df_1['number']=df_1[col].str[1:-1].astype(int)
    return df_1.reset_index(drop=True)

def dms_cond(df: pd.DataFrame, cond: str, wt:str, res: int, 
             sample: str='sample', edit: str='edit', psuedocount: int=0) -> pd.DataFrame:
    ''' 
    dms_cond(): returns DMS grid data in tidy format grouped by condition
    
    Parameters:
    df (dataframe): fastq outcomes dataframe
    cond (str): Condition column name for grouping fastq outcomes dataframe
    wt (str): Expected wildtype nucleotide sequence (in frame AA)
    res (int): First AA number
    sample (str, optional): Sample column name for splicing fastq outcomes dataframe (Default: 'sample')
    edit (str, optional): Edit column name within fastq outcomes dataframe (Default: 'edit')
    psuedocount (int, optional): psuedocount to avoid log(0) & /0 (Default: 0)
    
    Dependencies: Bio.Seq.Seq, pandas, numpy, tidy, edit_1(), & aa_props
    '''
    wt_prot = Seq(wt).translate(table=1) # Obtain WT protein sequence
    wt_nums = np.arange(res,res+len(wt_prot))
    print('Isolate single aa change fastq outcomes')
    dc=t.split(edit_1(df),sample) # Isolate single aa change fastq outcomes and split by sample
    
    print('Fill with DMS grid data for each sample:')
    dc2=dict() # Fill with DMS grid data in tidy format split by sample
    for key_sample,df_sample in dc.items():
        print(key_sample)
        wt_fastq = df[(df['edit']=='WT')&(df[sample]==key_sample)] # Obtain WT fastq outcome
        df_sample_DMS=pd.DataFrame(columns=wt_fastq.columns) # Fill with DMS grid data in tidy format
        
        for num in wt_nums: # Iterate through WT protein sequence
            vals=dict() # Create dictionary with all amino acid changes for a given residue
            
            # Add metadata that is the same for all genotypes
            meta = [x for x in df_sample.columns if x not in [edit,'count','fraction','before','after','number']]
            for m in meta: 
                vals[m]=[wt_fastq[m].to_list()[0]]*len(list(aa_props.keys()))
            
            # Create all amino acid changes
            vals['before']=[wt_prot[num-res]]*len(list(aa_props.keys()))
            vals['number']=[num]*len(list(aa_props.keys()))
            vals['after']=list(aa_props.keys())
            vals[edit]=[vals['before'][i]+str(num)+vals['after'][i] for i in range(len(vals['after']))]

            # Fill in counts (+ psuedocount) for amino acid changes, WT, and none
            counts=[]
            num_mut = df_sample[df_sample['number']==num]
            for a in vals['after']:
                if a == wt_prot[num-res]: counts.append(wt_fastq['count'].to_list()[0]+psuedocount) # Wild type
                elif a in num_mut['after'].to_list(): counts.append(num_mut[num_mut['after']==a]['count'].to_list()[0]+psuedocount) # Amino acid change present
                else: counts.append(psuedocount) # Amino acid change absent
            vals['count']=counts
            sum_counts = sum(vals['count'])
            vals['fraction']=[count/sum_counts for count in vals['count']]

            df_sample_DMS = pd.concat([df_sample_DMS,pd.DataFrame(vals)]).reset_index(drop=True) # Append residue DMS data
        
        df_sample_DMS['number']=df_sample_DMS['number'].astype(int) # Set number as type int
        df_sample_DMS['count']=df_sample_DMS['count'].astype(int) # Set count as type int for plotting

        df_sample_DMS[sample] = [key_sample]*df_sample_DMS.shape[0]
        dc2[key_sample]=df_sample_DMS # Append sample DMS data
    del dc # Remove dc to save memory

    print('Group samples by condition:')
    dc3=t.split(t.join(dc2,sample),cond) # Join samples back into 1 dataframe & split by condition
    del dc2 # Remove dc2 to save memory
    df_cond_stat = pd.DataFrame()
    for key_cond,df_cond in dc3.items(): # Iterate through conditions
        print(key_cond)
        edit_ls = []
        fraction_avg_ls = []
        fraction_ls = []
        count_avg_ls = []
        before_ls = []
        after_ls = []
        number_ls = []
        for e in df_cond[edit]: # iterate through edits
            df_cond_edit = df_cond[df_cond[edit]==e]
            edit_ls.append(e)
            fraction_avg_ls.append(sum(df_cond_edit['fraction'])/len(df_cond_edit['fraction']))
            fraction_ls.append(df_cond_edit['fraction'].tolist())
            count_avg_ls.append(sum(df_cond_edit['count'])/len(df_cond_edit['count']))
            before_ls.append(df_cond_edit.iloc[0]['before'])
            after_ls.append(df_cond_edit.iloc[0]['after'])
            number_ls.append(df_cond_edit.iloc[0]['number'])
        df_cond_stat = pd.concat([df_cond_stat,
                                  pd.DataFrame({'edit':edit_ls,
                                                'before':before_ls,
                                                'after':after_ls,
                                                'number':number_ls,
                                                'fraction_ls':fraction_ls,
                                                'fraction_avg':fraction_avg_ls,
                                                'count_avg':count_avg_ls,
                                                cond:[key_cond]*len(number_ls)})])
    return df_cond_stat.drop_duplicates(subset=['edit','Description']).reset_index(drop=True)

def dms_comp(df: pd.DataFrame, cond: str, cond_comp: str, wt: str, res: int, 
             sample: str='sample', edit: str='edit', psuedocount: int=1) -> pd.DataFrame:
    ''' 
    dms_comp(): returns comparison DMS grid dataframe in tidy format split by condition
    
    Parameters:
    df (dataframe): fastq outcomes dataframe
    cond (str): Condition column name for grouping fastq outcomes dataframe
    cond_comp (str): Condition for comparison group
    wt (str): Expected wildtype nucleotide sequence (in frame AA)
    res (int): First AA number
    sample (str, optional): Sample column name for splicing fastq outcomes dataframe (Default: 'sample')
    edit (str, optional): Edit column name within fastq outcomes dataframe (Default: 'edit')
    psuedocount (int, optional): psuedocount to avoid log(0) & /0 (Default: 1)
    
    Dependencies: Bio.Seq.Seq, pandas, numpy, tidy, edit_1(), dms_cond(), & aa_props
    '''
    df_cond_stat = dms_cond(df,cond,wt,res,sample,edit,psuedocount) # Execute dms_cond()

    # Fold change & p-value relative comparison group
    print(f'Compute FC & pval relative to {cond_comp}:')
    df_stat = pd.DataFrame()
    df_comp = df_cond_stat[df_cond_stat[cond]==cond_comp] # Isolate comparison group
    df_other = df_cond_stat[df_cond_stat[cond]!=cond_comp] # From other groups
    for e in set(df_other[edit].tolist()): # iterate through edits
        print(f'{e}')
        df_other_edit = df_other[df_other[edit]==e]
        df_comp_edit = df_comp[df_comp[edit]==e]
        df_other_edit['fraction_avg_compare'] = [df_comp_edit.iloc[0]['fraction_avg']]*df_other_edit.shape[0]
        df_other_edit['count_avg_compare'] = [df_comp_edit.iloc[0]['count_avg']]*df_other_edit.shape[0]
        df_other_edit['FC'] = df_other_edit['fraction_avg']/df_comp_edit.iloc[0]['fraction_avg']
        ttests = [ttest_ind(other_fraction_ls,df_comp_edit.iloc[0]['fraction_ls']) 
                                 for other_fraction_ls in df_other_edit['fraction_ls']]
        df_other_edit['pval'] = [ttest[1] for ttest in ttests]
        df_other_edit['tstat'] = [ttest[0] for ttest in ttests]
        df_stat = pd.concat([df_stat,df_other_edit])
    df_stat['compare'] = [cond_comp]*df_stat.shape[0]
    return df_stat[[edit,'before','after','number','FC','pval','tstat','fraction_avg','fraction_avg_compare','count_avg','count_avg_compare',cond,'compare']].sort_values(by=['number','after']).reset_index(drop=True)

def subscript(df: pd.DataFrame, tick: str='before',tick_sub: str='number') -> pd.DataFrame:
    ''' 
    subscript(): returns dataframe with subscripts to tick labels
    
    Parameters:
    df (dataframe): dataframe
    tick (str, optional): new tick label column name
    tick_sub (str, optional): previous numeric tick label that will become a subscript

    Dependencies: pandas
    '''
    ticks = []
    labels = []
    for (t,ts) in set(zip(df[tick],df[tick_sub])):
        ticks.append(ts)
        labels.append('$\\mathrm{'+t+'_{'+str(ts)+'}}$')
    return pd.DataFrame({'tick':ticks,'label':labels}).sort_values(by='tick').reset_index(drop=True)

# Plot methods
def scat(typ: str,df: pd.DataFrame,x: str,y: str, cols:bool=None, cols_ord:bool=None, stys:bool=None, cutoff: float=0.01, cols_exclude: list|str=None,
         file: bool=None, dir: bool=None, palette_or_cmap: str='colorblind', edgecol: str='black',
         figsize: tuple=(10,6), title: str='', title_size=18, title_weight: str='bold', title_font: str='Arial',
         x_axis: str='', x_axis_size: int=12, x_axis_weight: str='bold', x_axis_font: str='Arial', x_axis_scale: str='linear', x_axis_dims: tuple=(0,100),x_ticks_rot: int=0, x_ticks_font: str='Arial',x_ticks: list=[],
         y_axis: str='', y_axis_size: int=12, y_axis_weight: str='bold', y_axis_font: str='Arial', y_axis_scale: str='linear', y_axis_dims: tuple=(0,100),y_ticks_rot: int=0, y_ticks_font: str='Arial',y_ticks: list=[],
         legend_title: str='', legend_title_size: int=12, legend_size: int=9, legend_bbox_to_anchor: tuple=(1,1), legend_loc: str='upper left', legend_items: tuple=(0,0), show: bool=True, space_capitalized: bool=True,
         **kwargs):
    '''
    scat(): creates scatter plot related graphs

    Parameters:
    typ (str): plot type (scat, line, line_scat)
    df (dataframe): pandas dataframe
    x (str): x-axis column name
    y (str): y-axis column name
    cols (str, optional): color column name
    cols_ord (list, optional): color column values order
    stys (str, optional): styles column name
    cols_exclude (list | str, optional): color column values exclude
    file (str, optional): save plot to filename
    dir (str, optional): save plot to directory
    palette_or_cmap (str, optional): seaborn color palette or matplotlib color map
    edgecol (str, optional): point edge color
    figsize (tuple, optional): figure size
    title (str, optional): plot title
    title_size (int, optional): plot title font size
    title_weight (str, optional): plot title bold, italics, etc.
    title_font (str, optional): plot title font
    x_axis (str, optional): x-axis name
    x_axis_size (int, optional): x-axis name font size
    x_axis_weight (str, optional): x-axis name bold, italics, etc.
    x_axis_font (str, optional): x-axis font
    x_axis_scale (str, optional): x-axis scale linear, log, etc.
    x_axis_dims (tuple, optional): x-axis dimensions (start, end)
    x_ticks_rot (int, optional): x-axis ticks rotation
    x_ticks_font (str, optional): x-ticks font
    x_ticks (list, optional): x-axis tick values
    y_axis (str, optional): y-axis name
    y_axis_size (int, optional): y-axis name font size
    y_axis_weight (str, optional): y-axis name bold, italics, etc.
    y_axis_font (str, optional): y-axis font
    y_axis_scale (str, optional): y-axis scale linear, log, etc.
    y_axis_dims (tuple, optional): y-axis dimensions (start, end)
    y_ticks_rot (int, optional): y-axis ticks rotation
    y_ticks_font (str, optional): y_ticks font
    y_ticks (list, optional): y-axis tick values
    legend_title (str, optional): legend title
    legend_title_size (str, optional): legend title font size
    legend_size (str, optional): legend font size
    legend_bbox_to_anchor (tuple, optional): coordinates for bbox anchor
    legend_loc (str): legend location
    legend_ncol (tuple, optional): # of columns
    show (bool, optional): show plot (Default: True)
    space_capitalize (bool, optional): use re_un_cap() method when applicable (Default: True)
    
    Dependencies: os, matplotlib, seaborn, & plot
    '''
    # Omit data smaller than cutoff or excluded
    df_cut=df[df[y]>=cutoff]
    df_other=df[df[y]<cutoff]
    for sample in list(df_other['sample'].value_counts().keys()):
        df_temp = df_other[df_other['sample']==sample]
        df_temp['fraction']=sum(df_temp['fraction'])
        df_temp['edit']='Other'
        df_cut = pd.concat([df_cut,df_temp.iloc[0].to_frame().T])

    # Omit excluded
    if type(cols_exclude)==list: 
        for exclude in cols_exclude: df_cut=df_cut[df_cut[cols]!=exclude]
    elif type(cols_exclude)==str: df_cut=df_cut[df_cut[cols]!=cols_exclude]

    # Sort data by genotype position
    if cols_ord==None:
        genotypes = list(df_cut[cols].value_counts().keys())
        positions = list()
        for geno in genotypes:
            numbers = re.findall(r'\d+\.?\d*', geno)
            if len(numbers)==0: positions.append(100000) # Places WT and Indel at the end
            else: positions.append(sum([int(n) for n in numbers])/len(numbers))
        assign = pd.DataFrame({'positions':positions,
                               'genotypes':genotypes})
        cols_ord = list(assign.sort_values(by='positions')['genotypes'])

    p.scat(typ=typ,df=df_cut,x=x,y=y,cols=cols,cols_ord=cols_ord,cols_exclude=cols_exclude, stys=stys,
           file=file,dir=dir,palette_or_cmap=palette_or_cmap,edgecol=edgecol,
           figsize=figsize,title=title,title_size=title_size,title_weight=title_weight,title_font=title_font,
           x_axis=x_axis,x_axis_size=x_axis_size,x_axis_weight=x_axis_weight,x_axis_font=x_axis_font,x_axis_scale=x_axis_scale,x_axis_dims=x_axis_dims,x_ticks_rot=x_ticks_rot,x_ticks_font=x_ticks_font,x_ticks=x_ticks,
           y_axis=y_axis,y_axis_size=y_axis_size,y_axis_weight=y_axis_weight,y_axis_font=y_axis_font,y_axis_scale=y_axis_scale,y_axis_dims=y_axis_dims,y_ticks_rot=y_ticks_rot,y_ticks_font=y_ticks_font,y_ticks=y_ticks,
           legend_title=legend_title,legend_title_size=legend_title_size,legend_size=legend_size,legend_bbox_to_anchor=legend_bbox_to_anchor,legend_loc=legend_loc,legend_items=legend_items,show=show,space_capitalize=space_capitalized,
           **kwargs)

def cat(typ: str, df: pd.DataFrame, x: str, y: str, errorbar: str=None, cols: str=None, cols_ord: list=None, cutoff: float=0.01, cols_exclude: list|str=None,
        file: str=None, dir: str=None, palette_or_cmap: str='colorblind', edgecol: str='black', lw: int=1,
        figsize: tuple=(10,6), title: str='', title_size: int=18, title_weight: str='bold', title_font: str='Arial',
        x_axis: str='', x_axis_size=12, x_axis_weight: str='bold', x_axis_font: str='Arial', x_axis_scale: str='linear', x_axis_dims: tuple=(0,1), x_ticks_rot: int=0, x_ticks_font: str='Arial', x_ticks: list=[],
        y_axis: str='', y_axis_size=12, y_axis_weight: str='bold', y_axis_font: str='Arial', y_axis_scale: str='linear', y_axis_dims: tuple=(0,1), y_ticks_rot: int=0, y_ticks_font: str='Arial', y_ticks: list=[],
        legend_title: str='', legend_title_size: int=12, legend_size: int=9, legend_bbox_to_anchor=(1,1), legend_loc: str='upper left', legend_items: tuple=(0,0), show: bool=True, space_capitalize: bool=True,
        **kwargs):
    ''' 
    cat: creates category dependent graphs
    
    Parameters:
    typ (str): plot type (bar, box, violin, swarm, strip, point, count, bar_swarm, box_swarm, violin_swarm)
    df (dataframe): pandas dataframe
    x (str, optional): x-axis column name
    y (str, optional): y-axis column name
    cols (str, optional): color column name
    cols_ord (list, optional): color column values order
    cols_exclude (list | str, optional): color column values exclude
    file (str, optional): save plot to filename
    dir (str, optional): save plot to directory
    palette_or_cmap (str, optional): seaborn color palette or matplotlib color map
    edgecol (str, optional): point edge color
    lw (int, optional): line width
    errorbar (str, optional): error bar type (sd)
    errwid (int, optional): error bar line width
    errcap (int, optional): error bar cap line width
    figsize (tuple, optional): figure size
    title (str, optional): plot title
    title_size (int, optional): plot title font size
    title_weight (str, optional): plot title bold, italics, etc.
    title_font (str, optional): plot title font
    x_axis (str, optional): x-axis name
    x_axis_size (int, optional): x-axis name font size
    x_axis_weight (str, optional): x-axis name bold, italics, etc.
    x_axis_font (str, optional): x-axis font
    x_axis_scale (str, optional): x-axis scale linear, log, etc.
    x_axis_dims (tuple, optional): x-axis dimensions (start, end)
    x_ticks_rot (int, optional): x-axis ticks rotation
    x_ticks_font (str, optional): x-ticks font
    x_ticks (list, optional): x-axis tick values
    y_axis (str, optional): y-axis name
    y_axis_size (int, optional): y-axis name font size
    y_axis_weight (str, optional): y-axis name bold, italics, etc.
    y_axis_font (str, optional): y-axis font
    y_axis_scale (str, optional): y-axis scale linear, log, etc.
    y_axis_dims (tuple, optional): y-axis dimensions (start, end)
    y_ticks_rot (int, optional): y-axis ticks rotation
    y_ticks_font (str, optional): y_ticks font
    y_ticks (list, optional): y-axis tick values
    legend_title (str, optional): legend title
    legend_title_size (str, optional): legend title font size
    legend_size (str, optional): legend font size
    legend_bbox_to_anchor (tuple, optional): coordinates for bbox anchor
    legend_loc (str): legend location
    legend_ncol (tuple, optional): # of columns
    show (bool, optional): show plot (Default: True)
    space_capitalize (bool, optional): use re_un_cap() method when applicable (Default: True)
    
    Dependencies: os, matplotlib, seaborn, & plot
    '''
    # Omit data smaller than cutoff or excluded
    df_cut=df[df[y]>=cutoff]
    df_other=df[df[y]<cutoff]
    for sample in list(df_other['sample'].value_counts().keys()):
        df_temp = df_other[df_other['sample']==sample]
        df_temp['fraction']=sum(df_temp['fraction'])
        df_temp['edit']='Other'
        df_cut = pd.concat([df_cut,df_temp.iloc[0].to_frame().T])
    
    # Omit excluded
    if type(cols_exclude)==list: 
        for exclude in cols_exclude: df_cut=df_cut[df_cut[cols]!=exclude]
    elif type(cols_exclude)==str: df_cut=df_cut[df_cut[cols]!=cols_exclude]

    # Sort data by genotype position
    if cols_ord==None:
        genotypes = list(df_cut[cols].value_counts().keys())
        positions = list()
        for geno in genotypes:
            numbers = re.findall(r'\d+\.?\d*', geno)
            if len(numbers)==0: positions.append(100000) # Places WT and Indel at the end
            else: positions.append(sum([int(n) for n in numbers])/len(numbers))
        assign = pd.DataFrame({'positions':positions,
                               'genotypes':genotypes})
        cols_ord = list(assign.sort_values(by='positions')['genotypes'])

    p.cat(typ=typ,df=df_cut,x=x,y=y,errorbar=errorbar,cols=cols,cols_ord=cols_ord,cols_exclude=cols_exclude,
          file=file,dir=dir,palette_or_cmap=palette_or_cmap,edgecol=edgecol,lw=lw,
          figsize=figsize,title=title,title_size=title_size,title_weight=title_weight,title_font=title_font,
          x_axis=x_axis,x_axis_size=x_axis_size,x_axis_weight=x_axis_weight,x_axis_font=x_axis_font,x_axis_scale=x_axis_scale,x_axis_dims=x_axis_dims,x_ticks_rot=x_ticks_rot,x_ticks_font=x_ticks_font,x_ticks=x_ticks,
          y_axis=y_axis,y_axis_size=y_axis_size,y_axis_weight=y_axis_weight,y_axis_font=y_axis_font,y_axis_scale=y_axis_scale,y_axis_dims=y_axis_dims,y_ticks_rot=y_ticks_rot,y_ticks_font=y_ticks_font,y_ticks=y_ticks,
          legend_title=legend_title,legend_title_size=legend_title_size,legend_size=legend_size,legend_bbox_to_anchor=legend_bbox_to_anchor,legend_loc=legend_loc,legend_items=legend_items,show=show,space_capitalize=space_capitalize,
          **kwargs)

def stack(df: pd.DataFrame, x: str='sample', y: str='fraction', cols: str='edit', cutoff: float=0.01, cols_ord: list=[], x_ord: list=[],
          file: str=None, dir: str=None, palette_or_cmap: str='tab20', repeats: int=1, errcap: int=4, vertical: bool=True,
          figsize=(10,6), title: str='Editing Outcomes', title_size: int=18, title_weight: str='bold', title_font: str='Arial',
          x_axis: str='', x_axis_size: int=12, x_axis_weight: str='bold', x_axis_font: str='Arial', x_ticks_rot: int=0, x_ticks_font: str='Arial',
          y_axis: str='', y_axis_size: int=12, y_axis_weight: str='bold', y_axis_font: str='Arial', y_axis_dims: tuple=(0,0), y_ticks_rot: int=0, y_ticks_font: str='Arial',
          legend_title: str='', legend_title_size: int=12, legend_size: int=12,
          legend_bbox_to_anchor: tuple=(1,1), legend_loc: str='upper left', legend_ncol: int=1, show: bool=True, space_capitalize: bool=True, **kwargs):
    ''' 
    stack(): creates stacked bar plot

    Parameters:
    df (dataframe): pandas dataframe
    x (str, optional): x-axis column name
    y (str, optional): y-axis column name
    cols (str, optional): color column name
    cutoff (float, optional): y-axis values needs be greater than (e.g. 0.01)
    cols_ord (list, optional): color column values order
    file (str, optional): save plot to filename
    dir (str, optional): save plot to directory
    palette_or_cmap (str, optional): seaborn palette or matplotlib color map
    repeats (int, optional): number of color palette or map repeats (Default: 1)
    errcap (int, optional): error bar cap line width
    vertical (bool, optional): vertical orientation; otherwise horizontal (Default: True)
    figsize (tuple, optional): figure size
    title (str, optional): plot title
    title_size (int, optional): plot title font size
    title_weight (str, optional): plot title bold, italics, etc.
    title_font (str, optional): plot title font
    x_axis (str, optional): x-axis name
    x_axis_size (int, optional): x-axis name font size
    x_axis_weight (str, optional): x-axis name bold, italics, etc.
    x_axis_font (str, optional): x-axis font
    x_ticks_rot (int, optional): x-axis ticks rotation
    x_ticks_font (str, optional): x-ticks font
    y_axis (str, optional): y-axis name
    y_axis_size (int, optional): y-axis name font size
    y_axis_weight (str, optional): y-axis name bold, italics, etc.
    y_axis_font (str, optional): y-axis font
    y_axis_dims (tuple, optional): y-axis dimensions (start, end)
    y_ticks_rot (int, optional): y-axis ticks rotation
    y_ticks_font (str, optional): y-ticks font
    legend_title (str, optional): legend title
    legend_title_size (str, optional): legend title font size
    legend_size (str, optional): legend font size
    legend_bbox_to_anchor (tuple, optional): coordinates for bbox anchor
    legend_loc (str): legend location
    legend_ncol (tuple, optional): # of columns
    show (bool, optional): show plot (Default: True)
    space_capitalize (bool, optional): use re_un_cap() method when applicable (Default: True)
    
    Dependencies: re, os, pandas, numpy, matplotlib.pyplot & plot
    '''
    # Omit smaller than cutoff and convert it to <cutoff
    df_cut=df[df[y]>=cutoff]
    df_other=df[df[y]<cutoff]
    for x_val in list(df_other[x].value_counts().keys()):
        df_temp = df_other[df_other[x]==x_val]
        df_temp[y]=sum(df_temp[y])
        df_temp[cols]=f'<{cutoff}'
        df_cut = pd.concat([df_cut,df_temp.iloc[:1]])

    # Sort pivot table columns by genotype position
    if cols_ord==[]:
        genotypes = list(df_cut[cols].value_counts().keys())
        positions = list()
        for geno in genotypes:
            numbers = re.findall(r'\d+\.?\d*', geno)
            if geno==f'<{cutoff}':positions.append(100000) # Places <cutoff near the end
            elif len(numbers)==0: positions.append(100001) # Places WT at the end
            else: positions.append(sum([int(n) for n in numbers])/len(numbers))
        assign = pd.DataFrame({'positions':positions,
                               'genotypes':genotypes})
        cols_ord = list(assign.sort_values(by='positions')['genotypes'])
    
    # Make stacked barplot
    p.stack(df=df_cut,x=x,y=y,cols=cols,cutoff=0,cols_ord=cols_ord,x_ord=x_ord,
            file=file,dir=dir,palette_or_cmap=palette_or_cmap,repeats=repeats,errcap=errcap,vertical=vertical,
            figsize=figsize,title=title,title_size=title_size,title_weight=title_weight,title_font=title_font,
            x_axis=x_axis,x_axis_size=x_axis_size,x_axis_weight=x_axis_weight,x_axis_font=x_axis_font,x_ticks_rot=x_ticks_rot,x_ticks_font=x_ticks_font,
            y_axis=y_axis,y_axis_size=y_axis_size,y_axis_weight=y_axis_weight,y_axis_font=y_axis_font,y_axis_dims=y_axis_dims,y_ticks_rot=y_ticks_rot,y_ticks_font=y_ticks_font,
            legend_title=legend_title,legend_title_size=legend_title_size,legend_size=legend_size,
            legend_bbox_to_anchor=legend_bbox_to_anchor,legend_loc=legend_loc,legend_ncol=legend_ncol,show=show,space_capitalize=space_capitalize,**kwargs)

def heat(df: pd.DataFrame, cond: str, x: str='number', y: str='after', vals: str='fraction_avg', vals_dims:tuple=None,
         file=None, dir=None, edgecol: str='black', lw: int=1, annot=False, cmap: str="bone_r", sq: bool=True, cbar: bool=True,
         title: str='', title_size: int=12, title_weight: str='bold', title_font: str='Arial', figsize: tuple=(20,7),
         x_axis: str='', x_axis_size: int=12, x_axis_weight: str='bold', x_axis_font: str='Arial', x_ticks_rot: int=45, x_ticks_font: str='Arial',
         y_axis: str='', y_axis_size: int=12, y_axis_weight: str='bold', y_axis_font: str='Arial', y_ticks_rot: int=0, y_ticks_font: str='Arial',
         show: bool=True, space_capitalize: bool=True, **kwargs):
    ''' 
    heat(): creates heatmap
    
    Parameters:
    df (dataframe): tidy-formatted DMS dataframe (dms_cond() or dms_comp())
    x (str, optional): x-axis column name (AA residues number column)
    y (str, optional): y-axis column name (AA change column)
    vals (str, optional): values column name
    vals_dims (tuple, optional): vals minimum and maximum formatted (vmin, vmax; Default: None)
    file (str, optional): save plot to filename
    dir (str, optional): save plot to directory
    edgecol (str, optional): point edge color
    lw (int, optional): line width
    annot (bool, optional): annotate values
    cmap (str, optional): matplotlib color map
    title (str, optional): plot title
    title_size (int, optional): plot title font size
    title_weight (str, optional): plot title bold, italics, etc.
    title_font (str, optional): plot title font
    figsize (tuple, optional): figure size per subplot
    x_axis (str, optional): x-axis name
    x_axis_size (int, optional): x-axis name font size
    x_axis_weight (str, optional): x-axis name bold, italics, etc.
    x_axis_font (str, optional): x-axis font
    x_ticks_rot (int, optional): x-axis ticks rotation
    x_ticks_font (str, optional): x-ticks font
    y_axis (str, optional): y-axis name
    y_axis_size (int, optional): y-axis name font size
    y_axis_weight (str, optional): y-axis name bold, italics, etc.
    y_axis_font (str, optional): y-axis font
    y_ticks_rot (int, optional): y-axis ticks rotation
    y_ticks_font (str, optional): y-ticks font
    show (bool, optional): show plot (Default: True)
    space_capitalize (bool, optional): use re_un_cap() method when applicable (Default: True)
    
    Dependencies: matplotlib, seaborn, pandas, & aa_props
    '''
    # Find min and max values in the dataset for normalization
    if vals_dims is None:
        vmin = df[vals].values.min()
        vmax = df[vals].values.max()
    else:
        vmin = vals_dims[0]
        vmax = vals_dims[1]

    # Make DMS grids
    print('Make DMS grids')
    dc=t.split(df,cond) # Split by condition
    dc2={key:pd.pivot(df_cond,columns=x,index=y,values=vals).astype(float).reindex(list(aa_props.keys())) 
         for key,df_cond in dc.items()} # Generate pivot tables
    
    # Create a single figure with multiple heatmap subplots
    print('Create a single figure with multiple heatmap subplots')
    fig, axes = plt.subplots(nrows=len(list(dc2.keys())),ncols=1,figsize=(figsize[0],figsize[1]*len(list(dc2.keys()))),sharex=False,sharey=True)
    if isinstance(axes, np.ndarray)==False: axes = np.array([axes]) # Make axes iterable if there is only 1 heatmap
    for (ax, key) in zip(axes, list(dc2.keys())):
        print(f'{key}')
        sns.heatmap(dc2[key],annot=annot,cmap=cmap,ax=ax,linecolor=edgecol,linewidths=lw,cbar=cbar,square=sq,vmin=vmin,vmax=vmax, **kwargs)
        if len(list(dc2.keys()))>1: ax.set_title(key,fontsize=title_size,fontweight=title_weight,fontfamily=title_font)  # Add title to subplot
        else: ax.set_title(title,fontsize=title_size,fontweight=title_weight,fontfamily=title_font)
        if x_axis=='': 
            if space_capitalize: ax.set_xlabel(p.re_un_cap(x),fontsize=x_axis_size,fontweight=x_axis_weight,fontfamily=x_axis_font) # Add x axis label
            else: ax.set_xlabel(x,fontsize=x_axis_size,fontweight=x_axis_weight,fontfamily=x_axis_font) # Add x axis label
        else: ax.set_xlabel(x_axis,fontsize=x_axis_size,fontweight=x_axis_weight,fontfamily=x_axis_font)
        if y_axis=='': 
            if space_capitalize: ax.set_ylabel(p.re_un_cap(y),fontsize=y_axis_size,fontweight=y_axis_weight,fontfamily=y_axis_font) # Add y axis label
            else: ax.set_ylabel(y,fontsize=y_axis_size,fontweight=y_axis_weight,fontfamily=y_axis_font) # Add y axis label
        else: ax.set_ylabel(y_axis,fontsize=y_axis_size,fontweight=y_axis_weight,fontfamily=y_axis_font)
        ax.set_xticklabels(subscript(dc[key])['label'].to_list()) # Change x ticks to have subscript format
        # Format x ticks
        if (x_ticks_rot==0)|(x_ticks_rot==90): plt.setp(ax.get_xticklabels(), rotation=x_ticks_rot, ha="center",rotation_mode="anchor",fontname=x_ticks_font) 
        else: plt.setp(ax.get_xticklabels(), rotation=x_ticks_rot, ha="right",rotation_mode="anchor",fontname=x_ticks_font) 
        # Format y ticks
        plt.setp(ax.get_yticklabels(), rotation=y_ticks_rot, va='center', ha="right",rotation_mode="anchor",fontname=y_ticks_font)
        ax.set_facecolor('white')  # Set background to transparent

    # Save & show fig
    if file is not None and dir is not None:
        io.mkdir(dir) # Make output directory if it does not exist
        plt.savefig(fname=os.path.join(dir, file), dpi=600, bbox_inches='tight', format=f'{file.split(".")[-1]}')
    if show: plt.show()

def vol(df: pd.DataFrame, x: str, y: str, size: str=None, size_dims: tuple=None, include_wt: bool=False,
        FC_threshold: float=2, pval_threshold: float=0.05, file: str=None, dir: str=None, color: str='lightgray', alpha: float=0.5, edgecol: str='black', vertical: bool=True,
        figsize=(10,6), title: str='', title_size: int=18, title_weight: str='bold', title_font: str='Arial',
        x_axis: str='', x_axis_size: int=12, x_axis_weight: str='bold', x_axis_font: str='Arial', x_axis_dims: tuple=(0,0), x_ticks_rot: int=0, x_ticks_font: str='Arial', x_ticks: list=[],
        y_axis: str='', y_axis_size: int=12, y_axis_weight: str='bold', y_axis_font: str='Arial', y_axis_dims: tuple=(0,0), y_ticks_rot: int=0, y_ticks_font: str='Arial', y_ticks: list=[],
        legend_title: str='',legend_title_size: int=12, legend_size: int=9, legend_bbox_to_anchor: tuple=(1,1), legend_loc: str='upper left',
        legend_items: tuple=(0,0), legend_ncol: int=1, display_size: bool=True, display_labels: bool=True, return_df: bool=True, show: bool=True, space_capitalize: bool=True,
        **kwargs) -> pd.DataFrame:
    ''' 
    vol(): creates volcano plot
    
    Parameters:
    df (dataframe): pandas dataframe
    x (str): x-axis column name (FC)
    y (str): y-axis column name (pval)
    cols (str, optional): color column name
    size (str, optional): size column name
    size_dims (tuple, optional): (minimum,maximum) values in size column (Default: None)
    include_wt (bool, optional): include wildtype (Default: False)
    FC_threshold (float, optional): fold change threshold (Default: 2; log2(2)=1)
    pval_threshold (float, optional): p-value threshold (Default: 0.05; -log10(0.05)=1.3)
    file (str, optional): save plot to filename
    dir (str, optional): save plot to directory
    color (str, optional): matplotlib color for nonsignificant values
    alpha (float, optional): transparency for nonsignificant values (Default: 0.5)
    edgecol (str, optional): point edge color
    vertical (bool, optional): vertical orientation; otherwise horizontal (Default: True)
    figsize (tuple, optional): figure size
    title (str, optional): plot title
    title_size (int, optional): plot title font size
    title_weight (str, optional): plot title bold, italics, etc.
    title_font (str, optional): plot title font
    x_axis (str, optional): x-axis name
    x_axis_size (int, optional): x-axis name font size
    x_axis_weight (str, optional): x-axis name bold, italics, etc.
    x_axis_font (str, optional): x-axis font
    x_axis_dims (tuple, optional): x-axis dimensions (start, end)
    x_ticks_rot (int, optional): x-axis ticks rotation
    x_axis_font (str, optional): x-axis font
    x_ticks (list, optional): x-axis tick values
    y_axis (str, optional): y-axis name
    y_axis_size (int, optional): y-axis name font size
    y_axis_weight (str, optional): y-axis name bold, italics, etc.
    y_axis_font (str, optional): y-axis font
    y_axis_dims (tuple, optional): y-axis dimensions (start, end)
    y_ticks_rot (int, optional): y-axis ticks rotation
    y_ticks_font (str, optional): y_ticks font
    y_ticks (list, optional): y-axis tick values
    legend_title (str, optional): legend title
    legend_title_size (str, optional): legend title font size
    legend_size (str, optional): legend font size
    legend_bbox_to_anchor (tuple, optional): coordinates for bbox anchor
    legend_loc (str): legend location
    legend_ncol (tuple, optional): # of columns
    display_size (bool, optional): display size on plot (Default: True)
    display_labels (bool, optional): display labels for significant values (Default: True)
    return_df (bool, optional): return dataframe (Default: True)
    show (bool, optional): show plot (Default: True)
    space_capitalize (bool, optional): use re_un_cap() method when applicable (Default: True)
    
    Dependencies: os, matplotlib, seaborn, pandas, & edit_1()
    '''
    # Strings with subscripts
    log2 = 'log\u2082'
    log10 = 'log\u2081\u2080'
    
    # Log transform data
    df[f'{log2}({x})'] = [np.log10(xval)/np.log10(2) for xval in df[x]]
    df[f'-{log10}({y})'] = [-np.log10(yval) for yval in df[y]]
    
    # Organize data by significance
    signif = []
    for (log2FC,log10P) in zip(df[f'{log2}({x})'],df[f'-{log10}({y})']):
        if (np.abs(log2FC)>1)&(log10P>-np.log10(0.05)): signif.append('FC & p-value')
        elif (np.abs(log2FC)<=1)&(log10P>-np.log10(0.05)): signif.append('p-value')
        elif (np.abs(log2FC)>1)&(log10P<=-np.log10(0.05)): signif.append('FC')
        else: signif.append('NS')
    df['Significance']=signif
    #signif_order = ['NS','FC','p-value','FC & p-value']

    # Organize data by conservation (changed from)
    basic = ['R','K', 'H']
    acidic = ['D','E']
    polar = ['S', 'T', 'N', 'Q', 'Y', 'C']
    nonpolar = ['A','V','L','I','M','F','W','P','G']
    change = []

    df = edit_1(df)
    for (before,after) in zip(df['before'],df['after']):
        if (before in basic)&(after not in basic): change.append('Basic')
        elif (before in acidic)&(after not in acidic): change.append('Acidic')
        elif (before in polar)&(after not in polar): change.append('Polar')
        elif (before in nonpolar)&(after not in nonpolar): change.append('Nonpolar')
        else: change.append('Conserved')
    df['Change'] = change

    sty_order = ['Conserved','Basic','Acidic','Polar','Nonpolar']
    mark_order = ['D','^','v','<','>']

    # Remove wildtype
    if include_wt==False:
        wt_i = [i for i,(before,after) in enumerate(t.zip_cols(df=df,cols=['before','after'])) if before == after]
        df = df.drop(wt_i,axis=0).reset_index(drop=True)

    # Organize data by abundance
    sizes=(1,100)
    if size_dims is not None: df = df[(df[size]>=size_dims[0])&(df[size]<=size_dims[1])]

    # Set dimensions
    if x_axis_dims==(0,0): x_axis_dims=(min(df[f'{log2}({x})']),max(df[f'{log2}({x})']))
    if y_axis_dims==(0,0): y_axis_dims=(0,max(df[f'-{log10}({y})']))

    # Generate figure
    fig, ax = plt.subplots(figsize=figsize)
    
    if vertical: # orientation
        # with significance boundraries
        plt.vlines(x=-np.log10(FC_threshold)/np.log10(2), ymin=y_axis_dims[0], ymax=y_axis_dims[1], colors='k', linestyles='dashed', linewidth=1)
        plt.vlines(x=np.log10(FC_threshold)/np.log10(2), ymin=y_axis_dims[0], ymax=y_axis_dims[1], colors='k', linestyles='dashed', linewidth=1)
        plt.hlines(y=-np.log10(pval_threshold), xmin=x_axis_dims[0], xmax=x_axis_dims[1], colors='k', linestyles='dashed', linewidth=1)
    
        # with data
        if display_size==False: size=None
        sns.scatterplot(data=df[df['Significance']!='FC & p-value'], x=f'{log2}({x})', y=f'-{log10}({y})', 
                        edgecolor=edgecol, color=color, alpha=alpha, style='Change',
                        style_order=sty_order, markers=mark_order, 
                        size=size, sizes=sizes,
                        ax=ax, **kwargs)
        sns.scatterplot(data=df[(df['Significance']=='FC & p-value')&(df[f'{log2}({x})']<0)], 
                        x=f'{log2}({x})', y=f'-{log10}({y})', 
                        hue=f'{log2}({x})',
                        edgecolor=edgecol, palette='Blues_r', style='Change',
                        style_order=sty_order, markers=mark_order,
                        size=size, sizes=sizes, legend=False,
                        ax=ax, **kwargs)
        sns.scatterplot(data=df[(df['Significance']=='FC & p-value')&(df[f'{log2}({x})']>0)], 
                        x=f'{log2}({x})', y=f'-{log10}({y})', 
                        hue=f'{log2}({x})',
                        edgecolor=edgecol, palette='Reds', style='Change',
                        style_order=sty_order, markers=mark_order,
                        size=size, sizes=sizes, legend=False,
                        ax=ax, **kwargs)
        
        # with labels
        if display_labels:
            df_signif = df[df['Significance']=='FC & p-value']
            adjust_text([plt.text(x=df_signif.iloc[i][f'{log2}({x})'], 
                                  y=df_signif.iloc[i][f'-{log10}({y})'],
                                  s=edit) for i,edit in enumerate(df_signif['edit'])])
        
        # Set x axis
        if x_axis=='': x_axis=f'{log2}({x})'
        plt.xlabel(x_axis, fontsize=x_axis_size, fontweight=x_axis_weight,fontfamily=x_axis_font)
        if x_ticks==[]: 
            if (x_ticks_rot==0)|(x_ticks_rot==90): plt.xticks(rotation=x_ticks_rot,ha='center',fontfamily=x_ticks_font)
            else: plt.xticks(rotation=x_ticks_rot,ha='right',fontfamily=x_ticks_font)
        else: 
            if (x_ticks_rot==0)|(x_ticks_rot==90): plt.xticks(ticks=x_ticks,labels=x_ticks,rotation=x_ticks_rot, ha='center',fontfamily=x_ticks_font)
            else: plt.xticks(ticks=x_ticks,labels=x_ticks,rotation=x_ticks_rot,ha='right',fontfamily=x_ticks_font)

        # Set y axis
        if y_axis=='': y_axis=f'-{log10}({y})'
        plt.ylabel(y_axis, fontsize=y_axis_size, fontweight=y_axis_weight,fontfamily=y_axis_font)

        if y_ticks==[]: plt.yticks(rotation=y_ticks_rot,fontfamily=y_ticks_font)
        else: plt.yticks(ticks=y_ticks,labels=y_ticks,rotation=y_ticks_rot,fontfamily=y_ticks_font)

    else: # Horizontal orientation
        # with significance boundraries
        plt.hlines(y=-np.log10(FC_threshold)/np.log10(2), xmin=y_axis_dims[0], xmax=y_axis_dims[1], colors='k', linestyles='dashed', linewidth=1)
        plt.hlines(y=np.log10(FC_threshold)/np.log10(2), xmin=y_axis_dims[0], xmax=y_axis_dims[1], colors='k', linestyles='dashed', linewidth=1)
        plt.vlines(x=-np.log10(pval_threshold), ymin=x_axis_dims[0], ymax=x_axis_dims[1], colors='k', linestyles='dashed', linewidth=1)

        # with data
        if display_size==False: size=None
        sns.scatterplot(data=df[df['Significance']!='FC & p-value'], y=f'{log2}({x})', x=f'-{log10}({y})', 
                        edgecolor=edgecol, color=color, alpha=alpha, style='Change',
                        style_order=sty_order, markers=mark_order, 
                        size=size, sizes=sizes,
                        ax=ax, **kwargs)
        sns.scatterplot(data=df[(df['Significance']=='FC & p-value')&(df[f'{log2}({x})']<0)], 
                        y=f'{log2}({x})', x=f'-{log10}({y})', 
                        hue=f'{log2}({x})',
                        edgecolor=edgecol, palette='Blues_r', style='Change',
                        style_order=sty_order, markers=mark_order,
                        size=size, sizes=sizes, legend=False,
                        ax=ax, **kwargs)
        sns.scatterplot(data=df[(df['Significance']=='FC & p-value')&(df[f'{log2}({x})']>0)], 
                        y=f'{log2}({x})', x=f'-{log10}({y})', 
                        hue=f'{log2}({x})',
                        edgecolor=edgecol, palette='Reds', style='Change',
                        style_order=sty_order, markers=mark_order,
                        size=size, sizes=sizes, legend=False,
                        ax=ax, **kwargs)
        
        # with labels
        if display_labels:
            df_signif = df[df['Significance']=='FC & p-value']
            adjust_text([plt.text(y=df_signif.iloc[i][f'{log2}({x})'], 
                                  x=df_signif.iloc[i][f'-{log10}({y})'],
                                  s=edit) for i,edit in enumerate(df_signif['edit'])])
        
        # Set x axis
        if y_axis=='': y_axis=f'-{log10}({y})'
        plt.xlabel(y_axis, fontsize=y_axis_size, fontweight=y_axis_weight,fontfamily=y_axis_font)
        if y_ticks==[]: 
            if (y_ticks_rot==0)|(y_ticks_rot==90): plt.xticks(rotation=y_ticks_rot,ha='center',fontfamily=y_ticks_font)
            else: plt.xticks(rotation=y_ticks_rot,ha='right',fontfamily=y_ticks_font)
        else: 
            if (y_ticks_rot==0)|(y_ticks_rot==90): plt.xticks(ticks=y_ticks,labels=y_ticks,rotation=y_ticks_rot, ha='center',fontfamily=y_ticks_font)
            else: plt.xticks(ticks=y_ticks,labels=y_ticks,rotation=y_ticks_rot,ha='right',fontfamily=y_ticks_font)

        # Set y axis
        if x_axis=='': x_axis=f'{log2}({x})'
        plt.ylabel(x_axis, fontsize=x_axis_size, fontweight=x_axis_weight,fontfamily=x_axis_font)

        if x_ticks==[]: plt.yticks(rotation=x_ticks_rot,fontfamily=x_ticks_font)
        else: plt.yticks(ticks=x_ticks,labels=x_ticks,rotation=x_ticks_rot,fontfamily=x_ticks_font)

    # Set title
    if title=='' and file is not None: 
        if space_capitalize: title=p.re_un_cap(".".join(file.split(".")[:-1]))
        else: ".".join(file.split(".")[:-1])
    plt.title(title, fontsize=title_size, fontweight=title_weight, family=title_font)

    # Move legend to the right of the graph
    if legend_items==(0,0): ax.legend(title=legend_title,title_fontsize=legend_title_size,fontsize=legend_size,
                                        bbox_to_anchor=legend_bbox_to_anchor,loc=legend_loc,ncol=legend_ncol)
    else: 
        handles, labels = ax.get_legend_handles_labels()
        ax.legend(title=legend_title,title_fontsize=legend_title_size,fontsize=legend_size,
                  bbox_to_anchor=legend_bbox_to_anchor,loc=legend_loc,ncol=legend_ncol, # Move right of the graph
                  handles=handles[legend_items[0]:legend_items[1]],labels=labels[legend_items[0]:legend_items[1]]) # Only retains specified labels

    # Save & show fig; return dataframe
    if file is not None and dir is not None:
        io.mkdir(dir) # Make output directory if it does not exist
        plt.savefig(fname=os.path.join(dir, file), dpi=600, bbox_inches='tight', format=f'{file.split(".")[-1]}')
    if show: plt.show()
    if return_df: return df     