import edlib
import re
from Bio import pairwise2
from Bio.pairwise2 import format_alignment


def calculate_identity(seq1, seq2):
    # 进行全局比对 (globalxx 表示 match=1, mismatch=0, 不惩罚 gap)
    alignments = pairwise2.align.globalms(seq1, seq2, 3, -5, -2, -2)

    # 取第一个最佳比对
    best_alignment = alignments[0]
    aligned_seq1, aligned_seq2, score, begin, end = best_alignment

    # 计算 identity: 相同位置的个数 / 对齐长度（包含 gap）
    matches = sum(res1 == res2 for res1, res2 in zip(aligned_seq1, aligned_seq2))
    identity = matches / len(aligned_seq1)

    print(format_alignment(*best_alignment))
    print(f"Identity: {identity:.2%}")
    return identity


def compute_identity_from_eqx(eqx_str: str) -> float:
    # 匹配如 20=, 1X, 5=, 2D, 1I 等
    pattern = re.findall(r"(\d+)([=XID])", eqx_str)

    match_count = 0
    align_length = 0

    for count, op in pattern:
        count = int(count)
        if op == "=":
            match_count += count
            align_length += count
        elif op == "X":
            align_length += count
        elif op in ("I", "D"):
            align_length += (
                count  # Insertion/Deletion also count toward alignment length
            )

    if align_length == 0:
        return 0.0
    return match_count / align_length


def edlib_identity(seq1: str, seq2: str):
    aln_res = edlib.align(seq1, seq2, mode="NW", task="path")
    identity = compute_identity_from_eqx(aln_res["cigar"])
    print(f"identity:{identity}")
    print(f"aln_res:{aln_res}")


def main_cli():
    seq1 = "TTTTCCCCGGGGAAAGAGAGAGATATCGACCGCGGCGAGAGGCACCTGCTCTACCTGACGGTCATCAACTCCAAGGGCGAGATCCTCGAGCAGCGCAGCCTGAACACCATCCAGCAGTTCGACTACCAGAAGAAGCTGGACAACAGGGAGAAGGAGAGGGTGGCCGCCCGCAAGGCCTGGGGCGAGATCGAAACGATCAAGGAGCTCAAGTCCGGCTACCTCAGCCACGTCGTGCACCAGATCAACCAGCTCATGCTGAAGTACAACGCGATCGTGGTCCTCGAGGACCTGAACTTCGGCTTCAAGAGGGGCCGCTTCAAGGTGGAGAAGCAGATCTACCAGAACTTCGAGAACGCCCTGATCAAGAAGCTCAACCACCTGGTCCTCAAGGACAAGGCGGACGACGAGATCGGCAGCTACAAGAACGCGCTCCAGCTGACCAACAACTTCACGGACCTCAAGTCCATCGGCAAGCAGACGGGCTTCCTGTTCTACGTGCCGGCGTGGAACACCTCCAAGATCGACCCGGAAACCGGCTTTCGCGACCTGCTCAAGCCGCGCTACGAGAACATCGCGCAGTCCCAGGCGTTCTTCGGCAAGTTCGACAAGATCTGCTACAACACCGACAAGGGCTACTTCGAGTTCCACATCGACTACGCGAAGTTCACCGACAAGGCCAAGAACTCCAGGCAGAAGTGGGCCATCTGCAGCCACGGCGACAAGCGCTACGTGTACGACAAGACGGCGAACCAGAACAAGGGCGCGGCCAAGGCATCAACGTGAACGACGAGCTGAAGTCCCTCTTCGCGCGCTACCACATCAACGACAAGCAGCCGAACCTCGTCATGGACATCTGCCAGAACAACGACAAGGAGTTCCACAAGAGCCTGATGTGCCTGCTCAAGACCCTGCTCCAGATGAGGAACTCGAACGCGCGCGACCGGCGAGGACTACATCCTCAGCCCCGTCGCGAACGACGAGGGCGTGTTCTTCAACTCCGCCCTCGCGGACGACACGCAGCCGCAGAACGCCGACGCGAACGGCGCCTACCACATCGCCCTCAAGGGCCTGTGGCTGCTCAACGAGCTCAAGAACTCCGACGACCTGAACAAGGTGAAGCTCGCCATTGACAACCAGACGTGGCTGAATTTCGCTCAGAATAGGGGGGCGGGGGCTCAGGCGGGGGCGGGAGCGGCGGCGGGGGCTCTGGGGGCGGCGGCAGCGGCGGGGGCGGCAGCGGGGGCGGCGGGTCGGAAGAGCAGGAGTCTATCGAGAGGCTCGAGGCCATGGGCTTCGACAGGGCTATCGTGATCGAGGCCTTCCTGTCTTGCGATAGGAACGAGGAGCTGGCTGCCAACTACCTCCTGGAGCCGAAGAAGAAGCGCAAGGTGTCCGGCGGCAGCTCCGGCGGCAGCCCGAAGAAGAAGCGCAAAGTGTGAGCCAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACAAAGCACCAGTGGTCTAGTGGTAGAATAGTACCCTGCCACGGTACAGACCCGGGTTCGATTCCCGGCTGGTGCAAATTTCTACTGTTTGTAGATGTCACAGATCACAAACTTCAAATGAATTTCTACTGTTTGTAGATGGGAAAGACCGAGGAGAAGATCTAATTTCTACTGTTTGTAGATAACAAAGCACCAGTGGTCTAGTGGTAGAATAGTACCCTGCCACGGTACAGACCCGGGTTCGATTCCCGGCTGGTGCAGGGACCCTCCTGGGCCATGAAGCTGTCCTTCCAGGTTCACAAGTCATCTCTCTCTTTTCCCCGGGGAAAA"
    seq2 = "TTTTCCCGGGGGAAAAGAGAGAGATATCGACCGCGGCGGAGGCACCTGCTCTACCTGACCGATCATCCAATCTCCAAGGGCGAGATCCTCGAGCAGCGCAGGCCTAGAACATCCATCCAGCAGTTCGAACTACCAGAAGAAGCTGGACAACAGAGAGAAGGAGAGCGGTGGCGCCGCAACGCCTGGGGCGAGATCGAAACGATCAAGGGGCTCAAGTCCGGCTACCTCAGCACGTCGGTGCACCAGATCAACCAGCTCATCATGAACGTATCAAGCGATCGTGGTCCTCGAGGACCTGAACTTCGGCTTCAAGAGGGGCCGCTTCAAGGTGGAGAAGAGATCTACCAGAATTCGAGAACGCCCCTGACAGAACTCAACCACCTGAGTCCTCAAGGACAAGGCGGACGACGAGGATCGGCAGCTACAACAAGCGCTCACAGCTGACCAACAACTTCACGGACCTCACAGTCATCGGCAAGCAGACTGGGCTCACTGTTCTACTGTGCCGGCGTGGAACACCTCCAAGATCTGACCCGGAAACCGGCTTGTCGGACCTGCTCAAGCCGCTACGAGAATCGCGCGGCCAGTCCCAGGCGTTCTCGGCAATGGATTCGACAAGTCTTGCTACAACACCAGACAAGGTGCTACTTCCGGTTCAATCGACTACGCGAAGTTCACCGACATGCCCAAGACTCCAGGCAGAAGTGGGCCATCTGCAGCCACGGCGGACAAGCGCTACGTGTTACGACAAGATCGGCGATACCAGAACAAGGGCGCGGCCAAGGGCATCAACGTGAACGACGGGAGCTGAAGTCTCTTCGCGCGCTACCACATCAACGACAAGCAGCCGAACCTCTTGTCATGGACAATTCTGCCAGAACAAACGACAAGGAGTTCCACCAAGAGCCTGATGTGCCTGCTCTAAGACCCTGCCTCCAGATGAGTGAATCTGAACTGCCTGCGACCGGCGAGGACTACATCCCCAGCCCCGTCCCGAACGACGAGGGCGTGTTCTTCAACTCCGCCCTCGCGGGAACGACACGCAGCGACAGACGCCTGACGCGAACGCGCCTACCACATCGCCCTCAAGGGCCTGTGGCTGCTCAACGAGCTCCAAGAACTCCGACGACCTGAAACAAGGTGAAGCTGCCATTGACAACCAGACTGGCTGAATTTCAGCTCAGAATAGGGGGCGGGGGCTCAGGCGTGGGGCGGGAGCGGCGGCGGGGGCCTGGGGGCGGCGGCAGCGCGGGGCGGCAGCGGGGGCGCGGGCGGAAGAGCAGGAGTTATCGAGATGGCTCGAGGCCATGGGCTTCGACAGGGCTATCGTTGATCGGAGGGGCCTTTCCTGTCTTGGGATAGGAACGCAGGAGACTGGCTGCCAACTACCATCCGGGATACCGAAGAAGAGGCAGGTGTCCGGCGGCAGCTCCGGCGCAGCACGAAGAAGAAGCGCAAAGTGTGAGCCCCAAAAAAAAACAAAACAAAAAAAAAAAAAAAAAGAAAAAAAAAAAAAAAAAAACAAAGCACCAGTGGTCTAGTGGTAGAAATAGTACCCTGCCACCGGTACAACCGGGTTCGTCCGGCTGGTGCAAATTTCTACTGTTGTAGAATGTCACAGATCACAAACTTCAAATGAATTTCTACTGTTTGTAGATGGAAAGGACCGAGGAGAAGATCTAATTTCTACTGTTTGTAGATAACAAAGCACCGTGGTTAGTGGTAGAATAGTACACCTGCCACGTACAGACCCGGTTCGATTCCCGGCTGGTGCAGGACCCTCCTGGGCCATGAAGCTGTCCTCCAGGGTTCACAAGTCATCCTCTCTTTTCCACCGGGGAAAA"
    edlib_identity(seq1=seq1, seq2=seq2)
    calculate_identity(seq1=seq1, seq2=seq2)


if __name__ == "__main__":
    main_cli()
