"""
SpecPulse Core Implementation - Refactored Orchestrator

This is the refactored version that delegates to specialized services
instead of doing everything itself (God Object anti-pattern eliminated).

BEFORE: 1400+ lines, does everything
AFTER:  ~300 lines, orchestrates services

Architecture Pattern: Facade + Dependency Injection
"""

from pathlib import Path
from typing import Dict, List, Optional
import logging
from datetime import datetime

# Import version from the dedicated version file
try:
    from .. import __version__
except ImportError:
    try:
        from specpulse._version import __version__
    except ImportError:
        # Fallback for development environment
        __version__ = "2.3.2"

from .template_provider import TemplateProvider
from .memory_provider import MemoryProvider
from .script_generator import ScriptGenerator
from .ai_instruction_provider import AIInstructionProvider
from .decomposition_service import DecompositionService
from .service_container import ServiceContainer
from .interfaces import (
    ITemplateProvider,
    IMemoryProvider,
    IScriptGenerator,
    IAIInstructionProvider,
    IDecompositionService
)

logger = logging.getLogger(__name__)


class SpecPulse:
    """
    Core SpecPulse functionality - Refactored as Service Orchestrator.

    This class now delegates to specialized services instead of implementing
    everything directly. Services are injected via constructor or service container.

    Design Pattern: Facade Pattern + Dependency Injection
    """

    def __init__(
        self,
        project_path: Optional[Path] = None,
        container: Optional[ServiceContainer] = None
    ):
        """
        Initialize SpecPulse orchestrator.

        Args:
            project_path: Project root path (default: cwd)
            container: Optional service container for DI
        """
        from ..utils.error_handler import ResourceError
        import yaml

        self.project_path = project_path or Path.cwd()
        self.config = self._load_config()

        # Resource directory resolution
        try:
            from importlib.resources import files
            resource_anchor = files('specpulse')
            self.resources_dir = Path(str(resource_anchor / 'resources'))
        except (ImportError, TypeError, AttributeError) as e:
            self.resources_dir = Path(__file__).parent.parent / "resources"
            if not self.resources_dir.exists():
                raise ResourceError("resources", self.resources_dir) from e

        self.templates_dir = self.resources_dir / "templates"

        # Initialize services (Dependency Injection)
        if container:
            # Use provided container
            self.template_provider = container.resolve(ITemplateProvider)
            self.memory_provider = container.resolve(IMemoryProvider)
            self.script_generator = container.resolve(IScriptGenerator)
            self.ai_provider = container.resolve(IAIInstructionProvider)
            self.decomposition_service = container.resolve(IDecompositionService)
        else:
            # Create services directly (for backward compatibility)
            self.template_provider = TemplateProvider(self.resources_dir)
            self.memory_provider = MemoryProvider(self.resources_dir)
            self.script_generator = ScriptGenerator(self.resources_dir)
            self.ai_provider = AIInstructionProvider(self.resources_dir)
            self.decomposition_service = DecompositionService(
                self.resources_dir,
                self.template_provider
            )

    def _load_config(self) -> Dict:
        """Load project configuration"""
        import yaml
        config_path = self.project_path / ".specpulse" / "config.yaml"
        if config_path.exists():
            try:
                with open(config_path, 'r') as f:
                    return yaml.safe_load(f) or {}
            except:
                return {}
        return {}

    # ======================================================================
    # TEMPLATE METHODS - Delegate to TemplateProvider
    # ======================================================================

    def get_spec_template(self) -> str:
        """Get specification template (delegated)"""
        return self.template_provider.get_spec_template()

    def get_plan_template(self) -> str:
        """Get implementation plan template (delegated)"""
        return self.template_provider.get_plan_template()

    def get_task_template(self) -> str:
        """Get task list template (delegated)"""
        return self.template_provider.get_task_template()

    def get_template(self, template_name: str, variables: Optional[Dict] = None) -> str:
        """Get generic template (delegated)"""
        return self.template_provider.get_template(template_name, variables)

    def get_decomposition_template(self, template_type: str = "microservices") -> str:
        """Get decomposition template (delegated)"""
        return self.template_provider.get_decomposition_template(template_type)

    def get_microservice_template(self) -> str:
        """Get microservice template (delegated)"""
        return self.template_provider.get_microservice_template()

    def get_api_contract_template(self) -> str:
        """Get API contract template (delegated)"""
        return self.template_provider.get_api_contract_template()

    def get_interface_template(self) -> str:
        """Get interface template (delegated)"""
        return self.template_provider.get_interface_template()

    def get_service_plan_template(self) -> str:
        """Get service plan template (delegated)"""
        return self.template_provider.get_service_plan_template()

    def get_integration_plan_template(self) -> str:
        """Get integration plan template (delegated)"""
        return self.template_provider.get_integration_plan_template()

    # ======================================================================
    # MEMORY METHODS - Delegate to MemoryProvider
    # ======================================================================

    def get_constitution_template(self) -> str:
        """Get constitution template (delegated)"""
        return self.memory_provider.get_constitution_template()

    def get_context_template(self) -> str:
        """Get context template (delegated)"""
        return self.memory_provider.get_context_template()

    def get_decisions_template(self) -> str:
        """Get decisions template (delegated)"""
        return self.memory_provider.get_decisions_template()

    # ======================================================================
    # SCRIPT METHODS - Delegate to ScriptGenerator
    # ======================================================================

    def get_setup_script(self) -> str:
        """Get setup script (delegated)"""
        return self.script_generator.get_setup_script()

    def get_spec_script(self) -> str:
        """Get spec script (delegated)"""
        return self.script_generator.get_spec_script()

    def get_plan_script(self) -> str:
        """Get plan script (delegated)"""
        return self.script_generator.get_plan_script()

    def get_task_script(self) -> str:
        """Get task script (delegated)"""
        return self.script_generator.get_task_script()

    def get_validate_script(self) -> str:
        """Get validate script (delegated)"""
        return self.script_generator.get_validate_script()

    def get_generate_script(self) -> str:
        """Get generate script (delegated)"""
        return self.script_generator.get_generate_script()

    # ======================================================================
    # AI INSTRUCTION METHODS - Delegate to AIInstructionProvider
    # ======================================================================

    def get_claude_instructions(self) -> str:
        """Get Claude instructions (delegated)"""
        return self.ai_provider.get_claude_instructions()

    def get_gemini_instructions(self) -> str:
        """Get Gemini instructions (delegated)"""
        return self.ai_provider.get_gemini_instructions()

    def get_claude_pulse_command(self) -> str:
        """Get Claude pulse command (delegated)"""
        return self.ai_provider.get_claude_pulse_command()

    def get_claude_spec_command(self) -> str:
        """Get Claude spec command (delegated)"""
        return self.ai_provider.get_claude_spec_command()

    def get_claude_plan_command(self) -> str:
        """Get Claude plan command (delegated)"""
        return self.ai_provider.get_claude_plan_command()

    def get_claude_task_command(self) -> str:
        """Get Claude task command (delegated)"""
        return self.ai_provider.get_claude_task_command()

    def get_claude_execute_command(self) -> str:
        """Get Claude execute command (delegated)"""
        return self.ai_provider.get_claude_execute_command()

    def get_claude_validate_command(self) -> str:
        """Get Claude validate command (delegated)"""
        return self.ai_provider.get_claude_validate_command()

    def get_claude_decompose_command(self) -> str:
        """Get Claude decompose command (delegated)"""
        return self.ai_provider.get_claude_decompose_command()

    def get_gemini_pulse_command(self) -> str:
        """Get Gemini pulse command (delegated)"""
        return self.ai_provider.get_gemini_pulse_command()

    def get_gemini_spec_command(self) -> str:
        """Get Gemini spec command (delegated)"""
        return self.ai_provider.get_gemini_spec_command()

    def get_gemini_plan_command(self) -> str:
        """Get Gemini plan command (delegated)"""
        return self.ai_provider.get_gemini_plan_command()

    def get_gemini_task_command(self) -> str:
        """Get Gemini task command (delegated)"""
        return self.ai_provider.get_gemini_task_command()

    def get_gemini_execute_command(self) -> str:
        """Get Gemini execute command (delegated)"""
        return self.ai_provider.get_gemini_execute_command()

    def get_gemini_validate_command(self) -> str:
        """Get Gemini validate command (delegated)"""
        return self.ai_provider.get_gemini_validate_command()

    def get_gemini_decompose_command(self) -> str:
        """Get Gemini decompose command (delegated)"""
        return self.ai_provider.get_gemini_decompose_command()

    def generate_claude_commands(self) -> List[Dict]:
        """Generate Claude commands (delegated)"""
        return self.ai_provider.generate_claude_commands()

    def generate_gemini_commands(self) -> List[Dict]:
        """Generate Gemini commands (delegated)"""
        return self.ai_provider.generate_gemini_commands()

    # ======================================================================
    # DECOMPOSITION METHODS - Delegate to DecompositionService
    # ======================================================================

    def decompose_specification(self, spec_dir: Path, spec_content: str) -> Dict:
        """Decompose specification (delegated)"""
        return self.decomposition_service.decompose_specification(spec_dir, spec_content)

    # ======================================================================
    # PROJECT INITIALIZATION
    # ======================================================================

    def init(self, project_name: Optional[str] = None, here: bool = False,
             ai_assistant: Optional[str] = None, template_source: str = 'local',
             console=None) -> Dict:
        """
        Initialize a new SpecPulse project

        Args:
            project_name: Name of the project
            here: Initialize in current directory
            ai_assistant: AI assistant to configure (claude or gemini)
            template_source: Template source (local or remote)
            console: Console instance for output

        Returns:
            Dict with initialization result
        """
        import sys
        import os

        # Set UTF-8 encoding to avoid Windows charmap issues
        if sys.platform == "win32":
            os.system('chcp 65001 > nul')

        from pathlib import Path
        from datetime import datetime
        import yaml
        import re
        from ..utils.error_handler import ValidationError, ProjectStructureError
        from .. import __version__

        try:
            # Validate project name for invalid characters
            if project_name and not here:
                if not re.match(r'^[a-zA-Z0-9_-]+$', project_name):
                    raise ValidationError(
                        f"Project name contains invalid characters: {project_name}",
                        validation_type="project_name",
                        missing_items=["Valid characters: letters, numbers, underscore, hyphen"]
                    )

            if here:
                project_path = Path.cwd()
                project_name = project_path.name
            else:
                if not project_name:
                    # If no project name, initialize in current directory
                    project_path = Path.cwd()
                    project_name = project_path.name
                else:
                    project_path = Path.cwd() / project_name
                    if not project_path.exists():
                        project_path.mkdir(parents=True)

            # Validate project path
            if not project_path.exists():
                raise ProjectStructureError(
                    f"Project path does not exist: {project_path}",
                    missing_dirs=[str(project_path)]
                )

            # Import PathManager for centralized directory management
            from .path_manager import PathManager

            # Create path manager instance
            path_manager = PathManager(project_path, use_legacy_structure=False)

            # Create directory structure
            directories = [
                ".claude",
                ".claude/commands",
                ".gemini",
                ".gemini/commands",
                ".specpulse",
                ".specpulse/cache",
                ".specpulse/specs",
                ".specpulse/plans",
                ".specpulse/tasks",
                ".specpulse/memory",
                ".specpulse/templates",
                ".specpulse/templates/decomposition",
                ".specpulse/checkpoints",
                ".specpulse/memory/notes",
                ".specpulse/docs"
            ]

            # Create directories
            failed_dirs = []
            for dir_name in directories:
                try:
                    dir_path = project_path / dir_name
                    dir_path.mkdir(parents=True, exist_ok=True)
                except Exception as e:
                    failed_dirs.append(dir_name)

            if failed_dirs:
                raise ProjectStructureError(
                    f"Failed to create {len(failed_dirs)} directories: {', '.join(failed_dirs)}",
                    missing_dirs=failed_dirs
                )

            # Create config file
            config = {
                "version": __version__,
                "project": {
                    "name": project_name,
                    "type": "web",
                    "created": datetime.now().isoformat()
                },
                "ai": {
                    "primary": ai_assistant or "claude"
                },
                "templates": {
                    "spec": ".specpulse/templates/spec.md",
                    "plan": ".specpulse/templates/plan.md",
                    "task": ".specpulse/templates/task.md"
                },
                "conventions": {
                    "branch_naming": "{number:03d}-{feature-name}",
                    "spec_naming": "spec-{number:03d}.md",
                    "plan_naming": "plan-{number:03d}.md",
                    "task_naming": "task-{number:03d}.md"
                }
            }

            config_path = project_path / ".specpulse" / "config.yaml"
            with open(config_path, 'w') as f:
                yaml.dump(config, f, default_flow_style=False)

            # Copy templates from resources
            self._copy_templates(project_path)

            # Copy AI command files
            self._copy_ai_commands(project_path, ai_assistant)

            # Create documentation
            self._create_documentation(project_path)

            # Create initial memory files
            self._create_initial_memory(project_path)

            return {
                "status": "success",
                "project_path": str(project_path),
                "project_name": project_name,
                "directories_created": directories,
                "ai_assistant": ai_assistant
            }

        except Exception as e:
            return {
                "status": "error",
                "error": str(e)
            }

    def _copy_templates(self, project_path: Path) -> None:
        """Copy template files from resources to project"""
        import shutil

        project_templates_dir = project_path / ".specpulse" / "templates"

        # Copy core templates
        core_templates = ["spec.md", "plan.md", "task.md"]
        for template in core_templates:
            src = self.templates_dir / template
            dst = project_templates_dir / template
            if src.exists():
                shutil.copy2(src, dst)

        # Copy decomposition templates
        decomp_dir = project_templates_dir / "decomposition"
        src_decomp = self.templates_dir / "decomposition"
        if src_decomp.exists():
            for template_file in src_decomp.glob("*.md"):
                dst = decomp_dir / template_file.name
                shutil.copy2(template_file, dst)

    def _copy_ai_commands(self, project_path: Path, ai_assistant: Optional[str]) -> None:
        """Copy AI command files based on chosen assistant"""
        import shutil

        commands_dir = self.resources_dir / "commands"

        # Copy Claude commands
        claude_commands = commands_dir / "claude"
        if claude_commands.exists():
            dst_claude = project_path / ".claude" / "commands"
            for cmd_file in claude_commands.glob("*.md"):
                shutil.copy2(cmd_file, dst_claude / cmd_file.name)

        # Copy Gemini commands
        gemini_commands = commands_dir / "gemini"
        if gemini_commands.exists():
            dst_gemini = project_path / ".gemini" / "commands"
            for cmd_file in gemini_commands.glob("*.toml"):
                shutil.copy2(cmd_file, dst_gemini / cmd_file.name)

    def _create_documentation(self, project_path: Path) -> None:
        """Create centralized documentation for AI commands and fallback procedures"""
        docs_dir = project_path / ".specpulse" / "docs"

        # Create AI_FALLBACK_GUIDE.md
        fallback_guide_content = f"""# SpecPulse AI Fallback Guide

This guide provides fallback procedures when SpecPulse CLI commands fail, ensuring AI can continue operations.

## 🚨 CLI Failure Detection

AI should detect CLI failures through:
1. **Exit codes**: Non-zero exit codes indicate failure
2. **Error patterns**: "command not found", "No such file", "Permission denied"
3. **Timeout**: Commands taking too long (>30 seconds)
4. **Missing dependencies**: Required tools not available

## 🛠️ Standard Fallback Procedures

### 1. Directory Structure Creation

**CLI Command**: `specpulse feature init <name>`
**Fallback**: Manual directory creation
```bash
# Detect current project root
PROJECT_ROOT=$(pwd)

# Create feature directory structure
FEATURE_DIR="$PROJECT_ROOT/.specpulse/specs/001-feature-name"
mkdir -p "$FEATURE_DIR"
mkdir -p "$PROJECT_ROOT/.specpulse/plans/001-feature-name"
mkdir -p "$PROJECT_ROOT/.specpulse/tasks/001-feature-name"

# Update context.md
echo "### Active Feature: Feature Name" >> "$PROJECT_ROOT/.specpulse/memory/context.md"
echo "- Feature ID: 001" >> "$PROJECT_ROOT/.specpulse/memory/context.md"
```

### 2. Specification Creation

**CLI Command**: `specpulse spec create "description"`
**Fallback**: Manual spec file creation
```bash
# Find next spec number
SPEC_NUM=$(ls "$FEATURE_DIR" | grep "spec-" | wc -l | awk '{{printf "%03d", $1+1}}')

# Create spec file with embedded template
cat > "$FEATURE_DIR/spec-$SPEC_NUM.md" << 'EOF'
# Specification: [DESCRIPTION]

<!-- FEATURE_DIR: 001-feature-name -->
<!-- FEATURE_ID: 001 -->
<!-- SPEC_NUMBER: SPEC_NUM -->
<!-- STATUS: pending -->
<!-- CREATED: CURRENT_TIMESTAMP -->

## Description
[DESCRIPTION]

## Requirements

### Functional Requirements
- [ ] Requirement 1
- [ ] Requirement 2

### Non-Functional Requirements
- [ ] Performance requirement
- [ ] Security requirement

## Acceptance Criteria

### User Stories
- **As a** [user role], **I want** [functionality], **so that** [benefit]
  - **Given** [context]
  - **When** [action]
  - **Then** [expected outcome]

## Technical Specifications

### Architecture
[Technical details]

### Dependencies
[External dependencies]

## Out of Scope
[What's not included]

## Success Metrics
[How to measure success]

## [NEEDS CLARIFICATION: Any uncertainties?]
EOF
```

### 3. Plan Creation

**CLI Command**: `specpulse plan create "description"`
**Fallback**: Manual plan file creation
```bash
# Find next plan number
PLAN_NUM=$(ls "$PROJECT_ROOT/.specpulse/plans/001-feature-name" | grep "plan-" | wc -l | awk '{{printf "%03d", $1+1}}')

# Create plan file with embedded template
cat > "$PROJECT_ROOT/.specpulse/plans/001-feature-name/plan-$PLAN_NUM.md" << 'EOF'
# Implementation Plan: [DESCRIPTION]

<!-- FEATURE_DIR: 001-feature-name -->
<!-- FEATURE_ID: 001 -->
<!-- PLAN_NUMBER: PLAN_NUM -->
<!-- STATUS: pending -->
<!-- CREATED: CURRENT_TIMESTAMP -->

## Description
[DESCRIPTION]

## Implementation Phases

### Phase 1: Foundation
- [ ] Task 1.1
- [ ] Task 1.2

### Phase 2: Core Features
- [ ] Task 2.1
- [ ] Task 2.2

### Phase 3: Polish & Testing
- [ ] Task 3.1
- [ ] Task 3.2

## Technology Stack
- [Frontend]: [Technologies]
- [Backend]: [Technologies]
- [Database]: [Database]
- [Infrastructure]: [Infrastructure]

## Timeline
- Phase 1: [X] weeks
- Phase 2: [Y] weeks
- Phase 3: [Z] weeks

## Dependencies
- External: [Dependencies]
- Internal: [Dependencies]

## Risks & Mitigations
- [Risk 1]: [Mitigation]
- [Risk 2]: [Mitigation]

## Success Criteria
- [ ] [Success criteria 1]
- [ ] [Success criteria 2]
EOF
```

### 4. Task Breakdown

**CLI Command**: `specpulse task breakdown <plan-id>`
**Fallback**: Manual task file creation
```bash
# Create task breakdown marker
cat > "$PROJECT_ROOT/.specpulse/tasks/001-feature-name/_breakdown_from_plan-$PLAN_NUM.md" << 'EOF'
# Task Breakdown Request

**Plan ID**: $PLAN_NUM
**Description**: [Plan description]
**Created**: CURRENT_TIMESTAMP

## Tasks Generated from Plan

### Task Categories
- [ ] Backend Development
- [ ] Frontend Development
- [ ] Database Setup
- [ ] Testing

### Next Steps
1. Create individual task files
2. Assign task numbers
3. Define acceptance criteria
4. Set up dependencies
EOF

# Create initial task files
TASK_NUM=1

# Example task file
cat > "$PROJECT_ROOT/.specpulse/tasks/001-feature-name/task-$TASK_NUM.md" << 'EOF'
# Task T$TASK_NUM: [Task Title]

<!-- TASK_NUMBER: TASK_NUM -->
<!-- STATUS: pending -->
<!-- CREATED: CURRENT_TIMESTAMP -->

## Description
[Task description]

## Acceptance Criteria
- [ ] [Criterion 1]
- [ ] [Criterion 2]

## Implementation Steps
1. [Step 1]
2. [Step 2]
3. [Step 3]

## Dependencies
- [Depends on other tasks]

## Testing
- [Unit tests]
- [Integration tests]

## Notes
[Any additional notes]
EOF
```

## 🔄 Standard Error Recovery

### Permission Errors
```bash
# Try fixing permissions
chmod -R 755 "$PROJECT_ROOT/.specpulse"

# If still fails, work in temp directory
TEMP_DIR=$(mktemp -d)
echo "Working in temporary directory: $TEMP_DIR"
```

### Path Issues
```bash
# Convert to absolute paths
PROJECT_ROOT=$(pwd -P)

# Handle Windows paths
if [[ "$OSTYPE" == "msys" ]] || [[ "$OSTYPE" == "cygwin" ]]; then
    PROJECT_ROOT=$(cygpath -w "$PROJECT_ROOT")
fi
```

### Template Missing
```bash
# Use embedded templates as shown above
# Log the missing template
echo "Warning: Template not found, using embedded fallback" >&2
```

## 🤖 AI Integration Best Practices

### Always Try CLI First
1. **Execute CLI command with timeout**
2. **Check exit code and output**
3. **If failed, apply fallback immediately**
4. **Log the fallback usage**

### Error Handling Pattern
```python
# AI should use this pattern:
try:
    result = run_cli_command("specpulse spec create", description)
    if result.success:
        return result
except CLIError:
    log_warning("CLI failed, using manual fallback")
    return manual_spec_creation(description)
```

### Fallback Logging
Always log when fallbacks are used:
```bash
echo "[FALLBACK] CLI command failed: $cmd" >&2
echo "[FALLBACK] Using manual procedure" >&2
```

## 📞 Getting Help

If fallbacks fail repeatedly:

1. **Check SpecPulse Installation**
   ```bash
   python -m specpulse --version
   pip list | grep specpulse
   ```

2. **Check Environment**
   ```bash
   python --version
   which specpulse
   echo $PATH
   ```

3. **Manual Recovery**
   - Create directory structure manually
   - Use embedded templates
   - Track progress in simple text files
   - Focus on core functionality over tooling

## ✅ Success Criteria

Fallback is successful when:
- [ ] Files are created in correct locations
- [ ] Content follows expected format
- [ ] Metadata is properly set
- [ ] Progress can be tracked
- [ ] User can continue work without CLI

## 🚨 When to Escalate

Escalate to manual intervention when:
- Multiple fallbacks fail in sequence
- Critical system dependencies are missing
- File permissions cannot be resolved
- User reports persistent issues

**Remember: AI should always enable work to continue, even when tooling fails!**

---
*Generated by SpecPulse v{__version__}*
*Created: {datetime.now().isoformat()}*
"""

        with open(docs_dir / "AI_FALLBACK_GUIDE.md", 'w', encoding='utf-8') as f:
            f.write(fallback_guide_content)

        # Create AI_INTEGRATION.md
        integration_content = f"""# SpecPulse AI Integration Guide

This guide explains how AI assistants integrate with SpecPulse for specification-driven development.

## 🤖 Supported AI Platforms

### Claude Code
- **Location**: `.claude/commands/`
- **Commands**: `/sp-*` slash commands
- **Files**: Markdown format (.md)

### Gemini CLI
- **Location**: `.gemini/commands/`
- **Commands**: `/sp-*` commands
- **Files**: TOML format (.toml)

## 🔄 CLI-AI Collaboration Pattern

### Critical Design Principle: CLI First, AI Enhanced

```
User Request → AI Command → CLI Command → File Creation → AI Enhancement
     ↓              ↓           ↓            ↓           ↓
  /sp-spec     Claude/Gemini  specpulse    Empty spec   Detailed spec
  "OAuth2"    detects intent    create      template    expansion
```

### 1. AI Command Detection
AI platforms detect user intent and route to appropriate SpecPulse CLI commands:

**Claude Code**:
```bash
# User: /sp-spec "OAuth2 authentication with JWT"
# AI detects intent and runs:
specpulse spec create "OAuth2 authentication with JWT"
```

**Gemini CLI**:
```bash
# User: /sp-spec "User authentication system"
# AI detects intent and runs:
specpulse spec create "User authentication system"
```

### 2. CLI Command Execution
CLI commands create the foundation:
- ✅ Directory structure
- ✅ Empty templates with metadata
- ✅ File naming conventions
- ✅ Context updates

### 3. AI Content Enhancement
AI expands the CLI-created templates:
- 📝 Detailed specifications
- 🏗️ Implementation plans
- 📋 Task breakdowns
- 🔍 Technical insights

## 🛡️ Fallback Protection System

When CLI commands fail, AI automatically applies fallback procedures:

### Detection Patterns
```python
def detect_cli_failure(result):
    # Check exit codes
    if result.exit_code != 0:
        return True

    # Check error patterns
    error_patterns = [
        "command not found",
        "Permission denied",
        "No such file",
        "ModuleNotFoundError"
    ]

    for pattern in error_patterns:
        if pattern in result.stderr.lower():
            return True

    return False
```

### Fallback Procedures
1. **Directory Creation**: Manual mkdir commands
2. **Template Usage**: Embedded templates from AI
3. **Metadata Generation**: Automatic ID assignment
4. **File Operations**: Safe file creation with proper encoding

## 📋 Command Reference

### Feature Management
```bash
# Claude Code
/sp-pulse <feature-name>              # Initialize feature
/sp-continue <feature-id>            # Switch to existing feature
/sp-status                           # Show project status

# Gemini CLI
/sp-pulse <feature-name>              # Initialize feature
/sp-continue <feature-id>            # Switch to existing feature
/sp-status                           # Show project status

# CLI Equivalent
specpulse feature init <name>
specpulse feature continue <id>
specpulse feature list
```

### Specification Management
```bash
# Claude Code
/sp-spec create "description"         # Create specification
/sp-spec validate <spec-id>          # Validate specification
/sp-spec expand <spec-id>            # Expand with details

# Gemini CLI
/sp-spec create "description"         # Create specification
/sp-spec validate <spec-id>          # Validate specification
/sp-spec expand <spec-id>            # Expand with details

# CLI Equivalent
specpulse spec create "description"
specpulse spec validate <id>
```

### Planning & Tasks
```bash
# Claude Code
/sp-plan                             # Generate implementation plan
/sp-task <plan-id>                    # Create task breakdown
/sp-execute                          # Execute next task

# Gemini CLI
/sp-plan                             # Generate implementation plan
/sp-task <plan-id>                    # Create task breakdown
/sp-execute                          # Execute next task

# CLI Equivalent
specpulse plan create "description"
specpulse task breakdown <plan-id>
```

## 🔄 Workflow Examples

### Complete Feature Development

1. **Initialize Feature**
   ```bash
   /sp-pulse user-authentication
   # Creates: .specpulse/specs/001-user-authentication/
   ```

2. **Create Specification**
   ```bash
   /sp-spec create "OAuth2 login with JWT tokens"
   # Creates: spec-001.md with AI-enhanced content
   ```

3. **Generate Implementation Plan**
   ```bash
   /sp-plan
   # Creates: plan-001.md with detailed phases
   ```

4. **Break Down Tasks**
   ```bash
   /sp-task plan-001
   # Creates: task-001.md, task-002.md, etc.
   ```

5. **Execute Tasks**
   ```bash
   /sp-execute
   # Implements tasks sequentially
   ```

### Specification Refinement

1. **Initial Spec Creation**
   ```bash
   /sp-spec create "Payment processing system"
   ```

2. **Validation Check**
   ```bash
   /sp-spec validate spec-001
   ```

3. **Content Expansion**
   ```bash
   /sp-spec expand spec-001
   # AI adds detailed requirements, constraints, etc.
   ```

## 🔧 Configuration

### AI Assistant Setup
```yaml
# .specpulse/config.yaml
ai:
  primary: claude  # claude or gemini
  fallback: true   # Enable fallback protection
  logging: true    # Log AI operations

templates:
  spec: .specpulse/templates/spec.md
  plan: .specpulse/templates/plan.md
  task: .specpulse/templates/task.md
```

### Custom Commands
Users can create custom AI commands:

```markdown
# .claude/commands/custom-research.md
# Custom research command

## Usage
/research "topic"

## Implementation
AI should:
1. Research the topic using available tools
2. Create a specification in .specpulse/specs/
3. Generate implementation plan
4. Break down into actionable tasks
```

## 🚨 Error Handling

### CLI Failure Scenarios
1. **Command Not Found**
   - Apply fallback procedures
   - Use embedded templates
   - Log the failure

2. **Permission Denied**
   - Try alternative paths
   - Use temporary directory
   - Notify user

3. **Missing Dependencies**
   - Create manual workarounds
   - Use basic file operations
   - Continue with reduced functionality

### Recovery Procedures
```python
# AI should implement recovery logic
try:
    result = execute_cli_command(command)
except CLIError as e:
    log_error(f"CLI failed: {{e}}")
    fallback_result = apply_fallback_procedure()
    return fallback_result
```

## 📊 Success Metrics

### AI Integration Success
- ✅ **CLI Success Rate**: >95%
- ✅ **Fallback Success Rate**: >90%
- ✅ **User Satisfaction**: Seamless workflow
- ✅ **Error Recovery**: Zero user intervention required

### Quality Indicators
- ✅ **Specification Quality**: Complete and detailed
- ✅ **Plan Quality**: Realistic timelines and phases
- ✅ **Task Quality**: Actionable and specific
- ✅ **Progress Tracking**: Continuous monitoring

## 🔗 Resources

- **AI Fallback Guide**: `.specpulse/docs/AI_FALLBACK_GUIDE.md`
- **Project Status**: `.specpulse/memory/context.md`
- **Validation Rules**: `.specpulse/validation_rules.yaml`
- **Templates**: `.specpulse/templates/`

## 🤖 AI Best Practices

1. **Always try CLI first** - CLI provides structure and consistency
2. **Use fallbacks gracefully** - Never let user workflows stop
3. **Enhance, don't replace** - AI adds value to CLI foundation
4. **Maintain context** - Track project state and user preferences
5. **Validate continuously** - Ensure quality and completeness

---

*Generated by SpecPulse v{__version__}*
*Created: {datetime.now().isoformat()}*
"""

        with open(docs_dir / "AI_INTEGRATION.md", 'w', encoding='utf-8') as f:
            f.write(integration_content)

        # Create README.md for docs directory
        docs_readme = f"""# SpecPulse Documentation

This directory contains comprehensive documentation for SpecPulse AI integration and fallback procedures.

## 📚 Available Documents

### [AI_FALLBACK_GUIDE.md](AI_FALLBACK_GUIDE.md)
Complete fallback procedures when SpecPulse CLI commands fail. Essential for AI assistants to continue work even when CLI is unavailable.

### [AI_INTEGRATION.md](AI_INTEGRATION.md)
Comprehensive guide for AI platform integration with SpecPulse. Includes command reference, workflow examples, and best practices.

## 🎯 Quick Reference

### For AI Assistants
```bash
# If CLI fails, follow this guide:
.specpulse/docs/AI_FALLBACK_GUIDE.md

# For integration patterns:
.specpulse/docs/AI_INTEGRATION.md
```

### For Users
```bash
# Check project health
specpulse doctor

# Validate specifications
specpulse doctor --fix

# Get help
specpulse --help
```

## 🔗 External Resources

- **GitHub Repository**: https://github.com/specpulse/specpulse
- **PyPI Package**: https://pypi.org/project/specpulse/
- **Issues**: https://github.com/specpulse/specpulse/issues

---

*Documentation for SpecPulse v{__version__}*
"""

        with open(docs_dir / "README.md", 'w', encoding='utf-8') as f:
            f.write(docs_readme)

    def _create_initial_memory(self, project_path: Path) -> None:
        """Create initial memory files"""
        memory_dir = project_path / ".specpulse" / "memory"

        # Create context.md
        context_content = f"""# Project Context

## Project: {project_path.name}
- **Created**: {datetime.now().isoformat()}
- **SpecPulse Version**: {__version__}
- **AI Assistant**: Not configured

## Active Feature: None
No feature currently in progress.

## Recent Activity
Project initialized successfully.

---
*This file is automatically maintained by SpecPulse*
"""

        with open(memory_dir / "context.md", 'w', encoding='utf-8') as f:
            f.write(context_content)

        # Create decisions.md
        decisions_content = """# Architectural Decisions

No decisions recorded yet.

## Decision Log Format
- **AD-[number]**: [Decision Title] (Date)
  - **Status**: [Proposed/Approved/Implemented/Deprecated]
  - **Context**: Background and problem
  - **Decision**: What was decided
  - **Consequences**: Impact of this decision

---
*This file is automatically maintained by SpecPulse*
"""

        with open(memory_dir / "decisions.md", 'w', encoding='utf-8') as f:
            f.write(decisions_content)


__all__ = ['SpecPulse']
