# DepsPy

Асинхронная библиотека для работы с [Deps API](https://docs.depscian.tech/) на Python.

## Установка

```bash
pip install depspy
```

## Использование

```python
import asyncio
from depspy import DepsClient

async def main():
    async with DepsClient("YOUR_API_KEY") as client:
        player = await client.get_player("Nicolas_Reed", 5)
        if player:
            print(player)
        online = await client.get_online_players(5)
        if online and online.data:
            print(f"Онлайн игроков: {len(online.data)}")
        fractions = await client.get_fractions(5)
        if fractions and fractions.data:
            print(f"Доступно {len(fractions.data)} фракций: {fractions.data}")
        if fractions and fractions.data:
            for fraction_id in fractions.data:
                try:
                    fraction_online = await client.get_fraction_online(5, fraction_id)
                    if fraction_online and fraction_online.data:
                        print(f"Онлайн во фракции '{fraction_id}': {len(fraction_online.data)}")
                except Exception as e:
                    print(f"Не удалось получить онлайн для фракции '{fraction_id}': {e}")

if __name__ == "__main__":
    asyncio.run(main())
```

## Особенности

- Полностью асинхронный API
- Автоматическая обработка ошибок
- Поддержка корпоративных ключей
- Автоматические повторные попытки при ошибках
- Типизация данных с помощью Pydantic
- Поддержка контекстного менеджера
- Встроенное кэширование запросов
- Валидация входных данных
- Подробное логирование
- Поддержка прокси
- Настраиваемые таймауты
- SSL верификация

## Методы API

### Основные методы
- `get_player(nickname: str, server_id: int)` - Получение информации об игроке
- `get_interviews(server_id: int)` - Получение информации о собеседованиях
- `get_online_players(server_id: int)` - Получение списка онлайн игроков
- `get_fractions(server_id: int)` - Получение списка фракций
- `get_fraction_online(server_id: int, fraction_id: str)` - Получение онлайн игроков фракции
- `get_admins(server_id: int)` - Получение списка администраторов
- `get_status()` - Получение статуса серверов

### Дополнительные методы
- `get_player_by_id(player_id: int, server_id: int)` - Получение информации об игроке по ID
- `get_server_info(server_id: int)` - Получение информации о сервере
- `get_online_count(server_id: int)` - Получение количества онлайн игроков
- `get_fraction_members_count(server_id: int, fraction_id: str)` - Получение количества игроков во фракции
- `is_player_online(nickname: str, server_id: int)` - Проверка онлайн статуса игрока
- `get_player_level(nickname: str, server_id: int)` - Получение уровня игрока
- `get_player_money(nickname: str, server_id: int)` - Получение информации о деньгах игрока
- `get_player_organization(nickname: str, server_id: int)` - Получение информации об организации игрока
- `get_player_property(nickname: str, server_id: int)` - Получение информации о собственности игрока
- `get_player_vip_info(nickname: str, server_id: int)` - Получение VIP информации игрока

## Обработка ошибок

Библиотека предоставляет следующие исключения:

### Основные ошибки
- `UnauthorizedError` - Ошибка авторизации
- `RateLimitError` - Превышен лимит запросов
- `PlayerNotFoundError` - Игрок не найден
- `ServerNotFoundError` - Сервер не найден
- `APIError` - Общая ошибка API

### Дополнительные ошибки
- `InvalidAPIKeyError` - Недействительный API ключ
- `ExpiredAPIKeyError` - Истекший API ключ
- `InsufficientPermissionsError` - Недостаточно прав
- `InvalidServerIDError` - Недействительный ID сервера
- `InvalidNicknameError` - Недействительный никнейм
- `InvalidFractionIDError` - Недействительный ID фракции
- `ValidationError` - Ошибка валидации данных
- `MaintenanceError` - API на обслуживании
- `TimeoutError` - Таймаут запроса
- `ConnectionError` - Ошибка соединения
- `ProxyError` - Ошибка прокси
- `SSLError` - Ошибка SSL
- `DNSResolutionError` - Ошибка разрешения DNS

## Настройка клиента

```python
client = DepsClient(
    api_key="YOUR_API_KEY",
    corporate_key=False,  # Использовать корпоративный ключ
    base_url="https://api.depscian.tech/v2",  # Базовый URL API
    timeout=30,  # Таймаут запросов в секундах
    max_retries=3,  # Максимальное количество попыток
    cache_ttl=300,  # Время жизни кэша в секундах
    proxy="http://proxy.example.com:8080",  # Прокси сервер
    verify_ssl=True,  # Проверка SSL сертификата
    log_level=logging.INFO  # Уровень логирования
)
```

## Требования

- Python 3.8+
- aiohttp>=3.8.0
- pydantic>=2.0.0
