"""Constants for Pulp RPM plugin tests."""

# flake8: noqa

from urllib.parse import urljoin
from pulp_smash import config
from pulp_smash.pulp3.constants import (
    BASE_CONTENT_PATH,
    BASE_DISTRIBUTION_PATH,
    BASE_REPO_PATH,
    BASE_PATH,
    BASE_PUBLICATION_PATH,
    BASE_REMOTE_PATH,
)

RPM_COPY_PATH = urljoin(BASE_PATH, "rpm/copy/")
"""The URL used for copying RPM content between repos."""

PULP_FIXTURES_BASE_URL = config.get_config().get_fixtures_url()

DOWNLOAD_POLICIES = ["immediate", "on_demand", "streamed"]

RPM_PACKAGE_CONTENT_NAME = "rpm.package"

RPM_PACKAGECATEGORY_CONTENT_NAME = "rpm.packagecategory"

RPM_PACKAGEENVIRONMENT_CONTENT_NAME = "rpm.packageenvironment"

RPM_PACKAGEGROUP_CONTENT_NAME = "rpm.packagegroup"

RPM_PACKAGELANGPACKS_CONTENT_NAME = "rpm.packagelangpacks"

RPM_ADVISORY_CONTENT_NAME = "rpm.advisory"

RPM_ALT_LAYOUT_FIXTURE_URL = urljoin(PULP_FIXTURES_BASE_URL, "rpm-alt-layout/")
"""The URL to a signed RPM repository. See :data:`RPM_SIGNED_FIXTURE_URL`."""

RPM_CONTENT_PATH = urljoin(BASE_CONTENT_PATH, "rpm/packages/")
"""The location of RPM packages on the content endpoint."""

RPM_NAMESPACES = {
    "metadata/common": "http://linux.duke.edu/metadata/common",
    "metadata/filelists": "http://linux.duke.edu/metadata/filelists",
    "metadata/other": "http://linux.duke.edu/metadata/other",
    "metadata/repo": "http://linux.duke.edu/metadata/repo",
    "metadata/rpm": "http://linux.duke.edu/metadata/rpm",
}
"""Namespaces used by XML-based RPM metadata.

Many of the XML files generated by the ``createrepo`` utility make use of these
namespaces. Some of the files that use these namespaces are listed below:

metadata/common
    Used by ``repodata/primary.xml``.

metadata/filelists
    Used by ``repodata/filelists.xml``.

metadata/other
    Used by ``repodata/other.xml``.

metadata/repo
    Used by ``repodata/repomd.xml``.

metadata/rpm
    Used by ``repodata/repomd.xml``.
"""

RPM_DISTRIBUTION_PATH = urljoin(BASE_DISTRIBUTION_PATH, "rpm/rpm/")

RPM_REMOTE_PATH = urljoin(BASE_REMOTE_PATH, "rpm/rpm/")

RPM_REPO_PATH = urljoin(BASE_REPO_PATH, "rpm/rpm/")

RPM_PUBLICATION_PATH = urljoin(BASE_PUBLICATION_PATH, "rpm/rpm/")

RPM_SHA512_FIXTURE_URL = urljoin(PULP_FIXTURES_BASE_URL, "rpm-with-sha-512/")
"""The URL to an RPM repository with sha512 checksum."""

RPM_SIGNED_FIXTURE_URL = urljoin(PULP_FIXTURES_BASE_URL, "rpm-signed/")
"""The URL to a repository with signed RPM packages."""

RPM_COMPLEX_FIXTURE_URL = urljoin(PULP_FIXTURES_BASE_URL, "rpm-complex-pkg/")

RPM_SINGLE_REQUEST_UPLOAD = urljoin(BASE_PATH, "content/rpm/packages/")

RPM_UNSIGNED_FIXTURE_URL = urljoin(PULP_FIXTURES_BASE_URL, "rpm-unsigned/")

RPM_ZSTD_METADATA_FIXTURE_URL = urljoin(PULP_FIXTURES_BASE_URL, "rpm-zstd-metadata/")

RPM_INVALID_FIXTURE_URL = urljoin(PULP_FIXTURES_BASE_URL, "rpm-missing-filelists/")

RPM_MIRROR_LIST_GOOD_FIXTURE_URL = urljoin(PULP_FIXTURES_BASE_URL, "rpm-mirrorlist-good")
RPM_MIRROR_LIST_BAD_FIXTURE_URL = urljoin(PULP_FIXTURES_BASE_URL, "rpm-mirrorlist-bad")

RPM_SHA_FIXTURE_URL = urljoin(PULP_FIXTURES_BASE_URL, "rpm-with-sha")

RPM_PACKAGE_COUNT = 35
"""The number of packages available at
:data:`RPM_SIGNED_FIXTURE_URL` and :data:`RPM_UNSIGNED_FIXTURE_URL`
"""

RPM_PACKAGECATEGORY_COUNT = 1
"""The number of packagecategories."""

RPM_PACKAGEGROUP_COUNT = 2
"""The number of packagegroups."""

RPM_PACKAGELANGPACKS_COUNT = 1
"""The number of packagelangpacks."""

RPM_ADVISORY_COUNT = 4
"""The number of updated record units."""

RPM_FIXTURE_SUMMARY = {
    RPM_PACKAGE_CONTENT_NAME: RPM_PACKAGE_COUNT,
    RPM_ADVISORY_CONTENT_NAME: RPM_ADVISORY_COUNT,
    RPM_PACKAGECATEGORY_CONTENT_NAME: RPM_PACKAGECATEGORY_COUNT,
    RPM_PACKAGEGROUP_CONTENT_NAME: RPM_PACKAGEGROUP_COUNT,
    RPM_PACKAGELANGPACKS_CONTENT_NAME: RPM_PACKAGELANGPACKS_COUNT,
}
"""The breakdown of how many of each type of content unit are present in the
standard repositories, i.e. :data:`RPM_SIGNED_FIXTURE_URL` and
:data:`RPM_UNSIGNED_FIXTURE_URL`.  This matches the format output by the
"content_summary" field on "../repositories/../versions/../".
"""

RPM_UNSIGNED_FIXTURE_SIZE = 79260
"""Size in bytes of all the packages in the :data:`RPM_UNSIGNED_FIXTURE_URL`."""

FEDORA_MIRRORLIST_BASE = "https://mirrors.fedoraproject.org/mirrorlist"
FEDORA_MIRRORLIST_PARAMS = "?repo=epel-modular-8&arch=x86_64&infra=stock&content=centos"
RPM_EPEL_MIRROR_URL = FEDORA_MIRRORLIST_BASE + FEDORA_MIRRORLIST_PARAMS
"""The URL to retrieve a mirrorlist for the EPEL-8-MODULAR repo.

.. NOTE:: This repository is not generated by pulp-fixtures.
"""

RPM_LONG_UPDATEINFO_FIXTURE_URL = urljoin(PULP_FIXTURES_BASE_URL, "rpm-long-updateinfo/")
"""The URL to RPM with a long updateinfo.xml."""

RPM_MODULAR_FIXTURE_URL = urljoin(PULP_FIXTURES_BASE_URL, "rpm-modular/")
"""The URL to a modular RPM repository."""

RPM_MODULAR_PACKAGE_COUNT = 25
"""The number of packages present on `RPM_MODULAR_FIXTURE_URL`."""

RPM_MODULAR_MODULES_COUNT = 11
"""The number of modules present on `RPM_MODULAR_FIXTURE_URL`."""

RPM_MODULAR_DEFAULTS_COUNT = 4
"""The number of modules-default present on `RPM_MODULAR_FIXTURE_URL`."""

RPM_MODULAR_ADVISORY_COUNT = 1
"""The number of advisories present on `RPM_MODULAR_FIXTURE_URL`."""

RPM_MODULAR_MODULES_CONTENT_NAME = "rpm.modulemd"

RPM_MODULAR_DEFAULTS_CONTENT_NAME = "rpm.modulemd_defaults"

RPM_MODULES_OBSOLETE_CONTENT_NAME = "rpm.modulemd_obsolete"

RPM_MODULES_OBSOLETE_COUNT = 6
"""The number of modules-default present on `RPM_MODULAR_FIXTURE_URL`."""

RPM_MODULAR_FIXTURE_SUMMARY = {
    RPM_PACKAGE_CONTENT_NAME: RPM_MODULAR_PACKAGE_COUNT,
    RPM_MODULAR_MODULES_CONTENT_NAME: RPM_MODULAR_MODULES_COUNT,
    RPM_MODULAR_DEFAULTS_CONTENT_NAME: RPM_MODULAR_DEFAULTS_COUNT,
    RPM_MODULES_OBSOLETE_CONTENT_NAME: RPM_MODULES_OBSOLETE_COUNT,
    RPM_ADVISORY_CONTENT_NAME: RPM_MODULAR_ADVISORY_COUNT,
}

RPM_MODULES_STATIC_CONTEXT_FIXTURE_URL = urljoin(
    PULP_FIXTURES_BASE_URL, "rpm-modules-static-context/"
)
"""The URL to a modular RPM repository that uses the static_context field."""

RPM_MODULAR_STATIC_FIXTURE_SUMMARY = {
    RPM_PACKAGE_CONTENT_NAME: RPM_MODULAR_PACKAGE_COUNT,
    RPM_MODULAR_MODULES_CONTENT_NAME: RPM_MODULAR_MODULES_COUNT,
    RPM_MODULAR_DEFAULTS_CONTENT_NAME: RPM_MODULAR_DEFAULTS_COUNT,
    RPM_MODULES_OBSOLETE_CONTENT_NAME: RPM_MODULES_OBSOLETE_COUNT,
    RPM_ADVISORY_CONTENT_NAME: RPM_MODULAR_ADVISORY_COUNT,
}

"""The breakdown of how many of each type of content unit are present in the
i.e. :data:`RPM_MODULAR_FIXTURE_URL`."""

RPM_PACKAGE_DATA = {
    "name": "kangaroo",
    "epoch": "0",
    "version": "0.3",
    "release": "1",
    "arch": "noarch",
    "description": "A modular RPM fixture for testing Pulp.",
    "summary": "hop like a kangaroo in Australia",
    "rpm_license": "Public Domain",
    "rpm_group": "Unspecified",
    "rpm_vendor": "",
    # TODO: Complete this information once we figure out how to serialize
    # everything nicely
}
"""The metadata for one RPM package."""


RPM_COMPLEX_PACKAGE_DATA = {
    "arch": "x86_64",
    "artifact": None,
    "changelogs": [
        [
            "Lucille Bluth <lucille@bluthcompany.com> - 1.1.1-1",
            1617192000,
            "- It's a banana, Michael. How much could it cost, $10?",
        ],
        [
            "Job Bluth <job@alliance-of-magicians.com> - 2.2.2-2",
            1619352000,
            "- I've made a huge mistake",
        ],
        [
            "George Bluth <george@federalprison.gov> - 3.3.3-3",
            1623672000,
            "- There’s always money in the banana stand",
        ],
    ],
    "checksum_type": "sha256",
    "conflicts": [["foxnetwork", "GT", "0", "5555", None, False]],
    "description": "Complex package",
    "enhances": [["(bananas or magic)", None, None, None, None, False]],
    "epoch": "1",
    "files": [
        [None, "/etc/complex/", "pkg.cfg"],
        [None, "/usr/bin/", "complex_a"],
        ["dir", "/usr/share/doc/", "complex-package"],
        [None, "/usr/share/doc/complex-package/", "README"],
        ["dir", "/var/lib/", "complex"],
        ["ghost", "/var/log/", "complex.log"],
    ],
    "is_modular": False,
    "location_base": "",
    "location_href": "complex-package-2.3.4-5.el8.x86_64.rpm",
    "md5": None,
    "name": "complex-package",
    "obsoletes": [
        ["bluemangroup", "LT", "0", "32.1", "0", False],
        ["cornballer", "LT", "0", "444", None, False],
    ],
    "pkg_id": "bbb7b0e9350a0f75b923bdd0ef4f9af39765c668a3e70bfd3486ea9f0f618aaf",
    "provides": [
        ["/usr/bin/ls", None, None, None, None, False],
        ["complex-package", "EQ", "1", "2.3.4", "5.el8", False],
        ["complex-package(x86-64)", "EQ", "1", "2.3.4", "5.el8", False],
        ["laughter", "EQ", "0", "33", None, False],
        ["narration(ronhoward)", None, None, None, None, False],
    ],
    "recommends": [
        ["((hiding and attic) if light-treason)", None, None, None, None, False],
        ["GeneParmesan(PI)", None, None, None, None, False],
        ["yacht", "GT", "9", "11.0", "0", False],
    ],
    "release": "5.el8",
    "requires": [
        ["/usr/bin/bash", None, None, None, None, False],
        ["/usr/sbin/useradd", None, None, None, None, True],
        ["arson", "GE", "0", "1.0.0", "1", False],
        ["fur", "LE", "0", "2", None, False],
        ["staircar", "LE", "0", "99.1", "3", False],
    ],
    "rpm_buildhost": "localhost",
    "rpm_group": "Development/Tools",
    "rpm_header_end": 8413,
    "rpm_header_start": 4504,
    "rpm_license": "MPLv2",
    "rpm_packager": "Michael Bluth",
    "rpm_sourcerpm": "complex-package-2.3.4-5.el8.src.rpm",
    "rpm_vendor": "Bluth Company",
    "sha1": None,
    "sha224": None,
    "sha256": None,
    "sha384": None,
    "sha512": None,
    "size_archive": 932,
    "size_installed": 117,
    "size_package": 8680,
    "suggests": [
        ["(bobloblaw >= 1.1 if maritimelaw else anyone < 0.5.1-2)", None, None, None, None, False],
        ["(dove and return)", None, None, None, None, False],
        ["(job or money > 9000)", None, None, None, None, False],
    ],
    "summary": "A package for exercising many different features of RPM metadata",
    "supplements": [
        ["((hiding and illusion) unless alliance-of-magicians)", None, None, None, None, False],
        ["comedy", "EQ", "0", "11.1", "4", False],
    ],
    "time_build": 1627052743,
    "time_file": 1627056000,
    "url": "http://bobloblaw.com",
    "version": "2.3.4",
}


RPM_PACKAGE_DATA2 = {
    "name": "duck",
    "epoch": "0",
    "version": "0.7",
    "release": "1",
    "arch": "noarch",
    "description": "A dummy package of duck",
    "summary": "A dummy package of duck",
    "rpm_license": "GPLv2",
    "rpm_group": "Internet/Applications",
    "rpm_vendor": "",
    # TODO: Complete this information once we figure out how to serialize
    # everything nicely
}
"""The metadata for one RPM package."""

RPM_PACKAGE_NAME = "{}".format(RPM_PACKAGE_DATA["name"])
"""The name of one RPM package."""

RPM_PACKAGE_FILENAME = "{}-{}-{}.{}.rpm".format(
    RPM_PACKAGE_DATA["name"],
    RPM_PACKAGE_DATA["version"],
    RPM_PACKAGE_DATA["release"],
    RPM_PACKAGE_DATA["arch"],
)
"""The filename of one RPM package."""

RPM_PACKAGE_FILENAME2 = "{}-{}-{}.{}.rpm".format(
    RPM_PACKAGE_DATA2["name"],
    RPM_PACKAGE_DATA2["version"],
    RPM_PACKAGE_DATA2["release"],
    RPM_PACKAGE_DATA2["arch"],
)
"""The filename of one RPM package."""

RPM_REFERENCES_UPDATEINFO_URL = urljoin(PULP_FIXTURES_BASE_URL, "rpm-references-updateinfo/")
"""The URL to a repository with ``updateinfo.xml`` containing references.

This repository includes advisory with reference element (0, 1 or 2 references)
and without it.
"""

RPM_RICH_WEAK_FIXTURE_URL = urljoin(PULP_FIXTURES_BASE_URL, "rpm-richnweak-deps/")
"""The URL to an RPM repository with weak and rich dependencies."""

RPM_SIGNED_URL = urljoin(RPM_SIGNED_FIXTURE_URL, RPM_PACKAGE_FILENAME)
"""The path to a single signed RPM package."""

RPM_SIGNED_URL2 = urljoin(RPM_SIGNED_FIXTURE_URL, RPM_PACKAGE_FILENAME2)
"""The path to a single signed RPM package."""

RPM_UNSIGNED_URL = urljoin(RPM_UNSIGNED_FIXTURE_URL, RPM_PACKAGE_FILENAME)
"""The path to a single unsigned RPM package."""

RPM_UPDATED_UPDATEINFO_FIXTURE_URL = urljoin(PULP_FIXTURES_BASE_URL, "rpm-updated-updateinfo/")
"""The URL to a repository containing UpdateRecords (Advisory) with the same IDs
as the ones in the standard repositories, but with different metadata.

Note: This repository uses unsigned RPMs.
"""

RPM_ADVISORY_UPDATED_VERSION_URL = urljoin(PULP_FIXTURES_BASE_URL, "rpm-updated-updateversion")
"""The URL to a repository containing Advisories with same ID as the ones in the standard
unsigned rpm repository, but with updated Advisory version.
"""

RPM_ADVISORY_DIFFERENT_PKGLIST_URL = urljoin(PULP_FIXTURES_BASE_URL, "rpm-advisory-diffpkgs")
"""The URL to a repository containing Advisories with same ID and version as the ones in the
standard unsigned rpm repository, but with different pkglist.
"""

RPM_ADVISORY_DIFFERENT_REPO_URL = urljoin(PULP_FIXTURES_BASE_URL, "rpm-advisory-diff-repo")
"""The URL to a repository containing Advisories with same ID and version as the ones in the
standard unsigned rpm repository, but with different update_date and packages intersection.
"""

RPM_ADVISORY_INCOMPLETE_PKG_LIST_URL = urljoin(
    PULP_FIXTURES_BASE_URL, "rpm-advisory-incomplete-package-list"
)
"""The URL to a repository containing Advisories with same update_date and version as the ones
in the standard unsigned rpm repository, but pkglist intersection is non-empty and not equal
to either pkglist.
"""

RPM_ADVISORY_NO_UPDATED_DATE = urljoin(PULP_FIXTURES_BASE_URL, "rpm-advisory-no-update-date")
"""The URL to a repository containing Advisories with same id and version as the ones
in the standard unsigned rpm repository, but no updated_date.
"""

RPM_ADVISORY_TEST_ID = "RHEA-2012:0056"
RPM_ADVISORY_TEST_ID_NEW = "RHEA-2012:0058"
"""The ID of an UpdateRecord (advisory/erratum).

The package contained on this advisory is defined by
:data:`RPM_UPDATERECORD_RPM_NAME` and the advisory is present in the standard
repositories, i.e. :data:`RPM_SIGNED_FIXTURE_URL` and
:data:`RPM_UNSIGNED_FIXTURE_URL`.
"""
RPM_ADVISORY_TEST_REMOVE_COUNT = 3
RPM_ADVISORY_TEST_ADDED_COUNT = 6


RPM_REPO_METADATA_FIXTURE_URL = urljoin(PULP_FIXTURES_BASE_URL, "rpm-repo-metadata/")
"""The URL to RPM repository with custom repository metadata."""

RPM_UPDATERECORD_RPM_NAME = "gorilla"
"""The name of the RPM named by :data:`RPM_UPDATERECORD_ID`."""

RPM_WITH_NON_ASCII_NAME = "rpm-with-non-ascii"

RPM_WITH_NON_ASCII_URL = urljoin(
    PULP_FIXTURES_BASE_URL,
    "rpm-with-non-ascii/{}-1-1.fc25.noarch.rpm".format(RPM_WITH_NON_ASCII_NAME),
)
"""The URL to an RPM with non-ascii metadata in its header."""

RPM_WITH_NON_UTF_8_NAME = "rpm-with-non-utf-8"

RPM_WITH_NON_UTF_8_URL = urljoin(
    PULP_FIXTURES_BASE_URL,
    "rpm-with-non-utf-8/{}-1-1.fc25.noarch.rpm".format(RPM_WITH_NON_UTF_8_NAME),
)
"""The URL to an RPM with non-UTF-8 metadata in its header."""

SRPM_UNSIGNED_FIXTURE_URL = urljoin(PULP_FIXTURES_BASE_URL, "srpm-unsigned/")
"""The URL to a repository with unsigned SRPM packages."""

SRPM_UNSIGNED_FIXTURE_ADVISORY_COUNT = 2
"""Count of advisories in the repository."""

SRPM_UNSIGNED_FIXTURE_PACKAGE_COUNT = 3
"""Count of packages in the repository."""

UPDATERECORD_CONTENT_PATH = urljoin(BASE_CONTENT_PATH, "rpm/advisories/")
"""The location of RPM UpdateRecords on the content endpoint."""

KICKSTART_CONTENT_PATH = urljoin(BASE_CONTENT_PATH, "rpm/distribution_trees/")
"""The location of RPM Distribution Trees on the content endpoint."""

RPM_KICKSTART_FIXTURE_URL = urljoin(PULP_FIXTURES_BASE_URL, "rpm-distribution-tree/")
RPM_KICKSTART_ONLY_META_FIXTURE_URL = urljoin(
    PULP_FIXTURES_BASE_URL, "rpm-distribution-tree-metadata-only/"
)
RPM_DISTRIBUTION_TREE_CHANGED_ADDON_URL = urljoin(
    PULP_FIXTURES_BASE_URL, "rpm-distribution-tree-changed-addon/"
)
RPM_DISTRIBUTION_TREE_CHANGED_MAIN_URL = urljoin(
    PULP_FIXTURES_BASE_URL, "rpm-distribution-tree-changed-main/"
)
RPM_DISTRIBUTION_TREE_CHANGED_VARIANT_URL = urljoin(
    PULP_FIXTURES_BASE_URL, "rpm-distribution-tree-changed-variant/"
)

RPM_KICKSTART_CONTENT_NAME = "rpm.distribution_tree"

RPM_KICKSTART_COUNT = 1

RPM_KICKSTART_FIXTURE_SUMMARY = {
    RPM_KICKSTART_CONTENT_NAME: RPM_KICKSTART_COUNT,
    RPM_PACKAGE_CONTENT_NAME: 1,
    RPM_PACKAGECATEGORY_CONTENT_NAME: 1,
    RPM_PACKAGEENVIRONMENT_CONTENT_NAME: 1,
    RPM_PACKAGEGROUP_CONTENT_NAME: 1,
    RPM_PACKAGELANGPACKS_CONTENT_NAME: 1,
}

RPM_KICKSTART_FIXTURE_SIZE = 9917733

RPM_KICKSTART_REPOSITORY_ROOT_CONTENT = [
    ".treeinfo",
    "Dolphin/",
    "External/",
    "Land/",
    "LiveOS/",
    "Packages/",
    "Whale/",
    "images/",
]

RPM_KICKSTART_DATA = {
    "addons": [
        {
            "addon_id": "Whale",
            "name": "Whale addon",
            "packages": "Whale/Packages",
            "type": "addon",
            "uid": "Sea-Whale",
        },
        {
            "addon_id": "Dolphin",
            "name": "Dolphin addon",
            "packages": "Dolphin/Packages",
            "type": "addon",
            "uid": "Sea-Dolphin",
        },
    ],
    "arch": "x86_64",
    "base_product_name": None,
    "base_product_short": None,
    "base_product_version": None,
    "build_timestamp": 1539194952.0,
    "checksums": [
        {
            "checksum": "sha256:91052b444e73f3eebdb93d1fb1506597e96c92d8de9c1e3c3f36b07a57d0a18f",
            "path": "images/pxeboot/vmlinuz",
        },
        {
            "checksum": "sha256:30e14955ebf1352266dc2ff8067e68104607e750abb9d3b36582b8af909fcb58",
            "path": "images/pxeboot/upgrade.img",
        },
        {
            "checksum": "sha256:5647f05ec18958947d32874eeb788fa396a05d0bab7c1b71f112ceb7e9b31eee",
            "path": "images/pxeboot/initrd.img",
        },
        {
            "checksum": "sha256:54260288eda2bd151c71358c5f2c472654d54420dbf3f2c1aafcb933a42a3074",
            "path": "fixtures/rpm-distribution-tree/addons/whale/repodata/repomd.xml",
        },
        {
            "checksum": "sha256:b6ac17b542fc6188eafadb932d9635209e1498c1ee22ba2490f531fdbd4c49a3",
            "path": "fixtures/rpm-distribution-tree/addons/dolphin/repodata/repomd.xml",
        },
        {
            "checksum": "sha256:30e14955ebf1352266dc2ff8067e68104607e750abb9d3b36582b8af909fcb58",
            "path": "LiveOS/squashfs.img",
        },
    ],
    "discnum": 1,
    "header_version": "1.2",
    "images": [
        {
            "name": "upgrade",
            "path": "images/pxeboot/upgrade.img",
            "platforms": "x86_64, xen",
        },
        {
            "name": "kernel",
            "path": "images/pxeboot/vmlinuz",
            "platforms": "x86_64, xen",
        },
        {
            "name": "initrd",
            "path": "images/pxeboot/initrd.img",
            "platforms": "x86_64, xen",
        },
    ],
    "instimage": None,
    "mainimage": "LiveOS/squashfs.img",
    "release_is_layered": False,
    "release_name": "Red Hat Enterprise Linux",
    "release_short": "RHEL",
    "release_version": "7.6",
    "totaldiscs": 1,
    "variants": [
        {
            "debug_packages": None,
            "debug_repository": None,
            "identity": None,
            "name": "Sea",
            "packages": "Packages",
            "source_packages": None,
            "source_repository": None,
            "type": "variant",
            "uid": "Sea",
            "variant_id": "Sea",
        },
        {
            "debug_packages": None,
            "debug_repository": None,
            "identity": None,
            "name": "Land",
            "packages": "Land/Packages",
            "source_packages": None,
            "source_repository": None,
            "type": "variant",
            "uid": "Land",
            "variant_id": "Land",
        },
        {
            "debug_packages": None,
            "debug_repository": None,
            "identity": None,
            "name": "External",
            "packages": "External/Packages",
            "source_packages": None,
            "source_repository": None,
            "type": "variant",
            "uid": "External",
            "variant_id": "External",
        },
    ],
}

PULP_FIXTURES_COMMON_URL = "https://github.com/pulp/pulp-fixtures/raw/master/common/"
PUBLIC_GPG_KEY_URL = urljoin(PULP_FIXTURES_COMMON_URL, "GPG-KEY-fixture-signing")
PRIVATE_GPG_KEY_URL = urljoin(PULP_FIXTURES_COMMON_URL, "GPG-PRIVATE-KEY-fixture-signing")

RPM_CUSTOM_REPO_METADATA_FIXTURE_URL = urljoin(PULP_FIXTURES_BASE_URL, "rpm-repo-metadata/")
RPM_CUSTOM_REPO_METADATA_CHANGED_FIXTURE_URL = urljoin(
    PULP_FIXTURES_BASE_URL, "rpm-repo-metadata-changed/"
)

RPM_MD5_REPO_FIXTURE_URL = urljoin(PULP_FIXTURES_BASE_URL, "rpm-with-md5/")

RPM_DIFF_NAME_SAME_CONTENT_URL = urljoin(PULP_FIXTURES_BASE_URL, "diff-name-same-content/")

RPM_ONLY_METADATA_REPO_URL = urljoin(PULP_FIXTURES_BASE_URL, "rpm-unsigned-meta-only")

AMAZON_MIRROR = "http://amazonlinux.us-east-1.amazonaws.com/2/core/latest/x86_64/mirror.list"
CENTOS8_STREAM_BASEOS_URL = "http://vault.centos.org/centos/8-stream/BaseOS/x86_64/os/"
CENTOS8_STREAM_APPSTREAM_URL = "http://vault.centos.org/centos/8-stream/AppStream/x86_64/os/"
CENTOS9_STREAM_BASEOS_URL = "http://mirror.stream.centos.org/9-stream/BaseOS/x86_64/os/"
CENTOS9_STREAM_APPSTREAM_URL = "http://mirror.stream.centos.org/9-stream/AppStream/x86_64/os/"
EPEL8_URL = "https://dl.fedoraproject.org/pub/epel/8/Everything/x86_64/"
EPEL8_MIRRORLIST_URL = "https://mirrors.fedoraproject.org/mirrorlist?repo=epel-8&arch=x86_64"
RAWHIDE_KICKSTART_URL = (
    "https://dl.fedoraproject.org/pub/fedora/linux/development/rawhide/Server/x86_64/os/"
)
RHEL6_KICKSTART_CDN_URL = "https://cdn.redhat.com/content/dist/rhel/server/6/6.10/x86_64/kickstart/"
RHEL8_BASEOS_CDN_URL = "https://cdn.redhat.com/content/dist/rhel8/8/x86_64/baseos/os/"
RHEL8_APPSTREAM_CDN_URL = "https://cdn.redhat.com/content/dist/rhel8/8/x86_64/appstream/os/"
RHEL9_BASEOS_CDN_URL = "https://cdn.redhat.com/content/dist/rhel9/9/x86_64/baseos/os/"
RHEL9_APPSTREAM_CDN_URL = "https://cdn.redhat.com/content/dist/rhel9/9/x86_64/appstream/os/"

# TODO: static test fixtures for these
REPO_WITH_XML_BASE_URL = "https://harbottle.gitlab.io/harbottle-main/8/x86_64/"

PULP_TYPE_ADVISORY = "rpm.advisory"
PULP_TYPE_DISTRIBUTION_TREE = "rpm.distribution_tree"
PULP_TYPE_PACKAGE = "rpm.package"
PULP_TYPE_PACKAGE_CATEGORY = "rpm.packagecategory"
PULP_TYPE_PACKAGE_GROUP = "rpm.packagegroup"
PULP_TYPE_REPOMETADATA = "rpm.repo_metadata_file"
PULP_TYPE_MODULEMD = "rpm.modulemd"
PULP_TYPE_MODULEMD_DEFAULTS = "rpm.modulemd_defaults"

PACKAGES_DIRECTORY = "Packages"

"""The URL used for uploading comps.xml files."""
COMPS_API = urljoin(BASE_PATH, "rpm/comps/")
SMALL_COMPS_XML = """<?xml version="1.0" encoding="UTF-8"?>
<!DOCTYPE comps PUBLIC "-//Red Hat, Inc.//DTD Comps info//EN" "comps.dtd">
<comps>
  <group>
   <id>birds</id>
   <default>true</default>
   <uservisible>true</uservisible>
   <display_order>1024</display_order>
   <name>birds</name>
   <description></description>
    <packagelist>
      <packagereq type="mandatory">cockateel</packagereq>
      <packagereq type="mandatory">duck</packagereq>
      <packagereq type="mandatory">penguin</packagereq>
      <packagereq type="mandatory">stork</packagereq>
    </packagelist>
  </group>
  <group>
   <id>mammals</id>
   <default>true</default>
   <uservisible>true</uservisible>
   <display_order>1024</display_order>
   <name>mammals</name>
   <description></description>
    <packagelist>
      <packagereq type="mandatory">bear</packagereq>
      <packagereq type="mandatory">camel</packagereq>
      <packagereq type="mandatory">cat</packagereq>
      <packagereq type="mandatory">cheetah</packagereq>
      <packagereq type="mandatory">chimpanzee</packagereq>
      <packagereq type="mandatory">cow</packagereq>
      <packagereq type="mandatory">dog</packagereq>
      <packagereq type="mandatory">dolphin</packagereq>
      <packagereq type="mandatory">elephant</packagereq>
      <packagereq type="mandatory">fox</packagereq>
      <packagereq type="mandatory">giraffe</packagereq>
      <packagereq type="mandatory">gorilla</packagereq>
      <packagereq type="mandatory">horse</packagereq>
      <packagereq type="mandatory">kangaroo</packagereq>
      <packagereq type="mandatory">lion</packagereq>
      <packagereq type="mandatory">mouse</packagereq>
      <packagereq type="mandatory">squirrel</packagereq>
      <packagereq type="mandatory">tiger</packagereq>
      <packagereq type="mandatory">walrus</packagereq>
      <packagereq type="mandatory">whale</packagereq>
      <packagereq type="mandatory">wolf</packagereq>
      <packagereq type="mandatory">zebra</packagereq>
    </packagelist>
  </group>
  <category>
   <id>all</id>
   <display_order>99</display_order>
   <name>all</name>
   <description></description>
    <grouplist>
     <groupid>mammals</groupid>
     <groupid>birds</groupid>
    </grouplist>
  </category>
  <langpacks>
    <match install="gorilla-%s" name="gorilla-en"/>
    <match install="zebra-%s" name="zebra-docs"/>
  </langpacks>
</comps>"""
SMALL_GROUPS = 2
SMALL_CATEGORY = 1
SMALL_LANGPACK = 1
SMALL_ENVIRONMENTS = 0

BIG_COMPS_XML = """<?xml version="1.0" encoding="UTF-8"?>
<!DOCTYPE comps
  PUBLIC '-//Red Hat, Inc.//DTD Comps info//EN'
  'comps.dtd'>
<comps>
  <group>
    <id>additional-devel</id>
    <name>Additional Development</name>
    <name xml:lang="as">অতিৰিক্ত উন্নয়ন</name>
    <name xml:lang="bn">অতিরিক্ত ডিভেলপমেন্ট</name>
    <name xml:lang="cs">Doplňky pro vývoj</name>
    <name xml:lang="de">Zusätzliche Entwicklung</name>
    <name xml:lang="de_CH">Zusätzliche Entwicklung</name>
    <name xml:lang="es">Desarrollo adicional</name>
    <name xml:lang="fr">Développement supplémentaire</name>
    <name xml:lang="gu">વધારાનાં વિકાસ</name>
    <name xml:lang="hi">अतिरिक्त विकास</name>
    <name xml:lang="ia">Disveloppamento additional</name>
    <name xml:lang="it">Sviluppo aggiuntivo</name>
    <name xml:lang="ja">その他の開発</name>
    <name xml:lang="kn">ಹೆಚ್ಚುವರಿ ವಿಕಸನೆ</name>
    <name xml:lang="ko">추가 개발</name>
    <name xml:lang="ml">അധികമായ ഡവലപ്പ്മെന്റ്</name>
    <name xml:lang="mr">अगाऊ डेव्हलपमेंट</name>
    <name xml:lang="or">ଅତିରିକ୍ତ ସର୍ଭର ବିକାଶ</name>
    <name xml:lang="pa">ਵਧੀਕ ਵਿਕਾਸ</name>
    <name xml:lang="pl">Dodatkowe programowanie</name>
    <name xml:lang="pt">Desenvolvimento adicional</name>
    <name xml:lang="pt_BR">Desenvolvimento adicional</name>
    <name xml:lang="ru">Дополнительные средства разработки</name>
    <name xml:lang="sv">Ytterligare utveckling</name>
    <name xml:lang="ta">கூடுதல் உருவாக்கம்</name>
    <name xml:lang="te">అదనపు అభివృద్ది</name>
    <name xml:lang="uk">Додаткові інструменти розробки</name>
    <name xml:lang="zh">額外開發</name>
    <name xml:lang="zh_CN">附加开发</name>
    <name xml:lang="zh_TW">額外開發</name>
    <description>Additional development headers and libraries for building open-source</description>
    <description xml:lang="cs">Doplňkové knihovny a hlavičkové soubory pro sestavování</description>
    <description xml:lang="de">Zusätzliche Entwicklungs-Header und -Bibliotheken zur E</description>
    <description xml:lang="de_CH">Zusätzliche Entwicklungs-Header und -Bibliotheken zu</description>
    <description xml:lang="es">Encabezados adicionales y bibliotecas para compilar apl</description>
    <description xml:lang="fr">En-têtes et bibliothèques de développement supplémentai</description>
    <description xml:lang="it">Librerie e intestazioni per uno sviluppo aggiuntivo per</description>
    <description xml:lang="ko">오픈 소스 애플리케이션 구축을 위한 기타 개발 헤더 및 라이브러리입니다.</description>
    <description xml:lang="pl">Dodatkowe nagłówki i biblioteki do rozwijania aplikacji</description>
    <description xml:lang="pt">Cabeçalhos de desenvolvimento adiiconal e bibliotecas p</description>
    <description xml:lang="pt_BR">Cabeçalhos de desenvolvimento adiiconal e biblioteca</description>
    <description xml:lang="ru">Дополнительные библиотеки и файлы заголовков для разраб</description>
    <description xml:lang="uk">Додаткові пакунки з заголовками та бібліотеками для роз</description>
    <description xml:lang="zh">用來建置開放式原始碼應用程式的應用程式開發表頭與函式庫。</description>
    <description xml:lang="zh_CN">用于构建开源应用程序的附加开发标头及程序可。</description>
    <description xml:lang="zh_TW">用來建置開放式原始碼應用程式的應用程式開發表頭與函式庫。</description>
    <default>true</default>
    <uservisible>false</uservisible>
    <packagelist>
      <packagereq type="mandatory">atlas-devel</packagereq>
      <packagereq type="mandatory">audit-libs-devel</packagereq>
      <packagereq type="mandatory">bzip2-devel</packagereq>
      <packagereq type="mandatory">c-ares-devel</packagereq>
      <packagereq type="mandatory">cyrus-sasl-devel</packagereq>
      <packagereq type="mandatory">e2fsprogs-devel</packagereq>
      <packagereq type="mandatory">elfutils-devel</packagereq>
      <packagereq type="mandatory">elfutils-libelf-devel</packagereq>
      <packagereq type="mandatory">expat-devel</packagereq>
      <packagereq type="mandatory">fuse3-devel</packagereq>
      <packagereq type="mandatory">gmp-devel</packagereq>
      <packagereq type="mandatory">hmaccalc</packagereq>
      <packagereq type="mandatory">iptables-devel</packagereq>
      <packagereq type="mandatory">libacl-devel</packagereq>
      <packagereq type="mandatory">libaio-devel</packagereq>
      <packagereq type="mandatory">libattr-devel</packagereq>
      <packagereq type="mandatory">libblkid-devel</packagereq>
      <packagereq type="mandatory">libcap-devel</packagereq>
      <packagereq type="mandatory">libcap-ng-devel</packagereq>
      <packagereq type="mandatory">libcurl-devel</packagereq>
      <packagereq type="mandatory">libffi-devel</packagereq>
      <packagereq type="mandatory">libgcrypt-devel</packagereq>
      <packagereq type="mandatory">libhugetlbfs-devel</packagereq>
      <packagereq type="mandatory">libnl3-devel</packagereq>
      <packagereq type="mandatory">libselinux-devel</packagereq>
      <packagereq type="mandatory">libusbx-devel</packagereq>
      <packagereq type="mandatory">libuuid-devel</packagereq>
      <packagereq type="mandatory">lksctp-tools-devel</packagereq>
      <packagereq type="mandatory">lz4</packagereq>
      <packagereq type="mandatory">lz4-devel</packagereq>
      <packagereq type="mandatory">lzo</packagereq>
      <packagereq type="mandatory">lzo-devel</packagereq>
      <packagereq type="mandatory">numactl-devel</packagereq>
      <packagereq type="mandatory">pciutils-devel</packagereq>
      <packagereq type="mandatory">pcre-devel</packagereq>
      <packagereq type="mandatory">polkit-devel</packagereq>
      <packagereq type="mandatory">popt-devel</packagereq>
      <packagereq type="mandatory">rdma-core-devel</packagereq>
      <packagereq type="mandatory">readline-devel</packagereq>
      <packagereq type="mandatory">sqlite-devel</packagereq>
      <packagereq type="mandatory">systemd-devel</packagereq>
      <packagereq type="mandatory">tcl-devel</packagereq>
      <packagereq type="mandatory">xfsprogs-devel</packagereq>
      <packagereq type="mandatory">xz-devel</packagereq>
    </packagelist>
  </group>
  <group>
    <id>platform-devel</id>
    <name>Platform Development</name>
    <name xml:lang="as">প্লেটফৰ্ম উন্নয়ন</name>
    <name xml:lang="cs">Vývoj platformy</name>
    <name xml:lang="de">Plattformentwicklung</name>
    <name xml:lang="de_CH">Plattformentwicklung</name>
    <name xml:lang="es">Desarrollo de plataforma</name>
    <name xml:lang="fr">Développement plateforme</name>
    <name xml:lang="gu">પ્લેટફોર્મ વિકાસ</name>
    <name xml:lang="hi">प्लैटफॉर्म विकास</name>
    <name xml:lang="it">Sviluppo piattaforma</name>
    <name xml:lang="ja">プラットフォーム開発</name>
    <name xml:lang="kn">ಪ್ಲಾಟ್‌ಫಾರ್ಮ್ ವಿಕಸನ</name>
    <name xml:lang="ko">플랫폼 개발</name>
    <name xml:lang="ml">പ്ലാറ്റ്ഫോം വികസനം</name>
    <name xml:lang="mr">प्लॅटफॉर्म डेव्हलपमेंट</name>
    <name xml:lang="or">ପ୍ଲାଟଫର୍ମ ବିକାଶ</name>
    <name xml:lang="pa">ਮੰਚ ਵਿਕਾਸ</name>
    <name xml:lang="pl">Rozwijanie platformy</name>
    <name xml:lang="pt">Desenvolvimento de Plataforma</name>
    <name xml:lang="pt_BR">Desenvolvimento de Plataforma</name>
    <name xml:lang="ru">Средства разработки платформы</name>
    <name xml:lang="ta">இயங்கு தள உருவாக்கம்</name>
    <name xml:lang="te">ప్లాట్‌ఫాం అభివృద్ది</name>
    <name xml:lang="uk">Розробка для платформи</name>
    <name xml:lang="zh">平台開發</name>
    <name xml:lang="zh_CN">平台开发</name>
    <name xml:lang="zh_TW">平台開發</name>
    <description>Recommended development headers and libraries for developing applicat</description>
    <default>true</default>
    <uservisible>false</uservisible>
    <packagelist>
      <packagereq type="mandatory">fontconfig-devel</packagereq>
      <packagereq type="mandatory">freetype-devel</packagereq>
      <packagereq type="mandatory">glib2-devel</packagereq>
      <packagereq type="mandatory">glibc-devel</packagereq>
      <packagereq type="mandatory">krb5-devel</packagereq>
      <packagereq type="mandatory">libpng-devel</packagereq>
      <packagereq type="mandatory">ncurses-devel</packagereq>
      <packagereq type="mandatory">openldap-devel</packagereq>
      <packagereq type="mandatory">openssl-devel</packagereq>
      <packagereq type="mandatory">pam-devel</packagereq>
      <packagereq type="mandatory">zlib-devel</packagereq>
    </packagelist>
  </group>
  <group>
    <id>python-web</id>
    <name>Python Web</name>
    <name xml:lang="de">Python Web</name>
    <name xml:lang="es">Python Web</name>
    <name xml:lang="fr">Python Web</name>
    <name xml:lang="it">Python Web</name>
    <name xml:lang="ja">Python Web</name>
    <name xml:lang="ko">Python Web</name>
    <name xml:lang="pt_BR">Python Web</name>
    <name xml:lang="ru">Python Web</name>
    <name xml:lang="zh_CN">Python Web</name>
    <name xml:lang="zh_TW">Python Web</name>
    <description>Basic Python web application support.</description>
    <description xml:lang="as">মৌলিক Python ৱেব এপ্লিকেচন সমৰ্থন।</description>
    <description xml:lang="cs">Podpora pro základní webové aplikace v jazyce Python.</description>
    <description xml:lang="de">Grundlegende Unterstützung für die Python Webanwendung.</description>
    <description xml:lang="de_CH">Grundlegende Unterstützung für die Python Webanwendu</description>
    <description xml:lang="es">Soporte de aplicaciones web en Basic Python</description>
    <description xml:lang="fr">Prise en charge d'application web Python de base.</description>
    <description xml:lang="gu">મૂળભૂત Python વેબ કાર્યક્રમ આધાર.</description>
    <description xml:lang="hi">बेसिक पाइथन वेब अनुप्रयोग समर्थन.</description>
    <description xml:lang="it">Supporto applicazione web Python di base</description>
    <description xml:lang="ja">Python のベーシック web アプリケーションサポートです。</description>
    <description xml:lang="kn">ಮೂಲಭೂತ Python ಜಾಲ ಅನ್ವಯ ಬೆಂಬಲ.</description>
    <description xml:lang="ko">기본 Python 웹 애플리케이션 지원.</description>
    <description xml:lang="ml">അടിസ്ഥാന പൈഥണ്‍ വെബ് പ്രയോഗത്തിനുള്ള പിന്തുണ.</description>
    <description xml:lang="mr">बेसिक Python वेब ॲप्लिकेशन समर्थन.</description>
    <description xml:lang="or">ମୌଳିକ Python ୱେବ ପ୍ରୟୋଗ ସହାୟତା।</description>
    <description xml:lang="pa">ਮੁੱਢਲਾ ਪਾਇਥਨ ਵੈੱਬ ਐਪਲੀਕੇਸ਼ਨ ਸਮਰਥਨ।</description>
    <description xml:lang="pl">Podstawowa obsługa aplikacji WWW w języku Python.</description>
    <description xml:lang="pt">Suporte de aplicativo da Web Phyton Básico</description>
    <description xml:lang="pt_BR">Suporte de aplicativo da Web Phyton Básico</description>
    <description xml:lang="ru">Базовая поддержка веб-приложений Python.</description>
    <description xml:lang="ta">அடிப்படை Python வலைப் பயன்பாடு ஆதரவு.</description>
    <description xml:lang="te">సాధారణ Python వెబ్ అనువర్తన తోడ్పాటు.</description>
    <description xml:lang="uk">Базова підтримка вебпрограм мовою Python.</description>
    <description xml:lang="zh">基本的 Python 網站應用程式支援。</description>
    <description xml:lang="zh_CN">基本 Python 的 web 应用程序支持。</description>
    <description xml:lang="zh_TW">基本的 Python 網站應用程式支援。</description>
    <default>true</default>
    <uservisible>false</uservisible>
    <packagelist>
      <packagereq type="mandatory">freetype-devel</packagereq>
      <packagereq type="mandatory">libcurl</packagereq>
      <packagereq type="mandatory">libcurl-devel</packagereq>
      <packagereq type="mandatory">python3-magic</packagereq>
    </packagelist>
  </group>
  <category>
    <id>development</id>
    <name>Development</name>
    <name xml:lang="af">Ontwikkeling</name>
    <name xml:lang="am">እድገት</name>
    <name xml:lang="ar">تطوير</name>
    <name xml:lang="as">উন্নয়ন</name>
    <name xml:lang="bal">پیش روگ</name>
    <name xml:lang="be">Распрацоўка</name>
    <name xml:lang="bg">Разработка</name>
    <name xml:lang="bn">সফ্টওয়্যার নির্মাণ</name>
    <name xml:lang="bs">Razvoj</name>
    <name xml:lang="ca">Desenvolupament</name>
    <name xml:lang="cs">Vývoj</name>
    <name xml:lang="cy">Datblygiad</name>
    <name xml:lang="da">Udvikling</name>
    <name xml:lang="de">Softwareentwicklung</name>
    <name xml:lang="de_CH">Softwareentwicklung</name>
    <name xml:lang="el">Ανάπτυξη λογισμικού</name>
    <name xml:lang="en_GB">Development</name>
    <name xml:lang="es">Desarrollo</name>
    <name xml:lang="et">Arendus</name>
    <name xml:lang="eu">Garapena</name>
    <name xml:lang="fa">برنامه‌سازی</name>
    <name xml:lang="fa_IR">برنامه‌سازی</name>
    <name xml:lang="fi">Ohjelmistokehitys</name>
    <name xml:lang="fr">Développement</name>
    <name xml:lang="gl">Desenvolvemento</name>
    <name xml:lang="gu">વિકાસ</name>
    <name xml:lang="he">פיתוח</name>
    <name xml:lang="hi">विकास</name>
    <name xml:lang="hr">Razvoj</name>
    <name xml:lang="hu">Fejlesztőeszközök</name>
    <name xml:lang="hy">Ծրագրավորում</name>
    <name xml:lang="ia">Disveloppamento</name>
    <name xml:lang="id">Pengembangan</name>
    <name xml:lang="ilo">Panagbukel</name>
    <name xml:lang="is">Þróun</name>
    <name xml:lang="it">Sviluppo</name>
    <name xml:lang="ja">開発</name>
    <name xml:lang="ka">პრიოგრამირება</name>
    <name xml:lang="kn">ವಿಕಾಸನ</name>
    <name xml:lang="ko">개발</name>
    <name xml:lang="lv">Programmēšana</name>
    <name xml:lang="mai">विकास</name>
    <name xml:lang="mk">Развој</name>
    <name xml:lang="ml">ഡെവലപ്പ്മെന്‍റ്</name>
    <name xml:lang="mr">विकास</name>
    <name xml:lang="ms">Pembangunan</name>
    <name xml:lang="nb">Utviklingsverktøy</name>
    <name xml:lang="ne">विकास</name>
    <name xml:lang="nl">Ontwikkeling</name>
    <name xml:lang="no">Utviklingsverktøy</name>
    <name xml:lang="nso">Tšwetšopele</name>
    <name xml:lang="or">ବିକାଶ</name>
    <name xml:lang="pa">ਵਿਕਾਸ</name>
    <name xml:lang="pl">Programowanie</name>
    <name xml:lang="pt">Desenvolvimento</name>
    <name xml:lang="pt_BR">Desenvolvimento</name>
    <name xml:lang="ro">Dezvoltare</name>
    <name xml:lang="ru">Программирование</name>
    <name xml:lang="si">ප්‍රවර්ධනය</name>
    <name xml:lang="sk">Vývoj</name>
    <name xml:lang="sl">Razvoj</name>
    <name xml:lang="sq">Zhvillim</name>
    <name xml:lang="sr">Развој</name>
    <name xml:lang="sr@Latn">Razvoj</name>
    <name xml:lang="sr@latin">Razvoj</name>
    <name xml:lang="sv">Utveckling</name>
    <name xml:lang="ta">உருவாக்கம்</name>
    <name xml:lang="te">అభివృద్ధి</name>
    <name xml:lang="tg">Барноманависӣ</name>
    <name xml:lang="th">การพัฒนาซอฟต์แวร์</name>
    <name xml:lang="tr">Geliştirme</name>
    <name xml:lang="uk">Програмування</name>
    <name xml:lang="ur">تمیرات</name>
    <name xml:lang="vi">Phát triển</name>
    <name xml:lang="zh">程式開發</name>
    <name xml:lang="zh_CN">开发</name>
    <name xml:lang="zh_TW">程式開發</name>
    <name xml:lang="zu">Intuthuko</name>
    <description>Packages which provide functionality for developing and building appl</description>
    <description xml:lang="ar">تقدم الحزم امكانيات تطوير و بناء تطبيقات.</description>
    <description xml:lang="bal">بستگانی که په پیشروگ و شرکنگ کارمرزی عمل آرنت.</description>
    <description xml:lang="bg">Пакети предоставящи функционалност необходима за разраб</description>
    <description xml:lang="bn">অ্যাপ্লিকেশন ডিভেলপমেন্ট ও বিল্ড করতে সহায়ক প্যাকেজ।</description>
    <description xml:lang="bs">Paketi koji pružaju funkcionalnosti za razvoj i izgradn</description>
    <description xml:lang="ca">Paquets que proporcionen funcionalitats per a desenvolu</description>
    <description xml:lang="cs">Balíčky poskytující podporu pro vývoj a kompilaci aplik</description>
    <description xml:lang="da">Pakker som giver funktionalitet for udvikling og bygnin</description>
    <description xml:lang="de">Pakete, die Funktionalität für die Entwicklung und die </description>
    <description xml:lang="de_CH">Pakete, die Funktionalität für die Entwicklung und d</description>
    <description xml:lang="el">Πακέτα τα οποία προσφέρουν λειτουργικότητα για ανάπτυξη</description>
    <description xml:lang="en_GB">Packages which provide functionality for developing </description>
    <description xml:lang="es">Paquetes que proveen la funcionalidad para desarrollar </description>
    <description xml:lang="et">Rakenduste arendamiseks ja kompileerimiseks vajalik fun</description>
    <description xml:lang="fi">Paketit, jotka tarjoavat tuen sovellusten kehittämisell</description>
    <description xml:lang="fr">Packages qui offrent des fonctions de développement et </description>
    <description xml:lang="he">חבילות המאפשרות פיתוח ובניית תוכנה.</description>
    <description xml:lang="hr">Paketi koji pružaju funkcionalnosti za razvoj i izgradn</description>
    <description xml:lang="hu">Alkalmazások fejlesztésére szolgáló csomagok</description>
    <description xml:lang="ia">Pacchettos que provide functionalitate pro disveloppar</description>
    <description xml:lang="id">Paket yang menyediakan fungsionalitas untuk mengembangk</description>
    <description xml:lang="is">Pakkar sem innihalda virkni til að þróa og smíða hugbún</description>
    <description xml:lang="it">Pacchetti che forniscono funzionalità per sviluppare e </description>
    <description xml:lang="ja">アプリケーションの開発および構築機能を提供するパッケージです。</description>
    <description xml:lang="ko">이 패키지는 응용 프로그램을 개발하기 위한 기능을 제공합니다.</description>
    <description xml:lang="lv">Pakotnes, kas nodrošina programmu izstrādes procesu.</description>
    <description xml:lang="ms">Pakej yang menyediakan fungsi untuk membangun dan memb</description>
    <description xml:lang="nb">Pakker som gir funksjonalitet for utvikling og bygging </description>
    <description xml:lang="nl">Pakketten die functionaliteit bieden voor het ontwikkel</description>
    <description xml:lang="pl">Pakiety dostarczające funkcjonalność tworzenia i budowa</description>
    <description xml:lang="pt">Pacotes que oferecem funcionalidades para desenvolver e</description>
    <description xml:lang="pt_BR">Pacotes que oferecem funcionalidades para desenvolve</description>
    <description xml:lang="ro">Pachete care oferă funcţionalităţi pentru dezvoltarea ş</description>
    <description xml:lang="ru">Пакеты, предоставляющие возможность разработки и сборки</description>
    <description xml:lang="sk">Balíčky poskytujúce funkcionalitu pre vývoj a zostavova</description>
    <description xml:lang="sl">Paketi za razvoj programja.</description>
    <description xml:lang="sq">Paketa që ofrojnë funksione për zhvillim dhe ndërtim zb</description>
    <description xml:lang="sr">Пакети који пружају функционалност за развој и изградњу</description>
    <description xml:lang="sr@Latn">Paketi koji pružaju funkcionalnost za razvoj i i</description>
    <description xml:lang="sr@latin">Paketi koji pružaju funkcionalnost za razvoj i i</description>
    <description xml:lang="sv">Paket som tillhandahåller funktionalitet för utveckling</description>
    <description xml:lang="tg">Барномаҳои барноманависӣ ва эҷодкунии худи барномаҳо</description>
    <description xml:lang="th">ชุดแพกเกจที่ใช้สำหรับการพัฒนาและสร้างโปรแกรมประยุกต์</description>
    <description xml:lang="tr">Uygulama geliştirmek ve oluşturmak için gerekli işlevle</description>
    <description xml:lang="uk">Пакунки, що надають можливість розробляти та збирати пр</description>
    <description xml:lang="zh">提供開發及建立應用程式功能的套件</description>
    <description xml:lang="zh_CN">用于开发和构建应用程序的软件包。</description>
    <description xml:lang="zh_TW">提供開發及建立應用程式功能的套件</description>
    <display_order>90</display_order>
    <grouplist>
      <groupid>additional-devel</groupid>
      <groupid>development</groupid>
      <groupid>platform-devel</groupid>
    </grouplist>
  </category>
  <category>
    <id>servers</id>
    <name>Servers</name>
    <name xml:lang="af">Bedieners</name>
    <name xml:lang="am">ሰርቨሮች</name>
    <name xml:lang="ar">خوادم</name>
    <name xml:lang="as">চাৰ্ভাৰসমূহ</name>
    <name xml:lang="bal">سروران</name>
    <name xml:lang="be">Паслужнікі</name>
    <name xml:lang="bg">Сървъри</name>
    <name xml:lang="bn">সার্ভার</name>
    <name xml:lang="bs">Poslužitelji</name>
    <name xml:lang="ca">Servidors</name>
    <name xml:lang="cs">Servery</name>
    <name xml:lang="cy">Gweinyddion</name>
    <name xml:lang="da">Servere</name>
    <name xml:lang="de">Server</name>
    <name xml:lang="de_CH">Server</name>
    <name xml:lang="el">Εξυπηρετητές</name>
    <name xml:lang="en_GB">Servers</name>
    <name xml:lang="es">Servidores</name>
    <name xml:lang="et">Serverid</name>
    <name xml:lang="eu">Zerbitzariak</name>
    <name xml:lang="fa">کارگزارها</name>
    <name xml:lang="fa_IR">کارگزارها</name>
    <name xml:lang="fi">Palvelimet</name>
    <name xml:lang="fr">Serveurs</name>
    <name xml:lang="gl">Servidores</name>
    <name xml:lang="gu">સર્વરો</name>
    <name xml:lang="he">שרתים</name>
    <name xml:lang="hi">सर्वर</name>
    <name xml:lang="hr">Poslužitelji</name>
    <name xml:lang="hu">Kiszolgálók</name>
    <name xml:lang="hy">Սերվերներ</name>
    <name xml:lang="ia">Servitores</name>
    <name xml:lang="id">Server</name>
    <name xml:lang="ilo">Serbidor</name>
    <name xml:lang="is">Þjónar</name>
    <name xml:lang="it">Servers</name>
    <name xml:lang="ja">サーバー</name>
    <name xml:lang="ka">სერვერები</name>
    <name xml:lang="kn">ಪೂರೈಕೆಗಣಕಗಳು</name>
    <name xml:lang="ko">서버</name>
    <name xml:lang="lv">Serveri</name>
    <name xml:lang="mai">सर्वर</name>
    <name xml:lang="mk">Сервери</name>
    <name xml:lang="ml">സര്‍വറുകള്‍</name>
    <name xml:lang="mr">सेवक</name>
    <name xml:lang="ms">Pelayan-Pelayan</name>
    <name xml:lang="nb">Tjenere</name>
    <name xml:lang="ne">सर्भर</name>
    <name xml:lang="nl">Servers</name>
    <name xml:lang="no">Tjenere</name>
    <name xml:lang="nso">Baabi</name>
    <name xml:lang="or">ସେବକ ମାନେ</name>
    <name xml:lang="pa">ਸਰਵਰ</name>
    <name xml:lang="pl">Serwery</name>
    <name xml:lang="pt">Servidores</name>
    <name xml:lang="pt_BR">Servidores</name>
    <name xml:lang="ro">Servere</name>
    <name xml:lang="ru">Серверы</name>
    <name xml:lang="si">සේවා දායකයන්</name>
    <name xml:lang="sk">Servery</name>
    <name xml:lang="sl">Strežniki</name>
    <name xml:lang="sq">Shërbyesa</name>
    <name xml:lang="sr">Сервери</name>
    <name xml:lang="sr@Latn">Serveri</name>
    <name xml:lang="sr@latin">Serveri</name>
    <name xml:lang="sv">Servrar</name>
    <name xml:lang="ta">சேவையகங்கள்</name>
    <name xml:lang="te">సేవికలు</name>
    <name xml:lang="tg">Хидматгоҳҳо</name>
    <name xml:lang="th">ตัวให้บริการ</name>
    <name xml:lang="tr">Sunucular</name>
    <name xml:lang="uk">Сервери</name>
    <name xml:lang="ur">سرورز</name>
    <name xml:lang="vi">Server</name>
    <name xml:lang="zh">伺服器</name>
    <name xml:lang="zh_CN">服务器</name>
    <name xml:lang="zh_TW">伺服器</name>
    <name xml:lang="zu">Abalekeleli</name>
    <description>Software used for running network servers</description>
    <description xml:lang="ar">يستخدم البرنامج لادارة خادمات الشبكة</description>
    <description xml:lang="as">নেটৱাৰ্ক চাৰ্ভাৰ চলাব'লৈ চফ্টৱেৰ</description>
    <description xml:lang="bal">برنامه ای که په اجرای سروران شبکه ای استفاده بیت.</description>
    <description xml:lang="bg">Софтуер за пускане на мрежови услуги</description>
    <description xml:lang="bn">নেটওয়ার্ক সার্ভার সঞ্চালনের উদ্দেশ্যে সফ্টওয়্যার</description>
    <description xml:lang="bs">Softver koji se upotrebljava za pokretanje mrežnih posl</description>
    <description xml:lang="ca">Programari emprat per executar servidors de xarxa</description>
    <description xml:lang="cs">Software používaný pro běh síťových serverů</description>
    <description xml:lang="da">Programmel til brug for at køre netværksservere</description>
    <description xml:lang="de">Software, die für den Betrieb von Netzwerkservern verwe</description>
    <description xml:lang="de_CH">Software, die für den Betrieb von Netzwerkservern ve</description>
    <description xml:lang="el">Λογισμικό που χρησιμοποιείται για εκτέλεση δικτυακών εξ</description>
    <description xml:lang="en_GB">Software used for running network servers</description>
    <description xml:lang="es">Programas usados para correr servidores de red</description>
    <description xml:lang="et">Võrguserverite töölepanemiseks vajalik tarkvara</description>
    <description xml:lang="fi">Verkkopalvelimina käytettäviä ohjelmistoja</description>
    <description xml:lang="fr">Logiciels utilisés pour serveurs de réseau actifs</description>
    <description xml:lang="gu">નેટવર્ક સર્વરો ચલાવવા માટે વપરાતું સોફ્ટવેર</description>
    <description xml:lang="he">תוכנה המשמשת להרצת שרתי רשת</description>
    <description xml:lang="hi">संजाल सर्वर के लिये प्रयोग के लिये सॉफ्टवेयर</description>
    <description xml:lang="hr">Softver koji se upotrebljava za pokretanje mrežnih posl</description>
    <description xml:lang="hu">Hálózati kiszolgálók futtatásához szükséges szoftverek</description>
    <description xml:lang="hy">Ծրագրեր՝ ցանցային սերվերներ աշխատեցնելու համար</description>
    <description xml:lang="ia">Software usate pro exequer servitores de rete</description>
    <description xml:lang="id">Perangkat lunak yang dipergunakan untuk menjalankan ser</description>
    <description xml:lang="is">Hugbúnaður til að keyra netþjóna</description>
    <description xml:lang="it">Software usato per eseguire server di rete</description>
    <description xml:lang="ja">ネットワークサーバーの稼動に使用するソフトウェア</description>
    <description xml:lang="kn">ಜಾಲ ಪೂರೈಕೆಗಣಕಗಳನ್ನು ಕಾರ್ಯಗತಗೊಳಿಸಲು ಬಳಸುವ ತಂತ್ರಾಂಶ</description>
    <description xml:lang="ko">네트워크 서버를 실행하기 위해 사용되는 소프트웨어</description>
    <description xml:lang="lv">Programmas, kas tiek izmantotas tīkla serveriem</description>
    <description xml:lang="mai">संजाल सर्वर क' लेल प्रयोग क' लेल साफ्टवेयर</description>
    <description xml:lang="mr">संजाळ सर्व्हर चालवण्यासाठी वापरले जाणारे सॉफ्टवेअर</description>
    <description xml:lang="ms">Perisian digunakan untuk melaksanakan pelayan rangkaian</description>
    <description xml:lang="nb">Programvare som brukes for å kjøre nettverkstjenere</description>
    <description xml:lang="ne">सञ्जाल सर्भर चलाउनका लागि प्रयोग गरिने सफ्टवेयर</description>
    <description xml:lang="nl">Software gebruikt voor het uitvoeren van netwerk-server</description>
    <description xml:lang="or">ନେଟୱାର୍କ ସେବକକୁ ଚଳାଇବା ପାଇଁ ବ୍ଯବହୃତ ସଫଟୱେର</description>
    <description xml:lang="pa">ਨੈੱਟਵਰਕ ਸਰਵਰ ਚਲਾਉਣ ਲਈ ਲੋੜੀਦੇ ਸਾਫਟਵੇਅਰ</description>
    <description xml:lang="pl">Oprogramowanie używane do uruchamiania serwerów sieciow</description>
    <description xml:lang="pt">Aplicações usadas para executar servidores de rede</description>
    <description xml:lang="pt_BR">Aplicações usadas para executar servidores de rede</description>
    <description xml:lang="ro">Programe folosite pentru a rula servere de reţea</description>
    <description xml:lang="ru">Программы, используемые для запуска сетевых серверов</description>
    <description xml:lang="sk">Softvér používaný na beh sieťových serverov</description>
    <description xml:lang="sl">Programje za delo z omrežnimi strežniki</description>
    <description xml:lang="sq">Programe të përdorur për xhirim shërbyesash rrjeti</description>
    <description xml:lang="sr">Софтвер који се користи за извршавање мрежних сервера</description>
    <description xml:lang="sr@Latn">Softver koji se koristi za izvršavanje mrežnih ser</description>
    <description xml:lang="sr@latin">Softver koji se koristi za izvršavanje mrežnih se</description>
    <description xml:lang="sv">Mjukvara som används för körande nätverksservrar</description>
    <description xml:lang="ta">பிணைய சேவையகங்களை இயக்க பயன்படுத்தப்பட்ட மென்பொருள்</description>
    <description xml:lang="te">నెట్వర్క్‍ సేవికలను నడుపుటకు ఉపయోగపడే సాఫ్ట్‍‌వేర్</description>
    <description xml:lang="tg">Нармафзори истифодашудаи бо хидматгоҳҳои Интернет</description>
    <description xml:lang="th">โปรแกรมสำหรับให้บริการในเครือข่าย</description>
    <description xml:lang="tr">Ağ servislerini çalıştırmak için kullanılan yazılımlar</description>
    <description xml:lang="uk">Програми, що використовуються для запуску мережних серв</description>
    <description xml:lang="zh">用於運行網路伺服器的軟體</description>
    <description xml:lang="zh_CN">用于运行网络服务器的软件</description>
    <description xml:lang="zh_TW">用於運行網路伺服器的軟體</description>
    <display_order>20</display_order>
    <grouplist>
      <groupid>file-server</groupid>
      <groupid>mail-server</groupid>
      <groupid>network-server</groupid>
    </grouplist>
  </category>
  <category>
    <id>base-system</id>
    <name>System</name>
    <name xml:lang="af">Rekenaar</name>
    <name xml:lang="am">ሲስተም</name>
    <name xml:lang="ar">نظام</name>
    <name xml:lang="as">চিস্টেম</name>
    <name xml:lang="be">Сытэма</name>
    <name xml:lang="bn">সিস্টেম</name>
    <name xml:lang="bs">Sustav</name>
    <name xml:lang="ca">Sistema</name>
    <name xml:lang="cs">Systém</name>
    <name xml:lang="cy">System</name>
    <name xml:lang="de">System</name>
    <name xml:lang="de_CH">System</name>
    <name xml:lang="en_GB">System</name>
    <name xml:lang="es">Sistema</name>
    <name xml:lang="eu">Sistema</name>
    <name xml:lang="fa">سیستم</name>
    <name xml:lang="fa_IR">سیستم</name>
    <name xml:lang="fi">Järjestelmä</name>
    <name xml:lang="fr">Système</name>
    <name xml:lang="gl">Sistema</name>
    <name xml:lang="gu">સિસ્ટમ</name>
    <name xml:lang="hi">तंत्र</name>
    <name xml:lang="hr">Sustav</name>
    <name xml:lang="hu">Rendszer</name>
    <name xml:lang="hy">Համակարգ</name>
    <name xml:lang="id">Sistem</name>
    <name xml:lang="ilo">Sistema</name>
    <name xml:lang="it">Sistema</name>
    <name xml:lang="ja">システム</name>
    <name xml:lang="ka">სისტემა</name>
    <name xml:lang="kn">ವ್ಯವಸ್ಥೆ</name>
    <name xml:lang="ko">시스템</name>
    <name xml:lang="lv">Sistēma</name>
    <name xml:lang="mk">Систем</name>
    <name xml:lang="ml">സിസ്റ്റം</name>
    <name xml:lang="mr">प्रणाली</name>
    <name xml:lang="no">System</name>
    <name xml:lang="nso">Tshepedišo</name>
    <name xml:lang="or">ତନ୍ତ୍ର</name>
    <name xml:lang="pa">ਸਿਸਟਮ</name>
    <name xml:lang="pl">System</name>
    <name xml:lang="pt">Sistema </name>
    <name xml:lang="pt_BR">Sistema </name>
    <name xml:lang="ru">Система</name>
    <name xml:lang="si">පද්ධතිය</name>
    <name xml:lang="sl">Sistem</name>
    <name xml:lang="sq">Sistem</name>
    <name xml:lang="ta">கணினி</name>
    <name xml:lang="te">వ్యవస్థ</name>
    <name xml:lang="th">ระบบ</name>
    <name xml:lang="tr">Sistem</name>
    <name xml:lang="uk">Система</name>
    <name xml:lang="ur">نظام</name>
    <name xml:lang="vi">Hệ thống</name>
    <name xml:lang="zh">系統</name>
    <name xml:lang="zh_CN">系统</name>
    <name xml:lang="zh_TW">系統</name>
    <name xml:lang="zu">Isistimu</name>
    <description>Core system components.</description>
    <description xml:lang="as">চিস্টেমৰ মূখ্য উপাদানসমূহ।</description>
    <description xml:lang="bn">সিস্টেমের কোর সামগ্রী।</description>
    <description xml:lang="cs">Základní komponenty systému.</description>
    <description xml:lang="de">Zentrale Systemkomponenten.</description>
    <description xml:lang="de_CH">Zentrale Systemkomponenten.</description>
    <description xml:lang="es">Componentes de sistema Core</description>
    <description xml:lang="fr">Composants du système de base.</description>
    <description xml:lang="gu">કોર સિસ્ટમ સાધનો.</description>
    <description xml:lang="hi">प्रधान सिस्टम घटक</description>
    <description xml:lang="it">Componenti di base del sistema.</description>
    <description xml:lang="ja">コアシステムコンポーネント。</description>
    <description xml:lang="kn">ಪ್ರಮುಖ ವ್ಯವಸ್ಥೆಯ ಘಟಕಗಳು.</description>
    <description xml:lang="ko">핵심 시스템 콤포넌트.</description>
    <description xml:lang="ml">കോര്‍ സിസ്റ്റം ഘടകങ്ങള്‍.</description>
    <description xml:lang="mr">कोर प्रणाली घटके.</description>
    <description xml:lang="or">ମୂଖ୍ୟ ତନ୍ତ୍ର ଉପାଦାନଗୁଡ଼ିକ।</description>
    <description xml:lang="pa">ਮੁੱਖ ਸਿਸਟਮ ਹਿੱਸੇ।</description>
    <description xml:lang="pl">Główne składniki systemu.</description>
    <description xml:lang="pt">Componentes de sistema central</description>
    <description xml:lang="pt_BR">Componentes de sistema central</description>
    <description xml:lang="ru">Основные компоненты системы.</description>
    <description xml:lang="sv">Grundläggande systemkomponenter.</description>
    <description xml:lang="ta">உள்ளீடு கணினி ஆக்கக்கூறுகள்.</description>
    <description xml:lang="te">కోర్ సిస్టమ్ మూలకములు.</description>
    <description xml:lang="uk">Основні компонент систем.</description>
    <description xml:lang="zh">核心系統元件。</description>
    <description xml:lang="zh_CN">核新系统组件。</description>
    <description xml:lang="zh_TW">核心系統元件。</description>
    <display_order>10</display_order>
    <grouplist>
      <groupid>dial-up</groupid>
      <groupid>hardware-monitoring</groupid>
      <groupid>infiniband</groupid>
      <groupid>large-systems</groupid>
      <groupid>legacy-unix</groupid>
      <groupid>mainframe-access</groupid>
      <groupid>network-tools</groupid>
      <groupid>performance</groupid>
      <groupid>scientific</groupid>
      <groupid>security-tools</groupid>
      <groupid>smart-card</groupid>
      <groupid>standard</groupid>
    </grouplist>
  </category>
  <category>
    <id>desktops</id>
    <name>Desktops</name>
    <name xml:lang="af">Werkskerms</name>
    <name xml:lang="am">ዴስክቶፖች</name>
    <name xml:lang="ar">أسطح المكتب</name>
    <name xml:lang="as">ডেস্কটপ</name>
    <name xml:lang="be">Стальцы</name>
    <name xml:lang="bn">ডেস্কটপ</name>
    <name xml:lang="bs">Radne površine</name>
    <name xml:lang="ca">Escriptoris</name>
    <name xml:lang="cs">Pracovní prostředí</name>
    <name xml:lang="cy">Penbyrddau</name>
    <name xml:lang="de">Desktops</name>
    <name xml:lang="de_CH">Desktops</name>
    <name xml:lang="en_GB">Desktops</name>
    <name xml:lang="es">Escritorios</name>
    <name xml:lang="eu">Mahaigainak</name>
    <name xml:lang="fa">محیط‌های رومیزی</name>
    <name xml:lang="fa_IR">محیط‌های رومیزی</name>
    <name xml:lang="fi">Työpöydät</name>
    <name xml:lang="fr">Bureaux</name>
    <name xml:lang="gl">Escritorios</name>
    <name xml:lang="gu">ડેસ્કટોપો</name>
    <name xml:lang="hi">डेस्कटॉप</name>
    <name xml:lang="hr">Radne površine</name>
    <name xml:lang="hu">Munkaasztalok</name>
    <name xml:lang="hy">Դեսքթոփներ</name>
    <name xml:lang="ia">Scriptorios</name>
    <name xml:lang="id">Desktop</name>
    <name xml:lang="ilo">Eskritorio</name>
    <name xml:lang="it">Desktop</name>
    <name xml:lang="ja">デスクトップ</name>
    <name xml:lang="ka">სამუშაო გარემოები</name>
    <name xml:lang="kn">ಗಣಕತೆರೆಗಳು</name>
    <name xml:lang="ko">데스크탑</name>
    <name xml:lang="lv">Darbvirsmas</name>
    <name xml:lang="mk">Графички работни околини</name>
    <name xml:lang="ml">പണിയിടങ്ങള്‍</name>
    <name xml:lang="mr">डेस्कटॉप्स्</name>
    <name xml:lang="no">Skrivebord</name>
    <name xml:lang="nso">Diteseke</name>
    <name xml:lang="or">ଡେସ୍କଟପ୍ ମାନ</name>
    <name xml:lang="pa">ਡੈਸਕਟਾਪ</name>
    <name xml:lang="pl">Środowiska graficzne</name>
    <name xml:lang="pt">Desktops</name>
    <name xml:lang="pt_BR">Desktops</name>
    <name xml:lang="ru">Рабочие столы</name>
    <name xml:lang="si">Desktops</name>
    <name xml:lang="sl">Namizja</name>
    <name xml:lang="sq">Desktopë</name>
    <name xml:lang="sv">Skrivbord</name>
    <name xml:lang="ta">பணிமேடைகள்</name>
    <name xml:lang="te">డెస్కుటాప్స్</name>
    <name xml:lang="th">เดสก์ท็อป</name>
    <name xml:lang="tr">Masaüstleri</name>
    <name xml:lang="uk">Робочі середовища</name>
    <name xml:lang="ur">ڈيسک ٹاپس</name>
    <name xml:lang="vi">Desktop</name>
    <name xml:lang="zh">桌面環境</name>
    <name xml:lang="zh_CN">桌面</name>
    <name xml:lang="zh_TW">桌面環境</name>
    <name xml:lang="zu">Amadesktop</name>
    <description>Desktops and thin clients.</description>
    <description xml:lang="as">ডেস্কটপ আৰু ক্ষীণ ক্লাএন্ট।</description>
    <description xml:lang="bn">ডেস্কটপ ও থিন ক্লায়েন্ট।</description>
    <description xml:lang="cs">Pracovní prostředí a tencí klienti.</description>
    <description xml:lang="de">Desktops und Thin-Clients.</description>
    <description xml:lang="de_CH">Desktops und Thin-Clients.</description>
    <description xml:lang="es">Escritorios y clientes livianos.</description>
    <description xml:lang="fr">Bureaux et clients légers.</description>
    <description xml:lang="gu">ડેસ્કટોપ અને પાતળા ક્લાયન્ટો.</description>
    <description xml:lang="hi">डेस्कटॉप और थिन क्लाइंट</description>
    <description xml:lang="ia">Scriptorios e clientes legier.</description>
    <description xml:lang="it">Desktop e thin client.</description>
    <description xml:lang="ja">デスクトップとシンクライアント。</description>
    <description xml:lang="kn">ಗಣಕತೆರೆಗಳು ಹಾಗು ಥಿನ್‌ ಕ್ಲೈಂಟ್‌ಗಳು.</description>
    <description xml:lang="ko">데스크탑과 씬 클라이언트.</description>
    <description xml:lang="ml">ഡസ്ക്ടോപ്പുകളും ഥിന്‍ ക്ലയന്റുകളും്റുകള്‍</description>
    <description xml:lang="mr">डेस्कटॉप्स् व थीन क्लाएंट्स्.</description>
    <description xml:lang="or">ଡେସ୍କଟପ ଏବଂ ଥିନ କ୍ଲାଏଣ୍ଟମାନ।</description>
    <description xml:lang="pa">ਡੈਸਕਟਾਪ ਅਤੇ ਥਿੰਨ ਕਲਾਈਂਟ</description>
    <description xml:lang="pl">Środowiska graficzne i ciency klienci.</description>
    <description xml:lang="pt">Ambientes de Trabalho e clientes finos</description>
    <description xml:lang="pt_BR">Ambientes de Trabalho e clientes finos</description>
    <description xml:lang="ru">Рабочие столы и ?тонкие? клиенты.</description>
    <description xml:lang="sv">Skrivbord och tunna klienter.</description>
    <description xml:lang="ta">பணிமேடை மற்றும் ஒல்லியான வாடிக்கையாளர்கள்.</description>
    <description xml:lang="te">డెస్కుటాప్స్ మరియు తిన్ క్లైంట్లు.</description>
    <description xml:lang="uk">Робочі станції та «тонкі» клієнти.</description>
    <description xml:lang="zh">桌面環境與 thin clinet。</description>
    <description xml:lang="zh_CN">桌面和瘦客户端。</description>
    <description xml:lang="zh_TW">桌面環境與 thin clinet。</description>
    <display_order>70</display_order>
    <grouplist>
      <groupid>fonts</groupid>
      <groupid>gnome-desktop</groupid>
    </grouplist>
  </category>
  <environment>
    <id>custom-environment</id>
    <name>Custom Operating System</name>
    <name xml:lang="de">Benutzerdefiniertes Betriebssystem</name>
    <name xml:lang="es">Sistema operativo personalizado</name>
    <name xml:lang="fr">Custom Operating System</name>
    <name xml:lang="it">Sistema operativo personalizzato</name>
    <name xml:lang="ja">カスタムオペレーティングシステム</name>
    <name xml:lang="ko">사용자 정의 운영 체제</name>
    <name xml:lang="pt">Sistema Operacional Personalizado</name>
    <name xml:lang="pt_BR">Sistema Operacional Personalizado</name>
    <name xml:lang="ru">Пользовательская операционная система</name>
    <name xml:lang="zh_CN">定制操作系统</name>
    <name xml:lang="zh_TW">定制操作系統</name>
    <description>Basic building block for a custom CentOS system.</description>
    <display_order>99</display_order>
    <grouplist>
      <groupid>core</groupid>
    </grouplist>
    <optionlist>
      <groupid>guest-agents</groupid>
      <groupid>standard</groupid>
    </optionlist>
  </environment>
  <environment>
    <id>minimal-environment</id>
    <name>Minimal Install</name>
    <name xml:lang="as">নূন্যতম ইনস্টল</name>
    <name xml:lang="cs">Minimální instalace</name>
    <name xml:lang="de">Minimale Installation</name>
    <name xml:lang="de_CH">Minimale Installation</name>
    <name xml:lang="es">Instalación mínima</name>
    <name xml:lang="fr">Installation minimale</name>
    <name xml:lang="gu">ન્યૂનતમ સ્થાપન</name>
    <name xml:lang="hi">न्यूनतम संस्थापन</name>
    <name xml:lang="it">Installazione minima</name>
    <name xml:lang="ja">最小限のインストール</name>
    <name xml:lang="kn">ಕನಿಷ್ಟ ಅನುಸ್ಥಾಪನೆ</name>
    <name xml:lang="ko">최소 설치</name>
    <name xml:lang="ml">ഏറ്റവും കുറഞ്ഞ ഇന്‍സ്റ്റോള്‍</name>
    <name xml:lang="mr">किमान इंस्टॉल</name>
    <name xml:lang="or">ସର୍ବନିମ୍ନ ସ୍ଥାପନ</name>
    <name xml:lang="pa">ਘੱਟ ਤੋਂ ਘੱਟ ਇੰਸਟਾਲ</name>
    <name xml:lang="pl">Minimalna instalacja</name>
    <name xml:lang="pt">Instalações Mínimas</name>
    <name xml:lang="pt_BR">Instalações Mínimas</name>
    <name xml:lang="ru">Минимальная установка</name>
    <name xml:lang="ta">குறைந்தபட்ச நிறுவல்</name>
    <name xml:lang="te">కనీసపు సంస్థాపన</name>
    <name xml:lang="uk">Мінімальна система</name>
    <name xml:lang="zh">最小型安裝</name>
    <name xml:lang="zh_CN">最小安装</name>
    <name xml:lang="zh_TW">最小型安裝</name>
    <description>Basic functionality.</description>
    <description xml:lang="as">মৌলি কাৰ্য্যকৰীতা।</description>
    <description xml:lang="cs">Základní funkcionalita.</description>
    <description xml:lang="de">Grundlegende Funktionalität.</description>
    <description xml:lang="de_CH">Grundlegende Funktionalität.</description>
    <description xml:lang="es">Funcionalidad básica.</description>
    <description xml:lang="fr">Fonctionnalité de base.</description>
    <description xml:lang="gu">મૂળભૂત વિધેય.</description>
    <description xml:lang="hi">मौलिक प्रकार्यात्मकता.</description>
    <description xml:lang="it">Funzione di base.</description>
    <description xml:lang="ja">基本的な機能です。</description>
    <description xml:lang="kn">ಮೂಲಭೂತ ಕ್ರಿಯಾಶೀಲತೆ.</description>
    <description xml:lang="ko">기본적인 기능입니다.</description>
    <description xml:lang="ml">അടിസ്ഥാന പ്രവൃത്തിവിശേഷണം.</description>
    <description xml:lang="mr">मूळ कार्यक्षमता.</description>
    <description xml:lang="or">ସାଧାରଣ କାର୍ଯ୍ୟକାରିତା।</description>
    <description xml:lang="pa">ਮੁੱਢਲੀ ਕਾਰਜਸ਼ੀਲਤਾ।</description>
    <description xml:lang="pl">Podstawowa funkcjonalność.</description>
    <description xml:lang="pt">Função básica</description>
    <description xml:lang="pt_BR">Função básica</description>
    <description xml:lang="ru">Базовая функциональность.</description>
    <description xml:lang="ta">அடிப்படை செயலம்சம்.</description>
    <description xml:lang="te">ప్రాథమిక ఫంక్షనాలిటి.</description>
    <description xml:lang="uk">Основні можливості.</description>
    <description xml:lang="zh">基本功能。</description>
    <description xml:lang="zh_CN">基本功能。</description>
    <description xml:lang="zh_TW">基本功能。</description>
    <display_order>3</display_order>
    <grouplist>
      <groupid>core</groupid>
    </grouplist>
    <optionlist>
      <groupid>guest-agents</groupid>
      <groupid>standard</groupid>
    </optionlist>
  </environment>
  <environment>
    <id>server-product-environment</id>
    <name>Server</name>
    <name xml:lang="fr">Serveurs</name>
    <name xml:lang="ja">サーバー</name>
    <name xml:lang="ko">서버</name>
    <name xml:lang="zh_CN">服务器</name>
    <description>An integrated, easy-to-manage server.</description>
    <description xml:lang="de">Ein integrierter, einfach zu verwaltender Server.</description>
    <description xml:lang="es">Un servidor integrado y fácil de administrar.</description>
    <description xml:lang="fr">Un serveur intégré, facile à gérer.</description>
    <description xml:lang="it">Un server integrato e facile da gestire.</description>
    <description xml:lang="ja">統合された、管理が容易なサーバーです。</description>
    <description xml:lang="ko">통합되고 관리가 쉬운 서버.</description>
    <description xml:lang="pt">Um servidor integrado e fácil de gerir.</description>
    <description xml:lang="pt_BR">Um servidor integrado e fácil de gerir.</description>
    <description xml:lang="ru">Интегрированный, простой в управлении сервер.</description>
    <description xml:lang="zh_CN">集成的易于管理的服务器。</description>
    <description xml:lang="zh_TW">一個集成，易於管理的服務器。</description>
    <display_order>2</display_order>
    <grouplist>
      <groupid>core</groupid>
      <groupid>hardware-support</groupid>
      <groupid>headless-management</groupid>
      <groupid>server-product</groupid>
      <groupid>standard</groupid>
    </grouplist>
    <optionlist>
      <groupid>debugging</groupid>
      <groupid>dns-server</groupid>
      <groupid>file-server</groupid>
      <groupid>ftp-server</groupid>
      <groupid>gnome-desktop</groupid>
      <groupid>guest-agents</groupid>
      <groupid>hardware-monitoring</groupid>
      <groupid>infiniband</groupid>
      <groupid>mail-server</groupid>
      <groupid>network-file-system-client</groupid>
      <groupid>network-server</groupid>
      <groupid>performance</groupid>
      <groupid>remote-system-management</groupid>
      <groupid>smb-server</groupid>
      <groupid>virtualization-hypervisor</groupid>
      <groupid>web-server</groupid>
    </optionlist>
  </environment>
</comps>"""
BIG_GROUPS = 3
BIG_CATEGORY = 4
BIG_LANGPACK = 0
BIG_ENVIRONMENTS = 3

MODULEMD_FIELDS = [
    "arch",
    "artifacts",
    "context",
    "dependencies",
    "description",
    "name",
    "packages",
    "profiles",
    "pulp_created",
    "stream",
    "version",
]

RPM_MODULEMD_DEFAULTS_DATA = [
    {"module": "nodejs", "stream": "8", "profiles": {"8": ["default"], "10": ["default"]}},
    {"module": "postgresql", "stream": "9.6", "profiles": {"9.6": ["default"]}},
    {"module": "meson", "stream": "master", "profiles": {"master": ["default"]}},
    {
        "module": "ninja",
        "stream": "master",
        "profiles": {"master": ["default"], "development": ["default"]},
    },
]

RPM_MODULEMD_OBSOLETES_DATA = [
    {
        "modified": "2020-05-01T00:00Z",
        "module_name": "postgresql",
        "module_stream": "6",
        "message": "obsoletes with reset",
        "override_previous": "True",
        "module_context": "6c81f848",
        "eol_date": None,
        "obsoleted_by_module_name": None,
        "obsoleted_by_module_stream": None,
    },
    {
        "modified": "2020-05-01T00:00Z",
        "module_name": "nodejs",
        "module_stream": "8",
        "message": "immediate obsoletes",
        "override_previous": None,
        "module_context": "6c81f848",
        "eol_date": None,
        "obsoleted_by_module_name": None,
        "obsoleted_by_module_stream": None,
    },
    {
        "modified": "2020-05-01T00:00Z",
        "module_name": "nodejs",
        "module_stream": "5",
        "message": "obsoleting obsoletes",
        "override_previous": None,
        "module_context": "6c81f848",
        "eol_date": None,
        "obsoleted_by_module_name": "nodejs",
        "obsoleted_by_module_stream": "10",
    },
    {
        "modified": "2020-05-01T00:00Z",
        "module_name": "nodejs",
        "module_stream": "11",
        "message": "obsoletes without context affects all nodejs 11 contexts",
        "override_previous": None,
        "module_context": None,
        "eol_date": None,
        "obsoleted_by_module_name": None,
        "obsoleted_by_module_stream": None,
    },
    {
        "modified": "2020-05-01T00:00Z",
        "module_name": "nodejs",
        "module_stream": "10",
        "message": "eol is in far far future",
        "override_previous": None,
        "module_context": "6c81f848",
        "eol_date": "2990-01-01T00:00Z",
        "obsoleted_by_module_name": None,
        "obsoleted_by_module_stream": None,
    },
    {
        "modified": "2020-05-01T00:00Z",
        "module_name": "meson",
        "module_stream": "master",
        "message": "chain of obsoletes",
        "override_previous": None,
        "module_context": None,
        "eol_date": None,
        "obsoleted_by_module_name": "nodejs",
        "obsoleted_by_module_stream": "5",
    },
]

RPM_MODULEMDS_DATA = [
    {
        "name": "postgresql",
        "stream": "9.6",
        "version": "20180816142114",
        "static_context": None,
        "context": "6c81f848",
        "arch": "x86_64",
        "artifacts": [
            "postgresql-0:9.6.8-1.module_1710+b535a823.x86_64",
            "postgresql-0:9.6.8-1.module_1710+b535a823.src",
            "postgresql-libs-0:9.6.8-1.module_1710+b535a823.x86_64",
            "postgresql-server-0:9.6.8-1.module_1710+b535a823.x86_64",
            "postgresql-test-0:9.6.8-1.module_1710+b535a823.x86_64",
        ],
        "profiles": {
            "client": ["postgresql"],
            "server": ["postgresql-server"],
            "default": ["postgresql-server"],
        },
        "description": "PostgreSQL is an advanced Object-Relational database management system (DBMS). The PostgreSQL server can be found in the postgresql-server sub-package.",
        "dependencies": [
            {"requires": {"platform": ["f29"]}, "buildrequires": {"platform": ["f29"]}}
        ],
    },
    {
        "name": "postgresql",
        "stream": "6",
        "version": "20190329142114",
        "static_context": None,
        "context": "6c81f848",
        "arch": "x86_64",
        "artifacts": [
            "postgresql-0:6.1-1.module_2514+aa9aadc5.x86_64",
            "postgresql-0:6.1-1.module_2514+aa9aadc5.src",
            "postgresql-libs-0:6.1-1.module_2514+aa9aadc5.x86_64",
            "postgresql-server-0:6.1-1.module_2514+aa9aadc5.x86_64",
            "postgresql-test-0:6.1-1.module_2514+aa9aadc5.x86_64",
        ],
        "profiles": {
            "client": ["postgresql"],
            "server": ["postgresql-server"],
            "default": ["postgresql-server"],
        },
        "description": "PostgreSQL is an advanced Object-Relational database management system (DBMS). The PostgreSQL server can be found in the postgresql-server sub-package.",
        "dependencies": [
            {"requires": {"platform": ["f29"]}, "buildrequires": {"platform": ["f29"]}}
        ],
    },
    {
        "name": "nodejs",
        "stream": "8",
        "version": "20180801080000",
        "static_context": None,
        "context": "6c81f848",
        "arch": "x86_64",
        "artifacts": [
            "nodejs-1:8.11.4-1.module_2030+42747d40.x86_64",
            "nodejs-1:8.11.4-1.module_2030+42747d40.src",
            "nodejs-devel-1:8.11.4-1.module_2030+42747d40.x86_64",
            "nodejs-docs-1:8.11.4-1.module_2030+42747d40.noarch",
            "npm-1:8.11.4-1.module_2030+42747d40.x86_64",
        ],
        "profiles": {
            "default": ["nodejs", "npm"],
            "minimal": ["nodejs"],
            "development": ["nodejs", "nodejs-devel", "npm"],
        },
        "description": "Node.js is a platform built on Chrome''s JavaScript runtime for easily building fast, scalable network applications. Node.js uses an event-driven, non-blocking I/O model that makes it lightweight and efficient, perfect for data-intensive real-time applications that run across distributed devices.",
        "dependencies": [
            {"requires": {"platform": ["f29"]}, "buildrequires": {"platform": ["f29"]}}
        ],
    },
    {
        "name": "nodejs",
        "stream": "5",
        "version": "20150811143428",
        "static_context": None,
        "context": "6c81f848",
        "arch": "x86_64",
        "artifacts": [
            "nodejs-1:5.3.1-1.module_2011+41787af0.x86_64",
            "nodejs-1:5.3.1-1.module_2011+41787af0.src",
            "nodejs-devel-1:5.3.1-1.module_2011+41787af0.x86_64",
            "nodejs-docs-1:5.3.1-1.module_2011+41787af0.noarch",
            "npm-1:5.3.1-1.module_2011+41787af0.x86_64",
        ],
        "profiles": {
            "default": ["nodejs", "npm"],
            "minimal": ["nodejs"],
            "development": ["nodejs", "nodejs-devel", "npm"],
        },
        "description": "Node.js is a platform built on Chrome''s JavaScript runtime for easily building fast, scalable network applications. Node.js uses an event-driven, non-blocking I/O model that makes it lightweight and efficient, perfect for data-intensive real-time applications that run across distributed devices.",
        "dependencies": [
            {"requires": {"platform": ["f29"]}, "buildrequires": {"platform": ["f29"]}}
        ],
    },
    {
        "name": "nodejs",
        "stream": "11",
        "version": "20180920144611",
        "static_context": None,
        "context": "6c81f848",
        "arch": "x86_64",
        "artifacts": [
            "nodejs-1:11.0.0-1.module_2311+8d497411.x86_64",
            "nodejs-1:11.0.0-1.module_2311+8d497411.src",
            "nodejs-devel-1:11.0.0-1.module_2311+8d497411.x86_64",
            "nodejs-docs-1:11.0.0-1.module_2311+8d497411.noarch",
            "npm-1:11.0.0-1.module_2311+8d497411.x86_64",
        ],
        "profiles": {
            "default": ["nodejs", "npm"],
            "minimal": ["nodejs"],
            "development": ["nodejs", "nodejs-devel", "npm"],
        },
        "description": "Node.js is a platform built on Chrome''s JavaScript runtime for easily building fast, scalable network applications. Node.js uses an event-driven, non-blocking I/O model that makes it lightweight and efficient, perfect for data-intensive real-time applications that run across distributed devices.",
        "dependencies": [
            {
                "requires": {"platform": ["f29"], "postgresql": [9.6]},
                "buildrequires": {"platform": ["f29"]},
            }
        ],
    },
    {
        "name": "nodejs",
        "stream": "10",
        "version": "20180920144631",
        "static_context": None,
        "context": "6c81f848",
        "arch": "x86_64",
        "artifacts": [
            "nodejs-1:10.11.0-1.module_2200+adbac02b.x86_64",
            "nodejs-1:10.11.0-1.module_2200+adbac02b.src",
            "nodejs-devel-1:10.11.0-1.module_2200+adbac02b.x86_64",
            "nodejs-docs-1:10.11.0-1.module_2200+adbac02b.noarch",
            "npm-1:10.11.0-1.module_2200+adbac02b.x86_64",
        ],
        "profiles": {
            "default": ["nodejs", "npm"],
            "minimal": ["nodejs"],
            "development": ["nodejs", "nodejs-devel", "npm"],
        },
        "description": "Node.js is a platform built on Chrome''s JavaScript runtime for easily building fast, scalable network applications. Node.js uses an event-driven, non-blocking I/O model that makes it lightweight and efficient, perfect for data-intensive real-time applications that run across distributed devices.",
        "dependencies": [
            {"requires": {"platform": ["f29"]}, "buildrequires": {"platform": ["f29"]}}
        ],
    },
    {
        "name": "ninja",
        "stream": "master",
        "version": "20180816141421",
        "static_context": None,
        "context": "6c81f848",
        "arch": "x86_64",
        "artifacts": [
            "ninja-build-0:1.8.2-4.module_1991+4e5efe2f.x86_64",
            "ninja-build-debuginfo-0:1.8.2-4.module_1991+4e5efe2f.x86_64",
            "ninja-build-debugsource-0:1.8.2-4.module_1991+4e5efe2f.x86_64",
            "ninja-build-0:1.8.2-4.module_1991+4e5efe2f.src",
        ],
        "profiles": {"default": ["ninja-build"]},
        "description": "This module includes Ninja build system. Ninja is a small build system with a focus on speed. It differs from other build systems in two major respects: it is designed to have its input files generated by a higher-level build system, and it is designed to run builds as fast as possible.",
        "dependencies": [
            {"requires": {"platform": ["f29"]}, "buildrequires": {"platform": ["f29"]}}
        ],
    },
    {
        "name": "ninja",
        "stream": "legacy",
        "version": "20180816141421",
        "static_context": None,
        "context": "6c81f848",
        "arch": "x86_64",
        "artifacts": [
            "ninja-build-0:1.5.2-1.module_1991+4e5efe2f.x86_64",
            "ninja-build-0:1.5.2-1.module_1991+4e5efe2f.src",
        ],
        "profiles": {"default": ["ninja-build"]},
        "description": "This module includes Ninja build system. Ninja is a small build system with a focus on speed. It differs from other build systems in two major respects: it is designed to have its input files generated by a higher-level build system, and it is designed to run builds as fast as possible.",
        "dependencies": [
            {"requires": {"platform": ["f29"]}, "buildrequires": {"platform": ["f29"]}}
        ],
    },
    {
        "name": "ninja",
        "stream": "development",
        "version": "20180816141421",
        "static_context": None,
        "context": "6c81f848",
        "arch": "x86_64",
        "artifacts": ["ninja-build-0:1.9.2-1.module_1991+4e5efe2f.x86_64"],
        "profiles": {"default": ["ninja-build"]},
        "description": "This module includes Ninja build system. Ninja is a small build system with a focus on speed. It differs from other build systems in two major respects: it is designed to have its input files generated by a higher-level build system, and it is designed to run builds as fast as possible.",
        "dependencies": [
            {"requires": {"platform": ["f29"]}, "buildrequires": {"platform": ["f29"]}}
        ],
    },
    {
        "name": "meson",
        "stream": "master",
        "version": "20180816151613",
        "static_context": None,
        "context": "06d0a27d",
        "arch": "x86_64",
        "artifacts": [
            "meson-0:0.47.1-5.module_1993+7c0a4d1e.noarch",
            "meson-0:0.47.1-5.module_1993+7c0a4d1e.src",
            "meson-doc-0:0.47.1-5.module_1993+7c0a4d1e.noarch",
            "meson-doc-0:0.47.1-5.module_1993+7c0a4d1e.src",
        ],
        "dependencies": [
            {"requires": {"platform": ["f29"], "ninja": []}, "buildrequires": {"platform": ["f29"]}}
        ],
        "profiles": {"default": ["meson"]},
        "description": "Meson is an open source build system meant to be both extremely fast, and, even more importantly, as user friendly as possible.\nThe main design point of Meson is that every moment a developer spends writing or debugging build definitions is a second wasted. So is every second spent waiting for the build system to actually start compiling code.",
    },
    {
        "name": "dwm",
        "stream": "6.0",
        "version": "20180813144159",
        "static_context": None,
        "context": "6c81f848",
        "arch": "x86_64",
        "artifacts": [
            "dwm-0:6.0-1.module_1997+c375c79c.x86_64",
            "dwm-0:6.0-1.module_1997+c375c79c.src",
        ],
        "profiles": {
            "default": ["dwm"],
        },
        "description": "dwm is a dynamic window manager for X.  It manages windows in tiled, monocle, and floating layouts.  All of the layouts can be applied dynamically, optimizing the environment for the application in use and the task performed.",
        "dependencies": [
            {"requires": {"platform": ["f29"]}, "buildrequires": {"platform": ["f29"]}}
        ],
    },
]

BASE_TEST_JSON = """{
    "updated": "2014-09-28 00:00:00",
    "issued": "2014-09-24 00:00:00",
    "id": "RHSA-XXXX:XXXX",
     "pkglist": [
       {
        "packages": [
         {
          "arch": "noarch",
          "epoch": "0",
          "filename": "bear-4.1-1.noarch.rpm",
          "name": "bear",
          "reboot_suggested": false,
          "relogin_suggested": false,
          "restart_suggested": false,
          "release": "1",
          "src": "http://www.fedoraproject.org",
          "sum": "",
          "sum_type": "",
          "version": "4.1"
         }
        ]
       }
     ],
     "severity":  "",
     "description":  "Not available",
     "reboot_suggested":  false,
     "solution":  "Not available",
     "fromstr":  "centos-announce@centos.org"}"""

BEAR_JSON = """{
    "issued":  "2020-03-08 20:04:01",
    "id":  "CEBA-2019--666",
    "type":  "Bug Fix Advisory",
    "release":  "1",
    "version": "1",
    "pkglist": [
        {
            "packages": [
                {
                    "arch": "noarch",
                    "epoch": "0",
                    "filename": "bear-4.1-1.noarch.rpm",
                    "name": "bear",
                    "reboot_suggested": false,
                    "relogin_suggested": false,
                    "restart_suggested": false,
                    "release": "1",
                    "src": "http://www.fedoraproject.org",
                    "sum": "",
                    "sum_type": "",
                    "version": "4.1"
                }
            ]
        }
    ],
    "severity":  "",
    "description":  "Not available",
    "reboot_suggested":  false,
    "updated":  "2020-03-08 20:04:01",
    "solution":  "Not available",
    "fromstr":  "centos-announce@centos.org"
}"""

CAMEL_JSON = """{
    "issued":  "2020-03-08 20:04:01",
    "id":  "CEBA-2019--666",
    "type":  "Bug Fix Advisory",
    "release":  "1",
    "version": "1",
    "pkglist": [
        {
            "packages": [
                {
                    "arch": "noarch",
                    "epoch": "0",
                    "filename": "camel-0.1-1.noarch.rpm",
                    "name": "camel",
                    "reboot_suggested": false,
                    "relogin_suggested": false,
                    "restart_suggested": false,
                    "release": "1",
                    "src": "http://www.fedoraproject.org",
                    "sum": "",
                    "sum_type": "",
                    "version": "0.1"
                }
            ]
        }
    ],
    "severity":  "",
    "description":  "Not available",
    "reboot_suggested":  false,
    "updated":  "2020-03-08 20:04:01",
    "solution":  "Not available",
    "fromstr":  "centos-announce@centos.org"
}"""

CAMEL_BIRD_JSON = """{
    "issued":  "2020-03-08 20:04:01",
    "id":  "CEBA-2019--666",
    "type":  "Bug Fix Advisory",
    "release":  "1",
    "version": "1",
    "pkglist": [
        {
            "packages": [
                {
                    "arch": "noarch",
                    "epoch": "0",
                    "filename": "camel-0.1-1.noarch.rpm",
                    "name": "camel",
                    "reboot_suggested": false,
                    "relogin_suggested": false,
                    "restart_suggested": false,
                    "release": "1",
                    "src": "http://www.fedoraproject.org",
                    "sum": "",
                    "sum_type": "",
                    "version": "0.1"
                },
                {
                    "arch": "noarch",
                    "epoch": "0",
                    "filename": "bird-1.2-3.noarch.rpm",
                    "name": "bird",
                    "reboot_suggested": false,
                    "relogin_suggested": false,
                    "restart_suggested": false,
                    "release": "3",
                    "src": "http://www.fedoraproject.org",
                    "sum": "",
                    "sum_type": "",
                    "version": "1.2"
                }
            ]
        }
    ],
    "severity":  "",
    "description":  "Not available",
    "reboot_suggested":  false,
    "updated":  "2020-03-08 20:04:01",
    "solution":  "Not available",
    "fromstr":  "centos-announce@centos.org"
}"""

CAMEL_BEAR_DOG_JSON = """{
    "issued":  "2020-03-08 20:04:01",
    "id":  "CEBA-2019--666",
    "type":  "Bug Fix Advisory",
    "release":  "1",
    "version": "1",
    "pkglist": [
        {
            "packages": [
                {
                    "arch": "noarch",
                    "epoch": "0",
                    "filename": "camel-0.1-1.noarch.rpm",
                    "name": "camel",
                    "reboot_suggested": false,
                    "relogin_suggested": false,
                    "restart_suggested": false,
                    "release": "1",
                    "src": "http://www.fedoraproject.org",
                    "sum": "",
                    "sum_type": "",
                    "version": "0.1"
                },
                                    {
                    "arch": "noarch",
                    "epoch": "0",
                    "filename": "bear-4.1-1.noarch.rpm",
                    "name": "bear",
                    "reboot_suggested": false,
                    "relogin_suggested": false,
                    "restart_suggested": false,
                    "release": "1",
                    "src": "http://www.fedoraproject.org",
                    "sum": "",
                    "sum_type": "",
                    "version": "4.1"
                },
                {
                    "arch": "noarch",
                    "epoch": "0",
                    "filename": "dog-6.1-6.noarch.rpm",
                    "name": "dog",
                    "reboot_suggested": false,
                    "relogin_suggested": false,
                    "restart_suggested": false,
                    "release": "6",
                    "src": "http://www.fedoraproject.org",
                    "sum": "",
                    "sum_type": "",
                    "version": "6.1"
                }
            ]
        }
    ],
    "severity":  "",
    "description":  "Not available",
    "reboot_suggested":  false,
    "updated":  "2020-03-08 20:04:01",
    "solution":  "Not available",
    "fromstr":  "centos-announce@centos.org"
}"""

CESA_2020_5002 = """{
    "title": "Moderate CentOS curl Security Update",
    "type": "security",
    "description": "",
    "release": "el7",
    "version": "1",
    "severity": "Moderate",
    "status": "final",
    "updated": "2020-11-18 17:30:30",
    "issued": "2020-11-18 17:30:30",
    "pkglist": [
        {
          "packages": [
            {
              "arch": "x86_64",
              "epoch": "0",
              "filename": "curl-7.29.0-59.el7_9.1.x86_64.rpm",
              "release": "59.el7_9.1",
              "name": "curl",
              "sum": "dfc95bdd8057839d4b45153318acb4e09f4da257afee1c57c07781870a68ecef",
              "sum_type": "sha256"
            },
            {
              "arch": "i686",
              "epoch": "0",
              "filename": "libcurl-7.29.0-59.el7_9.1.i686.rpm",
              "release": "59.el7_9.1",
              "name": "libcurl",
              "sum": "3054ca1c0cc8eef5f08ce1d3be56c7a39e97d92361e8bd265bea14d06f590219",
              "sum_type": "sha256"
            },
            {
              "arch": "x86_64",
              "epoch": "0",
              "filename": "libcurl-7.29.0-59.el7_9.1.x86_64.rpm",
              "release": "59.el7_9.1",
              "name": "libcurl",
              "sum": "4ad0b71e3a6468fba1b43ab82fad024415b5296c7b77d1348fb9afa3f828f98e",
              "sum_type": "sha256"
            },
            {
              "arch": "i686",
              "epoch": "0",
              "filename": "libcurl-devel-7.29.0-59.el7_9.1.i686.rpm",
              "release": "59.el7_9.1",
              "name": "libcurl-devel",
              "sum": "7ab4f1b0aa285d3773fdbd8bfc529969ca101a627d3ea88bea1f99a42093e132",
              "sum_type": "sha256"
            },
            {
              "arch": "x86_64",
              "epoch": "0",
              "filename": "libcurl-devel-7.29.0-59.el7_9.1.x86_64.rpm",
              "release": "59.el7_9.1",
              "name": "libcurl-devel",
              "sum": "f92fde3f97c0034135796baa7cd55f87c0550a88ac79adbdcc9c7f64c595614b",
              "sum_type": "sha256"
            }
          ]
        }
        ],
        "id": "TEST-CESA-2020:5002",
        "from": "centos-announce@centos.org",
        "references": [
        {
          "href": "https://access.redhat.com/errata/RHSA-2020:5002",
          "ref_id": "CESA-2020:5002",
          "title": "Moderate CentOS curl Security Update",
          "ref_type": "security"
        },
        {
          "href": "https://lists.centos.org/pipermail/centos-announce/2020-November/035840.html",
          "ref_id": "CESA-2020:5002",
          "title": "Moderate CentOS curl Security Update",
          "ref_type": "security"
        }
    ]
}"""  # noqa

CESA_2020_4910 = """{
    "title": "Important CentOS xorg-x11-server Security Update",
    "type": "security",
    "description": "",
    "release": "el7",
    "version": "1",
    "severity": "Important",
    "status": "final",
    "updated": "2020-11-06 22:19:48",
    "issued": "2020-11-06 22:19:48",
    "pkglist": [
        {
            "packages": [
                {
                  "arch": "x86_64",
                  "epoch": "0",
                  "filename": "xorg-x11-server-Xdmx-1.20.4-12.el7_9.x86_64.rpm",
                  "release": "12.el7_9",
                  "name": "xorg-x11-server-Xdmx",
                  "sum": "0435f345b2b188c76dbb4a538bf0f878834a41e723491df1926231020fd88efd",
                  "sum_type": "sha256"
                },
                {
                  "arch": "x86_64",
                  "epoch": "0",
                  "filename": "xorg-x11-server-Xephyr-1.20.4-12.el7_9.x86_64.rpm",
                  "release": "12.el7_9",
                  "name": "xorg-x11-server-Xephyr",
                  "sum": "2d21d53b305e30b058ca88d8778bda67000a5d52ab320f04b35e63f6a78f2163",
                  "sum_type": "sha256"
                },
                {
                  "arch": "x86_64",
                  "epoch": "0",
                  "filename": "xorg-x11-server-Xnest-1.20.4-12.el7_9.x86_64.rpm",
                  "release": "12.el7_9",
                  "name": "xorg-x11-server-Xnest",
                  "sum": "51fbacc2e26050a7772549f1fe16c46bd8063ea187825ad89b237c34fa9b4250",
                  "sum_type": "sha256"
                },
                {
                  "arch": "x86_64",
                  "epoch": "0",
                  "filename": "xorg-x11-server-Xorg-1.20.4-12.el7_9.x86_64.rpm",
                  "release": "12.el7_9",
                  "name": "xorg-x11-server-Xorg",
                  "sum": "eb89964d5fd40ec94ee8db97a5a14cc8dd6329b83d82ab29ee1a595653ce5223",
                  "sum_type": "sha256"
                },
                {
                  "arch": "x86_64",
                  "epoch": "0",
                  "filename": "xorg-x11-server-Xvfb-1.20.4-12.el7_9.x86_64.rpm",
                  "release": "12.el7_9",
                  "name": "xorg-x11-server-Xvfb",
                  "sum": "ea32b047fba7fd327bf943da2a18413a1ed3e245cc1b077f34d1c8f6048d9813",
                  "sum_type": "sha256"
                },
                {
                  "arch": "x86_64",
                  "epoch": "0",
                  "filename": "xorg-x11-server-Xwayland-1.20.4-12.el7_9.x86_64.rpm",
                  "release": "12.el7_9",
                  "name": "xorg-x11-server-Xwayland",
                  "sum": "4a6ffb39008edd469d4365bb3bf858f5f5f466129eb9e330d978b28866906891",
                  "sum_type": "sha256"
                },
                {
                  "arch": "x86_64",
                  "epoch": "0",
                  "filename": "xorg-x11-server-common-1.20.4-12.el7_9.x86_64.rpm",
                  "release": "12.el7_9",
                  "name": "xorg-x11-server-common",
                  "sum": "339bcf68cb37a454eddff7218aff4153a36bafc0d36e2b5b6bde8311c6f3eed8",
                  "sum_type": "sha256"
                },
                {
                  "arch": "i686",
                  "epoch": "0",
                  "filename": "xorg-x11-server-devel-1.20.4-12.el7_9.i686.rpm",
                  "release": "12.el7_9",
                  "name": "xorg-x11-server-devel",
                  "sum": "55e13fc8624f8a63b785b5194281c38a4670f03113b0ff2b8fc1df1ca473e1e8",
                  "sum_type": "sha256"
                },
                {
                  "arch": "x86_64",
                  "epoch": "0",
                  "filename": "xorg-x11-server-devel-1.20.4-12.el7_9.x86_64.rpm",
                  "release": "12.el7_9",
                  "name": "xorg-x11-server-devel",
                  "sum": "e2dd0c67f3d88a9506f72fcc21ec0af786a377befabac8e1670d3e012d844b06",
                  "sum_type": "sha256"
                },
                {
                  "arch": "noarch",
                  "epoch": "0",
                  "filename": "xorg-x11-server-source-1.20.4-12.el7_9.noarch.rpm",
                  "release": "12.el7_9",
                  "name": "xorg-x11-server-source",
                  "sum": "1baa9cb2d4f8d4300ac333fbc7bc130dce9145c67aea3bd6efa4a0354fc92b6d",
                  "sum_type": "sha256"
                }
            ]
        }
    ],
    "id": "TEST-CESA-2020:4910",
    "from": "centos-announce@centos.org",
    "references": [
        {
          "href": "https://access.redhat.com/errata/RHSA-2020:4910",
          "ref_id": "CESA-2020:4910",
          "title": "Important CentOS xorg-x11-server Security Update",
          "ref_type": "security"
        },
        {
          "href": "https://lists.centos.org/pipermail/centos-cr-announce/2020-November/012889.html",
          "ref_id": "CESA-2020:4910",
          "title": "Important CentOS xorg-x11-server Security Update",
          "ref_type": "security"
        }
    ]
}"""  # noqa
