"""Generate the Cursor layer across TailorKey variants."""

from __future__ import annotations

from glove80.base import (
    KeySpec,
    Layer,
    LayerSpec,
    PatchSpec,
    build_layer_from_spec,
    copy_layer,
)
from glove80.families.tailorkey.alpha_layouts import base_variant_for
from glove80.layouts.common_patches import apply_indices_patch, apply_mac_morphs, command_binding

CURSOR_SPEC = LayerSpec(
    overrides={
        0: KeySpec("&none"),
        1: KeySpec("&none"),
        2: KeySpec("&none"),
        3: KeySpec("&none"),
        4: KeySpec("&none"),
        5: KeySpec("&none"),
        6: KeySpec("&none"),
        7: KeySpec("&none"),
        8: KeySpec("&none"),
        9: KeySpec("&none"),
        10: KeySpec("&none"),
        11: KeySpec("&kp", (KeySpec("ESC"),)),
        12: KeySpec("&kp", (KeySpec("INS"),)),
        13: KeySpec("&kp", (KeySpec("LS", (KeySpec("TAB"),)),)),
        14: KeySpec("&kp", (KeySpec("DEL"),)),
        15: KeySpec("&none"),
        16: KeySpec("&none"),
        17: KeySpec("&none"),
        18: KeySpec("&none"),
        19: KeySpec("&none"),
        20: KeySpec("&none"),
        21: KeySpec("&none"),
        22: KeySpec("&none"),
        23: KeySpec("&kp", (KeySpec("RET"),)),
        24: KeySpec("&kp", (KeySpec("SPACE"),)),
        25: KeySpec("&kp", (KeySpec("TAB"),)),
        26: KeySpec("&kp", (KeySpec("BSPC"),)),
        27: KeySpec("&kp", (KeySpec("LC", (KeySpec("X"),)),)),
        28: KeySpec("&kp", (KeySpec("LC", (KeySpec("X"),)),)),
        29: KeySpec("&kp", (KeySpec("LS", (KeySpec("TAB"),)),)),
        30: KeySpec("&kp", (KeySpec("LC", (KeySpec("Z"),)),)),
        31: KeySpec("&kp", (KeySpec("LC", (KeySpec("Y"),)),)),
        32: KeySpec("&kp", (KeySpec("TAB"),)),
        33: KeySpec("&none"),
        34: KeySpec("&sk", (KeySpec("LSHFT"),)),
        35: KeySpec("&kp", (KeySpec("LGUI"),)),
        36: KeySpec("&kp", (KeySpec("LALT"),)),
        37: KeySpec("&kp", (KeySpec("LCTRL"),)),
        38: KeySpec("&kp", (KeySpec("LSHFT"),)),
        39: KeySpec("&kp", (KeySpec("LC", (KeySpec("C"),)),)),
        40: KeySpec("&kp", (KeySpec("LC", (KeySpec("C"),)),)),
        41: KeySpec("&kp", (KeySpec("LEFT"),)),
        42: KeySpec("&kp", (KeySpec("UP"),)),
        43: KeySpec("&kp", (KeySpec("DOWN"),)),
        44: KeySpec("&kp", (KeySpec("RIGHT"),)),
        45: KeySpec("&sk", (KeySpec("RSHFT"),)),
        46: KeySpec("&kp", (KeySpec("LC", (KeySpec("L"),)),)),
        47: KeySpec("&kp", (KeySpec("LC", (KeySpec("A"),)),)),
        48: KeySpec("&cur_SELECT_LINE_v1_TKZ"),
        49: KeySpec("&cur_SELECT_WORD_v1_TKZ"),
        50: KeySpec("&kp", (KeySpec("LC", (KeySpec("F"),)),)),
        51: KeySpec("&kp", (KeySpec("LC", (KeySpec("V"),)),)),
        52: KeySpec("&mod_tab_v2_TKZ", (KeySpec("LALT"),)),
        53: KeySpec("&mod_tab_v2_TKZ", (KeySpec("LGUI"),)),
        54: KeySpec("&kp", (KeySpec("PSCRN"),)),
        55: KeySpec("&cur_SELECT_NONE_v1_TKZ"),
        56: KeySpec("&cur_EXTEND_LINE_v1_TKZ"),
        57: KeySpec("&cur_EXTEND_WORD_v1_TKZ"),
        58: KeySpec("&kp", (KeySpec("LC", (KeySpec("V"),)),)),
        59: KeySpec("&kp", (KeySpec("HOME"),)),
        60: KeySpec("&kp", (KeySpec("PG_UP"),)),
        61: KeySpec("&kp", (KeySpec("PG_DN"),)),
        62: KeySpec("&kp", (KeySpec("END"),)),
        63: KeySpec("&kp", (KeySpec("LC", (KeySpec("L"),)),)),
        64: KeySpec("&kp", (KeySpec("LC", (KeySpec("K"),)),)),
        65: KeySpec("&kp", (KeySpec("LC", (KeySpec("Z"),)),)),
        66: KeySpec("&kp", (KeySpec("LC", (KeySpec("Y"),)),)),
        67: KeySpec("&kp", (KeySpec("LC", (KeySpec("LS", (KeySpec("G"),)),)),)),
        68: KeySpec("&kp", (KeySpec("LC", (KeySpec("G"),)),)),
        69: KeySpec("&none"),
        70: KeySpec("&mod_tab_v2_TKZ", (KeySpec("LCTRL"),)),
        71: KeySpec("&kp", (KeySpec("LG", (KeySpec("D"),)),)),
        72: KeySpec("&kp", (KeySpec("LC", (KeySpec("A"),)),)),
        73: KeySpec("&cur_SELECT_LINE_v1_TKZ"),
        74: KeySpec("&cur_SELECT_WORD_v1_TKZ"),
        75: KeySpec("&kp", (KeySpec("LC", (KeySpec("F"),)),)),
        76: KeySpec("&kp", (KeySpec("LC", (KeySpec("LS", (KeySpec("G"),)),)),)),
        77: KeySpec("&kp", (KeySpec("LC", (KeySpec("G"),)),)),
        78: KeySpec("&kp", (KeySpec("LC", (KeySpec("H"),)),)),
        79: KeySpec("&kp", (KeySpec("LC", (KeySpec("K"),)),)),
    },
)

_BASE_CURSOR_LAYER: Layer = build_layer_from_spec(CURSOR_SPEC)


_LS_Z = KeySpec("LS", (KeySpec("Z"),))
_LS_G = KeySpec("LS", (KeySpec("G"),))

_MAC_CURSOR_COMMANDS: PatchSpec = {
    27: command_binding("X"),
    28: command_binding("X"),
    30: command_binding("Z"),
    31: command_binding(_LS_Z),
    39: command_binding("C"),
    40: command_binding("C"),
    46: command_binding("L"),
    47: command_binding("A"),
    50: command_binding("F"),
    51: command_binding("V"),
    58: command_binding("V"),
    63: command_binding("L"),
    64: command_binding("K"),
    65: command_binding("Z"),
    66: command_binding(_LS_Z),
    67: command_binding(_LS_G),
    68: command_binding("G"),
    71: command_binding("F3"),
    72: command_binding("A"),
    75: command_binding("F"),
    76: command_binding(_LS_G),
    77: command_binding("G"),
    79: command_binding("K"),
}

_MAC_CURSOR_CUSTOM_PATCH: PatchSpec = {
    35: KeySpec("&kp", (KeySpec("LCTRL"),)),
    37: KeySpec("&kp", (KeySpec("LGUI"),)),
    48: KeySpec("&cur_SELECT_LINE_macos_v1_TKZ"),
    49: KeySpec("&cur_SELECT_WORD_macos_v1_TKZ"),
    52: KeySpec("&mod_tab_v2_TKZ", (KeySpec("LGUI"),)),
    53: KeySpec("&mod_tab_v2_TKZ", (KeySpec("LALT"),)),
    56: KeySpec("&cur_EXTEND_LINE_macos_v1_TKZ"),
    57: KeySpec("&cur_EXTEND_WORD_macos_v1_TKZ"),
    73: KeySpec("&cur_SELECT_LINE_macos_v1_TKZ"),
    74: KeySpec("&cur_SELECT_WORD_macos_v1_TKZ"),
}


def build_cursor_layer(variant: str) -> Layer:
    layer = copy_layer(_BASE_CURSOR_LAYER)
    if base_variant_for(variant) in {"mac", "bilateral_mac"}:
        apply_mac_morphs(layer, _MAC_CURSOR_COMMANDS)
        apply_indices_patch(layer, _MAC_CURSOR_CUSTOM_PATCH)
    return layer
