from __future__ import annotations
from dataclasses import dataclass
from typing import Optional
from scanner3d.zemod.iar.zemod_iar import ZeModIAR


@dataclass(frozen=True, slots=True)
class ShotMeta:
    Nx: int
    Ny: int
    MinX: float
    MinY: float
    Dx: float
    Dy: float
    lens_file: str
    feature_description: str
    date_iso: str
    min_value: Optional[float] = None
    max_value: Optional[float] = None


    @classmethod
    def from_result(cls, result:ZeModIAR,grid_index:int) -> "ShotMeta":

        return cls(
            lens_file=result.lens_file,
            feature_description=result.description,
            date_iso=result.date_iso,
            Nx=result.get_data_grid(grid_index).nx,
            Ny=result.get_data_grid(grid_index).ny,
            MinX=result.get_data_grid(grid_index).min_x,
            MinY=result.get_data_grid(grid_index).min_y,
            Dx=result.get_data_grid(grid_index).dx,
            Dy=result.get_data_grid(grid_index).dy,
            min_value=result.get_data_grid(grid_index).value_min,
            max_value=result.get_data_grid(grid_index).value_max)

    def __str__(self) -> str:
        feature = (self.feature_description or "-").strip()
        lines = [
            "🧾 PSF Shot Meta:",
            "──────────────────────",
            f"{'Lens File:':18s} {self.lens_file or 'Unknown'}",
            f"{'Feature:':18s} {feature or '-'}",
            f"{'Date:':18s} {self.date_iso or '?'}",

            f"{'Grid (Nx×Ny):':18s} {self.Nx} × {self.Ny}",
            f"{'Step (Dx, Dy):':18s} ({self.Dx:.4f}, {self.Dy:.4f})",
            f"{'Range X:':18s} [{self.MinX:.3f} .. {self.MinX + self.Nx * self.Dx:.3f}]",
            f"{'Range Y:':18s} [{self.MinY:.3f} .. {self.MinY + self.Ny * self.Dy:.3f}]",
            f"{'Value range:':18s} [{self.min_value:.3e} .. {self.max_value:.3e}]"]
        return "\n".join(lines)

    __repr__ = __str__
