from __future__ import annotations
from dataclasses import dataclass
from typing import Tuple
from allytools.units import Length, LengthUnit

@dataclass(frozen=True, kw_only=True)
class ZRange:
    z_min: Length
    z_max: Length
    z_focus: Length

    def __post_init__(self) -> None:
        for name, z in (("z_min", self.z_min), ("z_max", self.z_max), ("z_focus", self.z_focus)):
            if z.value_mm < 0:
                raise ValueError(f"{name} must be non-negative; got {z}.")
        if self.z_min > self.z_max:
            raise ValueError(f"z_min ({self.z_min}) cannot be greater than z_max ({self.z_max}).")
        if not (self.z_min <= self.z_focus <= self.z_max):
            raise ValueError(f"z_focus ({self.z_focus}) must lie within [{self.z_min}, {self.z_max}].")

    @property
    def span(self) -> Length:
        return self.z_max - self.z_min

    @property
    def center(self) -> Length:
        return (self.z_min + self.z_max) / 2

    def contains(self, z: Length, *, inclusive: bool = True) -> bool:
        return (self.z_min <= z <= self.z_max) if inclusive else (self.z_min < z < self.z_max)

    def to_tuple(self) -> Tuple[Length, Length, Length]:
        return self.z_min, self.z_max, self.z_focus

    def __str__(self) -> str:
        return f"ZRange[{self.z_min} .. {self.z_max}] (focus={self.z_focus})"

    def __repr__(self) -> str:
        return (f"ZRange(z_min={self.z_min!s}, "
                f"z_max={self.z_max!s}, "
                f"z_focus={self.z_focus!s})")
