from __future__ import annotations
import numpy as np
from typing import Optional, Tuple, Callable, Literal, TYPE_CHECKING
from scanner3d.analysis.psf.frame import Frame

def read_vec(ds) -> np.ndarray:
    """Read HDF5 dataset that may be scalar () or vector; return 1-D float array."""
    arr = ds[()]  # works for scalar and array
    return np.atleast_1d(np.asarray(arr, dtype=float))


def suggest_chunks(shape: Tuple[int, ...]) -> Tuple[int, ...]:
    """
    Suggest chunks so that the last 2 dims are stored as full 2D blocks.

    For shape (n_shots, Ny, Nx) this returns (1, Ny, Nx):
    one shot per chunk, full 2D PSF each time.
    """
    if len(shape) < 2:
        raise ValueError("Expected at least 2 dims (Ny, Nx).")
    f_ndim = len(shape) - 2
    return (1,) * f_ndim + (shape[-2], shape[-1])


def frame_name(frame: Frame) -> str:
    wd = frame.profile.working_distance
    return f"psf_frame_at_{wd:.2f}" if wd is not None else "psf_frame_at_unknown"
