from __future__ import annotations
import h5py
from typing import Literal



def resolve_h5_name(
    container: h5py.Group | h5py.File,
    base: str,
    on_conflict: Literal["error", "overwrite", "skip", "rename"],
    what: str = "item",
) -> tuple[str, bool]:
    """
    Resolve name conflicts inside an HDF5 group/file.

    Returns:
        (final_name, skip_flag)

    - If skip_flag is True, caller should skip creating/writing this item.
    - If final_name differs from base, 'rename' policy was applied.
    """
    name = base

    if name in container:
        if on_conflict == "error":
            raise ValueError(f"{what} '{name}' already exists.")
        elif on_conflict == "overwrite":
            del container[name]
        elif on_conflict == "skip":
            # Leave existing item as-is, but tell caller to skip.
            return name, True
        elif on_conflict == "rename":
            k = 1
            while f"{base}_{k:03d}" in container:
                k += 1
            name = f"{base}_{k:03d}"

    return name, False
