import numpy as np
from allytools.strings import sanitize
from allytools.units import LengthUnit, Length
from scanner3d.camera3d.camera3d import Camera3D

def file_stem(camera: Camera3D, wd_mm: float | None = None) -> str:
    base = (
        f"{camera.objective.objectiveID}__"
        f"F{camera.objective.f_number:g}__"
        f"{camera.sensor.sensor_model}"
    )
    if wd_mm is not None:
        base += f"_{wd_mm:.0f}mm"
    return sanitize(base)

def wd_triplet(camera: Camera3D) -> list[float]:
    return [
        float(v.to(LengthUnit.MM))
        for v in (camera.z_range.z_min, camera.z_range.z_focus, camera.z_range.z_max)
        if v is not None
    ]

def wd_sequence(camera: Camera3D, step_z: Length) -> np.ndarray:
    if not isinstance(step_z, Length):
        raise TypeError("step_z must be a Length instance.")
    if step_z.value_mm <= 0:
        raise ValueError("step_z must be positive.")

    z_min = camera.z_range.z_min
    z_max = camera.z_range.z_max
    if z_min is None or z_max is None:
        raise ValueError("camera.z_range must define both z_min and z_max.")

    z_min_mm = z_min.value_mm
    z_max_mm = z_max.value_mm
    step_mm = step_z.value_mm

    # Generate raw sequence (might exceed z_max)
    seq = np.arange(z_min_mm, z_max_mm + step_mm, step_mm, dtype=np.float64)

    # Clamp to z_max_mm
    if seq[-1] > z_max_mm:
        seq[-1] = z_max_mm
    return seq


