import logging
import sys
from pathlib import Path

try:
    import colorama
    colorama.just_fix_windows_console()
except Exception:
    colorama = None


class ColorFormatter(logging.Formatter):
    """ANSI-colored logging formatter for console output."""
    COLORS = {
        "DEBUG": "\033[36m",    # Cyan
        "INFO": "\033[32m",     # Green
        "WARNING": "\033[33m",  # Yellow
        "ERROR": "\033[31m",    # Red
        "CRITICAL": "\033[41m"  # White on Red background
    }
    RESET = "\033[0m"

    def format(self, record):
        level = record.levelname
        color = self.COLORS.get(level, self.RESET)
        record.levelname = f"{color}{level}{self.RESET}"
        return super().format(record)



class LoggerSetup:
    """Centralized logger configuration for scanner3d."""

    @staticmethod
    def configure(
        name: str,
        level: int = logging.DEBUG,
        suppress_matplotlib: bool = True,
        log_file: str | Path | None = None,
    ) -> logging.Logger:
        """Configure global colored logging and optional file logging."""

        fmt = "%(asctime)s [%(levelname)s] %(name)s: %(message)s"

        handlers: list[logging.Handler] = []

        # Console (colored)
        console_handler = logging.StreamHandler(sys.stdout)
        console_handler.setFormatter(ColorFormatter(fmt))
        handlers.append(console_handler)

        # Optional file handler (no colors)
        if log_file is not None:
            fh = logging.FileHandler(log_file, encoding="utf-8")
            fh.setFormatter(logging.Formatter(fmt))
            handlers.append(fh)

        # Configure ROOT logger so all libraries propagate here
        logging.basicConfig(level=level, handlers=handlers, force=True)

        if suppress_matplotlib:
            logging.getLogger("matplotlib").setLevel(logging.WARNING)

        log = logging.getLogger(name)
        log.debug("Logger initialized for %s", name)
        return log