from __future__ import annotations
import numpy as np
from typing import Tuple, Sequence
from dataclasses import dataclass
from scanner3d.analysis.psf.Frame import Frame
from imagera.plotter import scalar2d_array, PlotParams, CMaps

@dataclass
class PsfSlicePlotter:
    """Holds rendered PSF slice image and associated working distance."""
    image: np.ndarray
    working_distance: float

    @classmethod
    def from_psf_slice(
        cls,
        frame: Frame,
        cmap=CMaps.JET,
        dpi: int = 300,
        fig_size_in: Tuple[float, float] = (3, 3),
    ) -> PsfSlicePlotter:
        """Render a single PSF slice into a result container."""
        ss = frame.system_snapshot
        title = (f"FFT PSF Analysis for {ss.objective_id}\n"
                 f"with sensor {ss.sensor_model} at {ss.working_distance:.3f} mm")
        X, Y = np.meshgrid(frame.x_seq, frame.y_seq)
        quadrant = np.stack((X, Y), axis=-1)
        image = scalar2d_array(
            grid=quadrant,
            values_array=frame.values,
            params=PlotParams(cmap=cmap, dpi=dpi, size_in=fig_size_in),
            with_colorbar=True,
            hide_ticks=True,
            tile_titles=False,
            mosaic_title=title,
        )
        return cls(image=image, working_distance=ss.working_distance)

    @classmethod
    def from_many(
        cls,
        psf_slices: Sequence[Frame],
        *,
        cmap=CMaps.JET,
        dpi: int = 300,
        fig_size_in: Tuple[float, float] = (3, 3),
    ) -> list[PsfSlicePlotter]:
        """Render multiple PSF slices."""
        return [
            cls.from_psf_slice(s, cmap=cmap, dpi=dpi, fig_size_in=fig_size_in)
            for s in psf_slices
        ]
