from __future__ import annotations
import time
import logging
from pathlib import Path
from dataclasses import dataclass, field
from scanner3d.tuner.tuner import Tuner
from scanner3d.test.tests.camera_test import CameraTest


log = logging.getLogger(__name__)
@dataclass
class TestSuite:
    """A bundle of CameraTest instances with convenient run helpers."""
    tests: list[CameraTest]
    tuner: Tuner | None = field(init=False, default=None)

    def names(self) -> list[str]:
        return [t.test_name for t in self.tests]

    def select(self, only: set[str] | None = None, skip: set[str] | None = None) -> TestSuite:
        filtered = self.tests
        if only:
            filtered = [t for t in filtered if t.test_name in only]
        if skip:
            filtered = [t for t in filtered if t.test_name not in skip]
        return TestSuite(filtered)

    def __iter__(self):
        return iter(self.tests)

    def run(self, zemod, camera, output_root: Path) -> None:
        self.tuner = Tuner(zemod=zemod, camera= camera)
        for t in self.tests:
            log.info("=== Running test: %s ===", t.describe())
            start_time = time.perf_counter()
            try:
                ok = t.run(zemod=zemod,camera=camera,output_root=output_root,tuner=self.tuner)
                t.elapsed = time.perf_counter() - start_time
                t.success = True
                log.info("=== %s -> %s (%.2f s) ===", t.test_name, "PASS" if ok else "FAIL", t.elapsed)
            except Exception:
                t.elapsed = time.perf_counter() - start_time
                log.exception("Test crashed: %s (%.2f s)", t.test_name, t.elapsed)
                t.success = False


