from pathlib import Path
from abc import ABC, abstractmethod
from scanner3d.test.tests.test_settings import TestSettings
from scanner3d.tuner.tuner import Tuner

class CameraTest(ABC):
    """Base class for camera tests with automatic output folder creation."""
    test_name: str
    settings: TestSettings
    output_root: Path | None = None
    success: bool = False
    elapsed: float = 0.0

    def run(self, *, zemod, camera, output_root: Path, tuner: Tuner) -> bool:
        self.output_root = output_root / self.test_name
        self.output_root.mkdir(parents=True, exist_ok=True)

        return self.perform(
            zemod=zemod, camera=camera, _output_root=self.output_root, tuner=tuner
        )

    def describe(self) -> str:
        params = ", ".join(
            f"{k}={v!s}"  # or just f"{k}={v}"
            for k, v in self.__dict__.items()
            if k not in {"output_root", "success", "elapsed"}
        )
        return f"{self.test_name}({params})"

    @abstractmethod
    def perform(self, *, zemod, camera, _output_root: Path, tuner: Tuner) -> bool:
        """
        Subclass implements core logic here.
        output_root = the directory created by the base class
        """
        ...

