from __future__ import annotations
from typing import Protocol
import numpy as np
from numpy.typing import NDArray

from scanner3d.zemod.iar.i_grid_meta import IGridMeta


class IDataGrid(IGridMeta, Protocol):

    @property
    def values(self) -> NDArray[np.float64]: ...

    @property
    def meta(self) -> IGridMeta:
        return self  # type: ignore[return-value]


    @property
    def shape(self) -> tuple[int, int]:
        """
        (ny, nx) – number of rows and columns.

        NOTE: Implementors must provide this so meta can be used
        without having direct access to the data array.
        """
        ...

    # --- default helpers derived from the primitives ---

    @property
    def nx(self) -> int:
        return self.shape[1]

    @property
    def ny(self) -> int:
        return self.shape[0]

    @property
    def x_max(self) -> float:
        return self.min_x + (self.nx - 1) * self.dx

    @property
    def y_max(self) -> float:
        return self.min_y + (self.ny - 1) * self.dy

    @property
    def extent(self) -> tuple[float, float, float, float]:
        return self.min_x, self.x_max, self.min_y, self.y_max

    @property
    def x_coords(self) -> NDArray[np.float64]:
        return self.min_x + self.dx * np.arange(self.nx, dtype=np.float64)

    @property
    def y_coords(self) -> NDArray[np.float64]:
        return self.min_y + self.dy * np.arange(self.ny, dtype=np.float64)

    def x(self, ix: int) -> float:
        return self.min_x + ix * self.dx

    def y(self, iy: int) -> float:
        return self.min_y + iy * self.dy

    @property
    def value_min(self) -> float:
        try:
            return float(np.nanmin(self.values))
        except ValueError:
            return float("nan")

    @property
    def value_max(self) -> float:
        try:
            return float(np.nanmax(self.values))
        except ValueError:
            return float("nan")

    def z(self, ix: int, iy: int) -> float:
        return float(self.values[iy, ix])
