from __future__ import annotations
from typing import TYPE_CHECKING, cast, Tuple
import numpy as np
from numpy.typing import NDArray

from scanner3d.zemod.core.native_adapter import NativeAdapter
from scanner3d.zemod.iar.i_data_grid import IDataGrid
from scanner3d.zemod.iar.grid_meta import GridMeta

if TYPE_CHECKING:
    from zempy.zosapi.analysis.data.protocols.iar_data_grid import IAR_DataGrid


class ZeModDataGrid(NativeAdapter["IAR_DataGrid"], IDataGrid):
    __slots__ = ("_meta",)

    def __init__(self, native: "IAR_DataGrid") -> None:
        super().__init__(native)
        # snapshot metadata once
        self._meta = GridMeta.from_zempy(native)

    # --- IDataGrid core ---

    @property
    def values(self) -> NDArray[np.float64]:
        return cast(NDArray[np.float64], self.native.Values)

    @property
    def min_x(self) -> float:
        return self._meta.min_x

    @property
    def min_y(self) -> float:
        return self._meta.min_y

    @property
    def dx(self) -> float:
        return self._meta.dx

    @property
    def dy(self) -> float:
        return self._meta.dy

    @property
    def shape(self) -> Tuple[int, int]:
        # (rows, cols) = (ny, nx)

        return self.ny, self.nx

    @property
    def meta(self) -> GridMeta:
        return self._meta

    @property
    def description(self) -> str:
        return self._meta.description

    @property
    def x_label(self) -> str:
        return self._meta.x_label

    @property
    def y_label(self) -> str:
        return self._meta.y_label

    @property
    def value_label(self) -> str:
        return self._meta.value_label
