from __future__ import annotations
import logging
from functools import cached_property
from typing import TYPE_CHECKING, Optional
from scanner3d.zemod.core.native_adapter import NativeAdapter
from zempy.zosapi.tools.general.enums.quick_focus_criterion import QuickFocusCriterion
from scanner3d.zemod.quickfocus_settings import QuickFocusSettings
from scanner3d.zemod.zemod_analyses import ZeModAnalyses
from scanner3d.zemod.zemod_analysis import ZeModAnalysis
from scanner3d.zemod.zemod_sd import ZeModSD
from scanner3d.zemod.zemod_lde import ZeModLDE
from zempy.zosapi.analysis.enums.analysis_idm import AnalysisIDM

if TYPE_CHECKING:
    from zempy.zosapi.system.protocols.i_optical_system import IOpticalSystem
    from zempy.zosapi.tools.protocols.i_optical_system_tools import IOpticalSystemTools
    from zempy.zosapi.systemdata.protocols.isd_title_notes import ISDTitleNotes

log = logging.getLogger(__name__)
class ZeMod(NativeAdapter["IOpticalSystem"]):
    __slots__ = ()

    @property
    def tools(self) -> "IOpticalSystemTools":
        return self.native.Tools

    @cached_property
    def analyses(self) -> ZeModAnalyses:
        return ZeModAnalyses(self.native.Analyses)

    @cached_property
    def sd(self) -> ZeModSD:
        return ZeModSD(self.native.SystemData)

    @cached_property
    def lde(self) -> ZeModLDE:
        return ZeModLDE(self.native.LDE)

    @classmethod
    def from_optical_system(cls, optical_system: "IOpticalSystem") -> "ZeMod":
        return cls(optical_system)


    def get_fftpsf(self) -> ZeModAnalysis:
        return self.analyses.new_analysis(AnalysisIDM.FftPsf)

    def quick_focus(self, settings: Optional[QuickFocusSettings] = None) -> bool:
        with self.tools.OpenQuickFocus() as qf:
            if settings is None:
                qf.Criterion = QuickFocusCriterion.SpotSizeRadial
                qf.UseCentroid = True
            else:
                qf.Criterion = settings.criterion
                qf.UseCentroid = settings.use_centroid
            result = qf.RunAndWaitForCompletion()
        return bool(result)

    def get_batch_ray_tracer(self):
        return self.tools.OpenBatchRayTrace()

    @property
    def file_name(self)->str:
        return self.native.SystemFile

    @property
    def system_name(self)->str:
        return self.native.SystemName