"""
Configuration management for Flexagg.

This module handles reading and writing of configuration files,
supporting both simple script directories and structured flexagg packages.
"""
import json
import os
from pathlib import Path
from typing import Optional, Set, Union, Literal, List

import yaml
from pydantic import BaseModel, ConfigDict, Field
from typing_extensions import Annotated

BUILTIN_APPS = {
    Path(__file__).parent.parent / "flexagg-builtin-apps",
}
CONFIG_FILE_NOTE = "# This file is auto-generated by flexagg, do not edit it manually.\n"


class PlaintextSecretsConfig(BaseModel):
    provider: Literal["plaintext"] = Field(default="plaintext")


class GPGSecretsConfig(BaseModel):
    provider: Literal["gpg"] = Field(default="gpg")
    recipients: List[str] = Field(default_factory=list, description="One or more GPG recipient KEYIDs or emails")
    use_agent: bool = Field(default=True, description="Use gpg-agent for passphrase/cert handling")
    home: Optional[Path] = Field(default=None, description="Optional GNUPGHOME directory")


class CryptoSecretsConfig(BaseModel):
    provider: Literal["crypto"] = Field(default="crypto")
    passphrase_hash: str = Field(description="SHA256 hex of symmetric passphrase", min_length=64, max_length=64)


SecretsConfig = Annotated[
    Union[PlaintextSecretsConfig, GPGSecretsConfig, CryptoSecretsConfig],
    Field(discriminator="provider"),
]


def env_true(env_key: str) -> bool:
    return os.getenv(env_key, "").lower() in {"yes", "y", "true", "ture", "t", "1", "ok"}


class Config(BaseModel):
    model_config = ConfigDict(
        extra="ignore",
        json_encoders={
            Path: lambda x: str(x),
        },
    )
    __CONFIG_PATH__: Optional[Path] = None

    dirs: Set[Path] = Field(default_factory=set, description="Script directories to scan for Typer instances")
    skip_external_path: bool = Field(default_factory=lambda: env_true("FA_SKIP_EXTERNAL_PATH"), exclude=True)
    is_complete: bool = Field(default_factory=lambda: bool(os.getenv("_FA_COMPLETE")), exclude=True)

    # Secrets subsystem configuration
    secrets: SecretsConfig = Field(default_factory=PlaintextSecretsConfig)

    @property
    def app_dirs(self) -> Set[Path]:
        if self.skip_external_path:
            return BUILTIN_APPS
        return self.dirs | BUILTIN_APPS

    @property
    def all_dirs(self) -> Set[Path]:
        return self.dirs | BUILTIN_APPS

    @classmethod
    def load(cls):
        user_config_path = Path.home() / ".fa" / "config.yaml"
        config_path = None

        if "FA_DIRS" in os.environ:
            dirs = {Path(p.strip()) for p in os.environ["FA_DIRS"].split(",")}
            ins = cls(dirs=dirs, __CONFIG_PATH__=user_config_path)
            return ins

        if "FA_CONFIG_PATH" in os.environ:
            config_path = os.environ["FA_CONFIG_PATH"]
        elif user_config_path.exists():
            config_path = user_config_path
        if config_path:
            with open(config_path, "r", encoding="utf-8") as f:
                ins = cls.model_validate(yaml.safe_load(f))
                ins.__CONFIG_PATH__ = Path(config_path)
                return ins
        return cls(__CONFIG_PATH__=user_config_path)

    def add_dir(self, path: Path):
        if path.absolute() in self.app_dirs:
            return
        self.dirs.add(path.absolute())

    def remove_dir(self, path: Path):
        if path.absolute() not in self.dirs:
            return False
        self.dirs.remove(path.absolute())
        return True

    def save(self):
        config_path = self.__CONFIG_PATH__
        if not config_path:
            config_path = Path.home() / ".fa" / "config.yaml"
        config_path.parent.mkdir(parents=True, exist_ok=True)
        with open(config_path, "w", encoding="utf-8") as f:
            f.write(CONFIG_FILE_NOTE)
            yaml.dump(json.loads(self.model_dump_json()), f)


config = Config.load()
