# RiSpec

AI-assisted change-management tool for senior engineers working on large, single-repo legacy codebases.

## Features

- Context acquisition and legacy awareness
- Clarification and planning before coding
- Boundary-aware incremental implementation
- Transparent, human-first workflow and traceability

## Installation

### Install from PyPI (Recommended)

```bash
pip install rispec
```

### Install from Source

If you want to install from source or contribute:

```bash
# Clone the repository
git clone <repository-url>
cd RiSpec

# Create and activate virtual environment
python -m venv venv
.\venv\Scripts\activate  # Windows PowerShell
# or: source venv/bin/activate  # Linux/Mac

# Install in development mode
pip install -e .
```

## Quick Start (Using RiSpec CLI)

### 1. Setup

1. Create and activate virtual environment:
```bash
python -m venv venv
.\venv\Scripts\activate  # Windows PowerShell
# Or: .\venv\Scripts\activate.bat  # Windows CMD
```

2. Install dependencies:
```bash
pip install -r requirements.txt
```

3. Install the package:
```bash
pip install -e .
```

### 2. Configuration

RiSpec can be configured via environment variables or an interactive menu. Configuration is stored in a `.env` file in your project directory.

#### Option A: Interactive Configuration (Recommended)

Use the interactive menu to configure RiSpec:

```bash
# Activate virtual environment first
.\venv\Scripts\activate

# Open interactive configuration menu
rispec config --interactive
```

The interactive menu allows you to:
- **Set OpenAI API Key** - Securely enter your OpenAI API key (input is hidden)
- **Select OpenAI Model** - Choose from available models:
  - `gpt-5.1`
  - `gpt-4-turbo-preview` (default)
  - `gpt-4`
  - `gpt-4-32k`
  - `gpt-3.5-turbo`
  - `gpt-3.5-turbo-16k`
- **View current configuration** - Display all settings
- **Exit** - Save and exit

All changes are automatically saved to `.env`.

#### Option B: Manual Configuration

Create a `.env` file in the project root:

```bash
# Create .env file
touch .env  # Linux/Mac
# Or create .env manually on Windows
```

Add the following variables:

```env
# OpenAI Configuration (Required for LLM features)
OPENAI_API_KEY=your_api_key_here
OPENAI_MODEL=gpt-4-turbo-preview

# Performance Thresholds
MAX_REPO_LOC=1000000          # Maximum lines of code before warning
INDEXING_TIMEOUT_SECONDS=60   # Indexing timeout in seconds
MAX_PATCH_SIZE_LINES=30       # Maximum patch size

# Legacy Pattern Detection Thresholds
LARGE_FILE_LOC_THRESHOLD=1000      # Lines of code for large file detection
LARGE_METHOD_LOC_THRESHOLD=100      # Lines of code for large method detection
HIGH_FAN_IN_THRESHOLD=10            # Fan-in threshold for high coupling
HIGH_FAN_OUT_THRESHOLD=15           # Fan-out threshold for high coupling

# Data Storage
DATA_DIR=data                       # Directory for storing session data
```

#### View Current Configuration

```bash
rispec config
```

This displays:
- Data directory location
- Max repository LOC threshold
- Indexing timeout
- Max patch size
- OpenAI model
- API key status (configured/not configured)

### 3. Usage

#### Activate Virtual Environment

**Windows PowerShell:**
```powershell
.\venv\Scripts\Activate.ps1
# Or if you get execution policy error:
.\venv\Scripts\activate.bat
```

**Windows Command Prompt:**
```cmd
venv\Scripts\activate.bat
```

#### Basic CLI Commands

Once the virtual environment is activated:

```bash
# Show help
rispec --help

# Analyze a repository
rispec analyze <path_to_repository>

# Analyze with options
rispec analyze <path_to_repository> --top-n 5
rispec analyze <path_to_repository> --json
rispec analyze <path_to_repository> --hotspots  # Include legacy pattern hotspots

# Context management (for iterative context retrieval)
rispec context init <repo_path> <task_id> "<description>"
rispec context expand <repo_path> <task_id> --symbols "Symbol1,Symbol2" --llm
rispec context show <repo_path> <task_id>
rispec context include <repo_path> <task_id> <file_path>
rispec context exclude <repo_path> <task_id> <file_path>
rispec context log <repo_path> <task_id>

# Legacy pattern detection
rispec hotspots <repo_path> --top-n 20 --area "path/to/module"

# Show configuration
rispec config
rispec config --interactive

# Show version
rispec --version
```

#### Alternative: Using Python Module

If you prefer not to activate the virtual environment:

```bash
# Windows PowerShell
.\venv\Scripts\python.exe -m rispec.cli --help
.\venv\Scripts\python.exe -m rispec.cli analyze <path_to_repository>
.\venv\Scripts\python.exe -m rispec.cli config --interactive
```

For detailed command documentation, see [docs/CLI_USAGE.md](docs/CLI_USAGE.md).

---

## Contributing (Local Development Setup)

This section is for developers who want to contribute to RiSpec development.

### Prerequisites

- Python 3.8 or higher
- Git
- Virtual environment support

### Setup Development Environment

1. **Clone the repository:**
```bash
git clone <repository-url>
cd RiSpec
```

2. **Create and activate virtual environment:**
```bash
# Create virtual environment
python -m venv venv

# Activate (Windows PowerShell)
.\venv\Scripts\Activate.ps1

# Or (Windows CMD)
.\venv\Scripts\activate.bat

# Or (Linux/Mac)
source venv/bin/activate
```

3. **Install dependencies:**
```bash
pip install -r requirements.txt
```

4. **Install package in development mode:**
```bash
pip install -e .
```

This installs RiSpec in "editable" mode, so changes to the source code are immediately available without reinstalling.

5. **Configure environment (optional but recommended):**
```bash
# Set up .env file for testing LLM features
rispec config --interactive
```

Or create `.env` manually with at least:
```env
OPENAI_API_KEY=your_test_api_key_here
```

### Development Workflow

#### Running Tests

```bash
# Run all tests
pytest

# Run with verbose output
pytest -v

# Run specific test file
pytest tests/test_repository.py

# Run specific test
pytest tests/test_repository.py::test_repository_analyzer_initialization
```

#### Running Tests with Coverage

```bash
# Generate coverage report (terminal)
pytest --cov=rispec --cov-report=term

# Generate HTML coverage report
pytest --cov=rispec --cov-report=html

# Open HTML report (Windows PowerShell)
Invoke-Item htmlcov\index.html
```

#### Code Quality

```bash
# Run linter (if configured)
# Example with flake8:
flake8 rispec/ tests/

# Run type checking (if configured)
# Example with mypy:
mypy rispec/
```

### Project Structure

```
RiSpec/
├── rispec/              # Main package
│   ├── __init__.py
│   ├── cli.py           # CLI interface
│   ├── config.py        # Configuration management
│   ├── repository.py    # Repository analysis
│   ├── legacy_patterns.py  # Legacy pattern detection
│   ├── context_manager.py  # Context management
│   └── llm_explainer.py    # LLM integration
├── tests/               # Test files
│   ├── test_repository.py
│   ├── test_legacy_patterns.py
│   ├── test_context_manager.py
│   └── test_cli.py
├── specs/               # Specifications
│   ├── MVP_UserStories.json
│   └── current_implementation.json
├── docs/                # Documentation
│   ├── CLI_USAGE.md
│   └── TESTING_CLI.md
├── data/                # Data storage (created at runtime)
├── htmlcov/            # Coverage reports (generated)
├── .env                 # Environment variables (not in git)
├── .gitignore
├── requirements.txt
├── pytest.ini
├── setup.py
└── README.md
```

### Making Changes

1. **Create a feature branch:**
```bash
git checkout -b feature/your-feature-name
```

2. **Make your changes:**
   - Write code following existing patterns
   - Add tests for new functionality
   - Update documentation as needed

3. **Run tests before committing:**
```bash
pytest
```

4. **Check test coverage:**
```bash
pytest --cov=rispec --cov-report=term-missing
```

5. **Commit your changes:**
```bash
git add .
git commit -m "Description of your changes"
```

### Testing Your Changes

#### Test CLI Commands Locally

```bash
# After making changes, test CLI commands
.\venv\Scripts\python.exe -m rispec.cli --help
.\venv\Scripts\python.exe -m rispec.cli analyze .
.\venv\Scripts\python.exe -m rispec.cli config
```

#### Integration Testing

```bash
# Run integration tests
pytest tests/test_cli.py -v

# Test with a sample repository
.\venv\Scripts\python.exe -m rispec.cli analyze . --hotspots
```

### Documentation

- Update `README.md` for user-facing changes
- Update `docs/CLI_USAGE.md` for CLI command changes
- Update `specs/current_implementation.json` for feature implementations
- Add docstrings to new functions/classes

### Before Submitting

- [ ] All tests pass: `pytest`
- [ ] Test coverage is maintained or improved
- [ ] Code follows existing style patterns
- [ ] Documentation is updated
- [ ] `.env` file is not committed (it's in `.gitignore`)
- [ ] No sensitive data in commits

### Getting Help

- Check existing tests for examples
- Review `specs/MVP_UserStories.json` for feature requirements
- Review `specs/current_implementation.json` for implementation status
