# Testing CLI Commands

This guide explains how to test CLI commands in RiSpec using pytest and Click's testing utilities.

## Overview

We use **Click's `CliRunner`** to test CLI commands. This allows us to:
- Invoke commands programmatically
- Capture output (stdout and stderr)
- Check exit codes
- Test error handling

## Setup

The test infrastructure is already set up in `tests/test_cli.py`. Key components:

1. **CliRunner**: Click's test runner
2. **Fixtures**: Reusable test data (sample repositories, runner instance)
3. **Assertions**: Verify command behavior and output

## Basic Test Structure

```python
from click.testing import CliRunner
from rispec.cli import cli

def test_my_command(runner):
    """Test a CLI command."""
    result = runner.invoke(cli, ['command', 'arg1', '--option', 'value'])
    
    assert result.exit_code == 0  # Success
    assert "expected output" in result.output
```

## Test Examples

### 1. Testing Help Commands

```python
def test_cli_help(runner):
    """Test CLI help command."""
    result = runner.invoke(cli, ['--help'])
    assert result.exit_code == 0
    assert "RiSpec" in result.output
```

### 2. Testing Command Success

```python
def test_analyze_command_success(runner, sample_repo):
    """Test analyze command with valid repository."""
    result = runner.invoke(cli, ['analyze', str(sample_repo)])
    
    assert result.exit_code == 0
    assert "Repository Analysis Summary" in result.output
```

### 3. Testing Command Options

```python
def test_analyze_command_with_top_n(runner, sample_repo):
    """Test analyze command with --top-n option."""
    result = runner.invoke(cli, ['analyze', str(sample_repo), '--top-n', '5'])
    assert result.exit_code == 0
```

### 4. Testing JSON Output

```python
def test_analyze_command_json_output(runner, sample_repo):
    """Test analyze command with --json flag."""
    result = runner.invoke(cli, ['analyze', str(sample_repo), '--json'])
    
    assert result.exit_code == 0
    # Parse and validate JSON
    import json
    data = json.loads(result.output)
    assert "total_files" in data
```

### 5. Testing Error Cases

```python
def test_analyze_command_nonexistent_path(runner):
    """Test analyze command with nonexistent path."""
    result = runner.invoke(cli, ['analyze', '/nonexistent/path'])
    
    assert result.exit_code != 0
    assert "Error" in result.output
```

### 6. Testing Multiple Options

```python
def test_analyze_command_combined_options(runner, sample_repo):
    """Test analyze command with multiple options."""
    result = runner.invoke(cli, [
        'analyze',
        str(sample_repo),
        '--top-n', '3',
        '--json'
    ])
    assert result.exit_code == 0
```

## Running Tests

### Run all CLI tests:
```bash
pytest tests/test_cli.py -v
```

### Run a specific test:
```bash
pytest tests/test_cli.py::test_analyze_command_success -v
```

### Run with coverage:
```bash
pytest tests/test_cli.py --cov=rispec.cli --cov-report=term-missing
```

### Run all tests:
```bash
pytest tests/ -v
```

## Test Coverage

Current CLI test coverage: **89%**

- ✅ Help and version commands
- ✅ Analyze command (success, options, JSON output, errors)
- ✅ Config command
- ✅ Error handling
- ✅ Combined options
- ⚠️ Error path in analyze command (lines 39-41) - low priority

## Best Practices

1. **Use fixtures** for reusable test data (sample repositories, runner)
2. **Test both success and failure cases**
3. **Verify exit codes** (`0` for success, non-zero for errors)
4. **Check output content** for expected strings
5. **Test options individually and in combination**
6. **Use descriptive test names** that explain what's being tested

## Common Patterns

### Creating Temporary Test Data

```python
@pytest.fixture
def sample_repo():
    """Create a temporary sample repository for testing."""
    repo_dir = tempfile.mkdtemp()
    repo_path = Path(repo_dir)
    
    # Create test files
    (repo_path / "main.py").write_text("...")
    
    yield repo_path
    
    # Cleanup
    shutil.rmtree(repo_dir)
```

### Testing Output Format

```python
def test_config_command_output_format(runner):
    """Test config command output format."""
    result = runner.invoke(cli, ['config'])
    
    assert result.exit_code == 0
    assert "=" * 50 in result.output  # Separator line
    assert "Data directory" in result.output
```

### Testing JSON Output

```python
def test_json_output(runner, sample_repo):
    """Test JSON output parsing."""
    result = runner.invoke(cli, ['analyze', str(sample_repo), '--json'])
    
    # Find JSON in output (may have stderr messages)
    output_lines = result.output.strip().split('\n')
    json_start = None
    for i, line in enumerate(output_lines):
        if line.strip().startswith('{'):
            json_start = i
            break
    
    if json_start is not None:
        json_output = '\n'.join(output_lines[json_start:])
        data = json.loads(json_output)
        assert "expected_key" in data
```

## Adding New CLI Tests

When adding a new CLI command:

1. Add test functions in `tests/test_cli.py`
2. Test the command with various inputs
3. Test success and error cases
4. Test all options and flags
5. Verify output format
6. Run tests: `pytest tests/test_cli.py -v`
7. Check coverage: `pytest tests/test_cli.py --cov=rispec.cli`

## Troubleshooting

### Test fails with "command not found"
- Make sure the CLI is properly imported: `from rispec.cli import cli`

### Output doesn't match expected
- Remember that Click's CliRunner captures both stdout and stderr
- Use `click.echo(..., err=True)` for progress messages
- Check for formatting differences (e.g., number formatting with commas)

### JSON parsing fails
- The output may include stderr messages before JSON
- Extract JSON portion from output before parsing

