"""CLI interface for RiSpec."""

import click
import json
from pathlib import Path
from typing import Optional, List, Dict
from rispec.repository import RepositoryAnalyzer
from rispec.legacy_patterns import LegacyPatternDetector
from rispec.context_manager import ContextManager
from rispec.config import Config


@click.group()
@click.version_option(version="0.1.0")
def cli():
    """RiSpec - AI-assisted change-management tool for legacy codebases."""
    pass


@cli.command()
@click.argument('repo_path', type=click.Path(exists=True, path_type=Path))
@click.option('--top-n', default=10, help='Number of top modules to show')
@click.option('--json', 'output_json', is_flag=True, help='Output as JSON')
@click.option('--hotspots', is_flag=True, help='Also detect and show legacy pattern hotspots')
@click.option('--hotspots-only', is_flag=True, help='Only show hotspots, skip repository summary')
def analyze(repo_path: Path, top_n: int, output_json: bool, hotspots: bool, hotspots_only: bool):
    """Analyze repository structure and dependencies.
    
    REPO_PATH: Path to the repository root directory
    """
    try:
        analyzer = RepositoryAnalyzer(repo_path)
        click.echo("Indexing repository...", err=True)
        
        results = analyzer.index()
        
        # Display indexing results with timing
        stats = analyzer.get_indexing_stats()
        click.echo(f"Indexed {results['total_files']} files, {results['total_modules']} modules", err=True)
        click.echo(f"Indexing completed in {results['indexing_time_seconds']:.2f} seconds", err=True)
        
        # Warn if LOC exceeds threshold
        if stats['exceeds_loc_threshold']:
            click.echo(
                f"⚠ Warning: Repository has {stats['total_loc']:,} LOC, "
                f"exceeds threshold of {Config.MAX_REPO_LOC:,} LOC",
                err=True
            )
        
        # Warn if timeout exceeded
        if results.get('timeout_warning', False):
            click.echo(
                f"⚠ Warning: Indexing took {results['indexing_time_seconds']:.2f} seconds, "
                f"exceeds timeout of {Config.INDEXING_TIMEOUT_SECONDS} seconds",
                err=True
            )
        
        # Handle hotspots if requested
        if hotspots or hotspots_only:
            detector = LegacyPatternDetector(analyzer)
            detected_hotspots = detector.detect_all_hotspots()
            # Generate explanations (will use LLM if API key available, templates otherwise)
            detector.generate_explanations(detected_hotspots)
            
            if hotspots_only:
                # Only show hotspots
                if output_json:
                    click.echo(_format_hotspots_json(detected_hotspots))
                else:
                    click.echo(_format_hotspots_text(detected_hotspots))
            else:
                # Show summary first, then hotspots
                if output_json:
                    summary = json.loads(analyzer.get_summary_json(top_n))
                    summary['hotspots'] = _hotspots_to_dict(detected_hotspots)
                    click.echo(json.dumps(summary, indent=2))
                else:
                    output = analyzer.get_summary_text(top_n)
                    click.echo(output)
                    click.echo()
                    click.echo("=" * 50)
                    click.echo("Legacy Pattern Hotspots")
                    click.echo("=" * 50)
                    click.echo()
                    click.echo(_format_hotspots_text(detected_hotspots))
        else:
            # Normal summary output
            if output_json:
                output = analyzer.get_summary_json(top_n)
                click.echo(output)
            else:
                output = analyzer.get_summary_text(top_n)
                click.echo(output)
            
    except Exception as e:
        click.echo(f"Error: {e}", err=True)
        raise click.Abort()


@cli.command()
@click.option('--interactive', '-i', is_flag=True, help='Open interactive configuration menu')
def config(interactive: bool):
    """Show or configure RiSpec settings."""
    if interactive:
        _interactive_config()
    else:
        _show_config()


def _show_config():
    """Display current configuration."""
    Config.reload()  # Reload to get latest values
    click.echo("RiSpec Configuration")
    click.echo("=" * 50)
    click.echo(f"Data directory: {Config.DATA_DIR}")
    click.echo(f"Max repo LOC: {Config.MAX_REPO_LOC:,}")
    click.echo(f"Indexing timeout: {Config.INDEXING_TIMEOUT_SECONDS}s")
    click.echo(f"Max patch size: {Config.MAX_PATCH_SIZE_LINES} lines")
    click.echo(f"OpenAI model: {Config.OPENAI_MODEL}")
    click.echo(f"API key configured: {'Yes' if Config.OPENAI_API_KEY else 'No'}")
    click.echo()
    click.echo("Run 'rispec config --interactive' to configure settings")


def _interactive_config():
    """Interactive configuration menu."""
    Config.reload()  # Reload to get latest values
    
    click.echo()
    click.echo("=" * 50)
    click.echo("RiSpec Interactive Configuration")
    click.echo("=" * 50)
    click.echo()
    
    while True:
        click.echo("\nCurrent Configuration:")
        click.echo(f"  OpenAI Model: {Config.OPENAI_MODEL}")
        click.echo(f"  API Key: {'*' * 20 if Config.OPENAI_API_KEY else 'Not set'}")
        click.echo()
        click.echo("Options:")
        click.echo("  1. Set OpenAI API Key")
        click.echo("  2. Select OpenAI Model")
        click.echo("  3. Show full configuration")
        click.echo("  4. Exit")
        click.echo()
        
        choice = click.prompt("Select an option", type=int, default=4)
        
        if choice == 1:
            _set_api_key()
        elif choice == 2:
            _select_model()
        elif choice == 3:
            _show_config()
        elif choice == 4:
            click.echo("\nConfiguration saved. Exiting...")
            break
        else:
            click.echo("Invalid option. Please try again.")


def _set_api_key():
    """Set OpenAI API key."""
    click.echo()
    click.echo("OpenAI API Key Configuration")
    click.echo("-" * 50)
    
    if Config.OPENAI_API_KEY:
        click.echo(f"Current API key: {Config.OPENAI_API_KEY[:8]}...{Config.OPENAI_API_KEY[-4:]}")
        if not click.confirm("Do you want to update the API key?", default=False):
            return
    
    api_key = click.prompt(
        "Enter your OpenAI API key",
        type=str,
        hide_input=True,
        default="" if not Config.OPENAI_API_KEY else Config.OPENAI_API_KEY
    )
    
    if api_key:
        Config.update_env_file("OPENAI_API_KEY", api_key)
        click.echo("✓ API key saved successfully!")
    else:
        click.echo("No API key provided. Skipping...")


def _select_model():
    """Select OpenAI model from available options."""
    click.echo()
    click.echo("OpenAI Model Selection")
    click.echo("-" * 50)
    click.echo()
    click.echo("Available models:")
    
    for i, model in enumerate(Config.OPENAI_MODELS, 1):
        marker = " ← Current" if model == Config.OPENAI_MODEL else ""
        click.echo(f"  {i}. {model}{marker}")
    
    click.echo()
    
    while True:
        try:
            choice = click.prompt(
                f"Select model (1-{len(Config.OPENAI_MODELS)})",
                type=int,
                default=Config.OPENAI_MODELS.index(Config.OPENAI_MODEL) + 1 if Config.OPENAI_MODEL in Config.OPENAI_MODELS else 1
            )
            
            if 1 <= choice <= len(Config.OPENAI_MODELS):
                selected_model = Config.OPENAI_MODELS[choice - 1]
                Config.update_env_file("OPENAI_MODEL", selected_model)
                click.echo(f"✓ Model set to: {selected_model}")
                break
            else:
                click.echo(f"Please enter a number between 1 and {len(Config.OPENAI_MODELS)}")
        except ValueError:
            click.echo("Please enter a valid number")


@cli.command()
@click.argument('repo_path', type=click.Path(exists=True, path_type=Path))
@click.option('--top-n', default=20, help='Number of top hotspots to show')
@click.option('--area', help='Filter by directory path or package name (prefix match)')
@click.option('--json', 'output_json', is_flag=True, help='Output as JSON')
@click.option('--with-explanations', is_flag=True, help='Generate LLM explanations (requires API key)')
def hotspots(repo_path: Path, top_n: int, area: Optional[str], output_json: bool, with_explanations: bool):
    """Detect legacy pattern hotspots in repository.
    
    REPO_PATH: Path to the repository root directory
    """
    try:
        analyzer = RepositoryAnalyzer(repo_path)
        click.echo("Indexing repository...", err=True)
        analyzer.index()
        click.echo("Detecting hotspots...", err=True)
        
        detector = LegacyPatternDetector(analyzer)
        detected_hotspots = detector.detect_all_hotspots(
            filter_area=area,
            top_n=top_n
        )
        
        if with_explanations:
            click.echo("Generating explanations...", err=True)
            detector.generate_explanations(detected_hotspots)
        
        if output_json:
            click.echo(_format_hotspots_json(detected_hotspots))
        else:
            click.echo(_format_hotspots_text(detected_hotspots))
            
    except Exception as e:
        click.echo(f"Error: {e}", err=True)
        raise click.Abort()


def _format_hotspots_text(hotspots) -> str:
    """Format hotspots as human-readable text."""
    if not hotspots:
        return "No hotspots detected."
    
    lines = [f"Found {len(hotspots)} hotspot(s):", ""]
    
    for i, hotspot in enumerate(hotspots, 1):
        lines.append(f"{i}. {hotspot.type.upper().replace('_', ' ')}")
        lines.append(f"   File: {hotspot.file_path}")
        lines.append(f"   Module: {hotspot.module_name}")
        lines.append(f"   Severity Score: {hotspot.severity_score:.2f}")
        
        if hotspot.line_range:
            lines.append(f"   Lines: {hotspot.line_range[0]}-{hotspot.line_range[1]}")
        
        if hotspot.details:
            detail_str = ", ".join(f"{k}: {v}" for k, v in hotspot.details.items())
            lines.append(f"   Details: {detail_str}")
        
        if hotspot.explanation:
            lines.append(f"   Explanation: {hotspot.explanation}")
        
        lines.append("")
    
    return "\n".join(lines)


def _format_hotspots_json(hotspots) -> str:
    """Format hotspots as JSON."""
    return json.dumps(_hotspots_to_dict(hotspots), indent=2)


def _hotspots_to_dict(hotspots) -> List[Dict]:
    """Convert hotspots to dictionary format."""
    result = []
    for hotspot in hotspots:
        hotspot_dict = {
            "type": hotspot.type,
            "file_path": hotspot.file_path,
            "module_name": hotspot.module_name,
            "severity_score": hotspot.severity_score,
            "details": hotspot.details
        }
        if hotspot.line_range:
            hotspot_dict["line_range"] = list(hotspot.line_range)
        if hotspot.explanation:
            hotspot_dict["explanation"] = hotspot.explanation
        result.append(hotspot_dict)
    return result


@cli.group()
def context():
    """Manage context sessions for iterative context retrieval."""
    pass


@context.command()
@click.argument('repo_path', type=click.Path(exists=True, path_type=Path))
@click.argument('task_id')
@click.argument('description')
@click.option('--json', 'output_json', is_flag=True, help='Output as JSON')
def init(repo_path: Path, task_id: str, description: str, output_json: bool):
    """Initialize a new context session for a task.
    
    REPO_PATH: Path to the repository root directory
    TASK_ID: Unique identifier for the task
    DESCRIPTION: Task description or change request
    """
    try:
        analyzer = RepositoryAnalyzer(repo_path)
        click.echo("Indexing repository...", err=True)
        analyzer.index()
        
        manager = ContextManager(analyzer, task_id, description)
        manager.save_session()
        
        context_files = manager.get_context_files()
        
        if output_json:
            result = {
                "task_id": task_id,
                "repo_path": str(repo_path),
                "description": description,
                "context_files": context_files,
                "context_file_count": len(context_files)
            }
            click.echo(json.dumps(result, indent=2))
        else:
            click.echo(f"Context session '{task_id}' initialized.")
            click.echo(f"Identified {len(context_files)} initial context files:")
            for file_path in context_files[:20]:
                click.echo(f"  - {file_path}")
            if len(context_files) > 20:
                click.echo(f"  ... and {len(context_files) - 20} more files")
            click.echo(f"\nSession saved. Use 'context expand' to expand context.")
            
    except Exception as e:
        click.echo(f"Error: {e}", err=True)
        raise click.Abort()


@context.command()
@click.argument('repo_path', type=click.Path(exists=True, path_type=Path))
@click.argument('task_id')
@click.option('--symbols', help='Comma-separated list of symbols to expand for')
@click.option('--llm', 'use_llm', is_flag=True, help='Use LLM to detect missing information')
@click.option('--json', 'output_json', is_flag=True, help='Output as JSON')
def expand(repo_path: Path, task_id: str, symbols: Optional[str], use_llm: bool, output_json: bool):
    """Expand context for a task session.
    
    REPO_PATH: Path to the repository root directory
    TASK_ID: Task identifier
    """
    try:
        analyzer = RepositoryAnalyzer(repo_path)
        analyzer.index()
        
        manager = ContextManager.load_session(analyzer, task_id)
        
        symbols_list = [s.strip() for s in symbols.split(',')] if symbols else None
        llm_suggestion = None
        
        if use_llm:
            click.echo("Detecting missing information with LLM...", err=True)
            llm_suggestion = manager.detect_missing_information()
            if llm_suggestion:
                click.echo(f"LLM suggestion: {llm_suggestion[:100]}...", err=True)
        
        manager.expand_context(symbols=symbols_list, llm_suggestion=llm_suggestion)
        manager.save_session()
        
        context_files = manager.get_context_files()
        
        if output_json:
            result = {
                "task_id": task_id,
                "context_files": context_files,
                "context_file_count": len(context_files),
                "expansion_applied": True
            }
            click.echo(json.dumps(result, indent=2))
        else:
            click.echo(f"Context expanded for task '{task_id}'.")
            click.echo(f"Total context files: {len(context_files)}")
            if symbols_list:
                click.echo(f"Expanded for symbols: {', '.join(symbols_list)}")
            if llm_suggestion:
                click.echo(f"LLM suggested additional files.")
            
    except FileNotFoundError:
        click.echo(f"Error: Session '{task_id}' not found. Use 'context init' first.", err=True)
        raise click.Abort()
    except Exception as e:
        click.echo(f"Error: {e}", err=True)
        raise click.Abort()


@context.command()
@click.argument('repo_path', type=click.Path(exists=True, path_type=Path))
@click.argument('task_id')
@click.option('--json', 'output_json', is_flag=True, help='Output as JSON')
def show(repo_path: Path, task_id: str, output_json: bool):
    """List context files for a task session.
    
    REPO_PATH: Path to the repository root directory
    TASK_ID: Task identifier
    """
    try:
        analyzer = RepositoryAnalyzer(repo_path)
        analyzer.index()
        
        manager = ContextManager.load_session(analyzer, task_id)
        context_files = manager.get_context_files()
        session = manager.session
        
        if output_json:
            result = {
                "task_id": task_id,
                "description": session.description,
                "created_at": session.created_at,
                "current_state": session.current_state,
                "context_files": [
                    {
                        "path": entry.path,
                        "reason": entry.reason,
                        "added_at": entry.added_at,
                        "source": entry.source
                    }
                    for entry in session.context_files
                ],
                "excluded_files": list(session.excluded_files),
                "context_file_count": len(context_files)
            }
            click.echo(json.dumps(result, indent=2))
        else:
            click.echo(f"Context for task '{task_id}':")
            click.echo(f"Description: {session.description}")
            click.echo(f"State: {session.current_state}")
            click.echo(f"Created: {session.created_at}")
            click.echo(f"\nContext Files ({len(context_files)}):")
            for entry in session.context_files:
                click.echo(f"  - {entry.path} [{entry.source}]")
                click.echo(f"    Reason: {entry.reason}")
            if session.excluded_files:
                click.echo(f"\nExcluded Files ({len(session.excluded_files)}):")
                for file_path in sorted(session.excluded_files):
                    click.echo(f"  - {file_path}")
            
    except FileNotFoundError:
        click.echo(f"Error: Session '{task_id}' not found. Use 'context init' first.", err=True)
        raise click.Abort()
    except Exception as e:
        click.echo(f"Error: {e}", err=True)
        raise click.Abort()


@context.command()
@click.argument('repo_path', type=click.Path(exists=True, path_type=Path))
@click.argument('task_id')
@click.argument('file_path')
@click.option('--reason', default='Manually included', help='Reason for inclusion')
def include(repo_path: Path, task_id: str, file_path: str, reason: str):
    """Force include a file or directory in context.
    
    REPO_PATH: Path to the repository root directory
    TASK_ID: Task identifier
    FILE_PATH: File or directory path (relative to repo root)
    """
    try:
        analyzer = RepositoryAnalyzer(repo_path)
        analyzer.index()
        
        manager = ContextManager.load_session(analyzer, task_id)
        manager.force_include(file_path, reason)
        manager.save_session()
        
        click.echo(f"Included '{file_path}' in context for task '{task_id}'.")
        
    except FileNotFoundError:
        click.echo(f"Error: Session '{task_id}' not found. Use 'context init' first.", err=True)
        raise click.Abort()
    except Exception as e:
        click.echo(f"Error: {e}", err=True)
        raise click.Abort()


@context.command()
@click.argument('repo_path', type=click.Path(exists=True, path_type=Path))
@click.argument('task_id')
@click.argument('file_path')
@click.option('--reason', default='Manually excluded', help='Reason for exclusion')
def exclude(repo_path: Path, task_id: str, file_path: str, reason: str):
    """Exclude a file or directory from context.
    
    REPO_PATH: Path to the repository root directory
    TASK_ID: Task identifier
    FILE_PATH: File or directory path (relative to repo root)
    """
    try:
        analyzer = RepositoryAnalyzer(repo_path)
        analyzer.index()
        
        manager = ContextManager.load_session(analyzer, task_id)
        manager.exclude(file_path, reason)
        manager.save_session()
        
        click.echo(f"Excluded '{file_path}' from context for task '{task_id}'.")
        
    except FileNotFoundError:
        click.echo(f"Error: Session '{task_id}' not found. Use 'context init' first.", err=True)
        raise click.Abort()
    except Exception as e:
        click.echo(f"Error: {e}", err=True)
        raise click.Abort()


@context.command()
@click.argument('repo_path', type=click.Path(exists=True, path_type=Path))
@click.argument('task_id')
@click.option('--json', 'output_json', is_flag=True, help='Output as JSON')
def log(repo_path: Path, task_id: str, output_json: bool):
    """Show exploration log for a task session.
    
    REPO_PATH: Path to the repository root directory
    TASK_ID: Task identifier
    """
    try:
        analyzer = RepositoryAnalyzer(repo_path)
        analyzer.index()
        
        manager = ContextManager.load_session(analyzer, task_id)
        exploration_log = manager.get_exploration_log()
        
        if output_json:
            result = {
                "task_id": task_id,
                "exploration_log": exploration_log,
                "log_entry_count": len(exploration_log)
            }
            click.echo(json.dumps(result, indent=2))
        else:
            click.echo(f"Exploration log for task '{task_id}':")
            click.echo(f"Total entries: {len(exploration_log)}\n")
            for entry in exploration_log:
                click.echo(f"[{entry['timestamp']}] {entry['action']}")
                if entry['file_path']:
                    click.echo(f"  File: {entry['file_path']}")
                click.echo(f"  Reason: {entry['reason']}")
                if entry['details']:
                    click.echo(f"  Details: {entry['details']}")
                click.echo()
            
    except FileNotFoundError:
        click.echo(f"Error: Session '{task_id}' not found. Use 'context init' first.", err=True)
        raise click.Abort()
    except Exception as e:
        click.echo(f"Error: {e}", err=True)
        raise click.Abort()


if __name__ == '__main__':
    cli()

