"""Configuration management for RiSpec."""

import os
import re
from pathlib import Path
from dotenv import load_dotenv

# Load environment variables
load_dotenv()


class Config:
    """Application configuration."""
    
    # Available OpenAI models
    OPENAI_MODELS = [
        "gpt-5.1",
        "gpt-4-turbo-preview",
        "gpt-4",
        "gpt-4-32k",
        "gpt-3.5-turbo",
        "gpt-3.5-turbo-16k",
    ]
    
    # OpenAI settings
    OPENAI_API_KEY = os.getenv("OPENAI_API_KEY", "")
    OPENAI_MODEL = os.getenv("OPENAI_MODEL", "gpt-4-turbo-preview")
    
    # Performance thresholds
    MAX_REPO_LOC = int(os.getenv("MAX_REPO_LOC", "1000000"))  # 1M LOC default
    INDEXING_TIMEOUT_SECONDS = int(os.getenv("INDEXING_TIMEOUT_SECONDS", "60"))
    
    # Patch settings
    MAX_PATCH_SIZE_LINES = int(os.getenv("MAX_PATCH_SIZE_LINES", "30"))
    
    # Legacy pattern detection thresholds
    LARGE_FILE_LOC_THRESHOLD = int(os.getenv("LARGE_FILE_LOC_THRESHOLD", "1000"))
    LARGE_METHOD_LOC_THRESHOLD = int(os.getenv("LARGE_METHOD_LOC_THRESHOLD", "100"))
    HIGH_FAN_IN_THRESHOLD = int(os.getenv("HIGH_FAN_IN_THRESHOLD", "10"))
    HIGH_FAN_OUT_THRESHOLD = int(os.getenv("HIGH_FAN_OUT_THRESHOLD", "15"))
    
    # Data storage
    DATA_DIR = Path(os.getenv("DATA_DIR", "data"))
    
    @classmethod
    def ensure_data_dir(cls):
        """Ensure data directory exists."""
        cls.DATA_DIR.mkdir(parents=True, exist_ok=True)
    
    @classmethod
    def get_env_file_path(cls) -> Path:
        """Get the path to the .env file."""
        # Look for .env in the current directory or project root
        current_dir = Path.cwd()
        env_file = current_dir / ".env"
        if env_file.exists():
            return env_file
        
        # Try parent directory (project root)
        parent_env = current_dir.parent / ".env"
        if parent_env.exists():
            return parent_env
        
        # Create in current directory
        return current_dir / ".env"
    
    @classmethod
    def update_env_file(cls, key: str, value: str):
        """Update or add a key-value pair in the .env file."""
        env_file = cls.get_env_file_path()
        
        # Read existing content
        content = ""
        if env_file.exists():
            content = env_file.read_text(encoding='utf-8')
        
        # Pattern to match the key (handles various formats)
        pattern = rf'^{re.escape(key)}\s*=\s*.*$'
        
        # Check if key exists
        if re.search(pattern, content, re.MULTILINE):
            # Replace existing key
            new_content = re.sub(
                pattern,
                f'{key}={value}',
                content,
                flags=re.MULTILINE
            )
        else:
            # Add new key (append if file has content, otherwise just the key)
            if content and not content.endswith('\n'):
                content += '\n'
            new_content = content + f'{key}={value}\n'
        
        # Write back to file
        env_file.write_text(new_content, encoding='utf-8')
        
        # Reload environment variables
        load_dotenv(override=True)
        
        # Update class attributes
        if key == "OPENAI_API_KEY":
            cls.OPENAI_API_KEY = value
        elif key == "OPENAI_MODEL":
            cls.OPENAI_MODEL = value
    
    @classmethod
    def reload(cls):
        """Reload configuration from environment."""
        load_dotenv(override=True)
        cls.OPENAI_API_KEY = os.getenv("OPENAI_API_KEY", "")
        cls.OPENAI_MODEL = os.getenv("OPENAI_MODEL", "gpt-4-turbo-preview")

