"""Legacy pattern detection for identifying code hotspots."""

import ast
from pathlib import Path
from typing import Dict, List, Set, Optional, Tuple
from collections import defaultdict
from dataclasses import dataclass
from rispec.repository import RepositoryAnalyzer
from rispec.config import Config


@dataclass
class MethodInfo:
    """Information about a method/function."""
    name: str
    line_start: int
    line_end: int
    line_count: int
    is_class_method: bool
    class_name: Optional[str] = None


@dataclass
class Hotspot:
    """Represents a code hotspot (legacy pattern)."""
    type: str  # 'large_file', 'large_method', 'high_fan_in', 'high_fan_out'
    file_path: str
    module_name: str
    severity_score: float
    details: Dict
    line_range: Optional[Tuple[int, int]] = None
    explanation: Optional[str] = None


class LegacyPatternDetector:
    """Detects legacy patterns and code hotspots."""
    
    def __init__(self, analyzer: RepositoryAnalyzer):
        """Initialize detector with a repository analyzer.
        
        Args:
            analyzer: RepositoryAnalyzer instance (must be indexed)
        """
        if not analyzer.is_indexed():
            analyzer.index()
        
        self.analyzer = analyzer
        self._method_info: Dict[str, List[MethodInfo]] = defaultdict(list)
        self._function_fan_in: Dict[str, int] = defaultdict(int)
        self._function_fan_out: Dict[str, int] = defaultdict(int)
        self._module_fan_in: Dict[str, int] = defaultdict(int)
        self._module_fan_out: Dict[str, int] = defaultdict(int)
        
        # Build method info and calculate fan-in/fan-out
        self._build_method_info()
        self._calculate_fan_metrics()
    
    def _build_method_info(self):
        """Build method information from AST parsing."""
        for file_path in self.analyzer._source_files:
            if file_path.suffix != '.py':
                continue
            
            try:
                with open(file_path, 'r', encoding='utf-8') as f:
                    content = f.read()
                    tree = ast.parse(content, filename=str(file_path))
                
                methods = self._extract_methods(tree, file_path)
                module_name = self.analyzer._get_module_name(file_path)
                self._method_info[module_name] = methods
            except (SyntaxError, UnicodeDecodeError):
                continue
    
    def _extract_methods(self, tree: ast.AST, file_path: Path) -> List[MethodInfo]:
        """Extract method/function information from AST."""
        methods = []
        lines = []
        
        try:
            with open(file_path, 'r', encoding='utf-8') as f:
                lines = f.readlines()
        except (UnicodeDecodeError, IOError):
            return methods
        
        # Use a visitor to track class context properly
        class MethodVisitor(ast.NodeVisitor):
            def __init__(self, lines, file_path):
                self.lines = lines
                self.file_path = file_path
                self.methods = []
                self.class_stack = []
            
            def visit_ClassDef(self, node):
                self.class_stack.append(node.name)
                self.generic_visit(node)
                self.class_stack.pop()
            
            def visit_FunctionDef(self, node):
                is_class_method = len(self.class_stack) > 0
                class_name = self.class_stack[-1] if is_class_method else None
                
                # Calculate line count
                line_start = node.lineno
                line_end = self._get_function_end_line(node, self.lines)
                line_count = max(1, line_end - line_start + 1)
                
                method_info = MethodInfo(
                    name=node.name,
                    line_start=line_start,
                    line_end=line_end,
                    line_count=line_count,
                    is_class_method=is_class_method,
                    class_name=class_name
                )
                self.methods.append(method_info)
                self.generic_visit(node)
            
            def _get_function_end_line(self, node, lines):
                """Get the end line of a function."""
                if not node.body:
                    return node.lineno
                
                # Find the last statement
                last_node = node.body[-1]
                if hasattr(last_node, 'lineno'):
                    return last_node.lineno
                
                return node.lineno
        
        visitor = MethodVisitor(lines, file_path)
        visitor.visit(tree)
        return visitor.methods
    
    
    def _calculate_fan_metrics(self):
        """Calculate fan-in and fan-out metrics for modules and functions."""
        # Module-level fan-in/fan-out from dependency graph
        for module, imports in self.analyzer._dependency_graph.items():
            # Fan-out: number of modules this module depends on
            self._module_fan_out[module] = len(imports)
            
            # Fan-in: count how many modules depend on this module
            for other_module, other_imports in self.analyzer._dependency_graph.items():
                if module in other_imports:
                    self._module_fan_in[module] += 1
        
        # Function-level fan-in/fan-out from call graph
        for module, calls in self.analyzer._call_graph.items():
            # Fan-out: number of functions this module calls
            self._function_fan_out[module] = len(calls)
            
            # Fan-in: count how many modules call functions in this module
            for other_module, other_calls in self.analyzer._call_graph.items():
                if module != other_module:
                    # Check if any calls match methods in this module
                    module_methods = {m.name for m in self._method_info[module]}
                    if any(call in module_methods for call in other_calls):
                        self._function_fan_in[module] += 1
    
    def detect_large_files(self) -> List[Hotspot]:
        """Detect files that exceed the LOC threshold."""
        hotspots = []
        
        for module_name, info in self.analyzer._module_info.items():
            file_path = Path(self.analyzer.repo_path) / info['path']
            line_count = info['line_count']
            
            if line_count > Config.LARGE_FILE_LOC_THRESHOLD:
                hotspot = Hotspot(
                    type='large_file',
                    file_path=str(info['path']),
                    module_name=module_name,
                    severity_score=line_count / Config.LARGE_FILE_LOC_THRESHOLD,
                    details={
                        'line_count': line_count,
                        'threshold': Config.LARGE_FILE_LOC_THRESHOLD
                    }
                )
                hotspots.append(hotspot)
        
        return hotspots
    
    def detect_large_methods(self) -> List[Hotspot]:
        """Detect methods that exceed the LOC threshold."""
        hotspots = []
        
        for module_name, methods in self._method_info.items():
            if module_name not in self.analyzer._module_info:
                continue
            
            file_path = Path(self.analyzer.repo_path) / self.analyzer._module_info[module_name]['path']
            
            for method in methods:
                if method.line_count > Config.LARGE_METHOD_LOC_THRESHOLD:
                    full_name = f"{method.class_name}.{method.name}" if method.class_name else method.name
                    
                    hotspot = Hotspot(
                        type='large_method',
                        file_path=str(self.analyzer._module_info[module_name]['path']),
                        module_name=module_name,
                        severity_score=method.line_count / Config.LARGE_METHOD_LOC_THRESHOLD,
                        details={
                            'method_name': full_name,
                            'line_count': method.line_count,
                            'threshold': Config.LARGE_METHOD_LOC_THRESHOLD,
                            'is_class_method': method.is_class_method
                        },
                        line_range=(method.line_start, method.line_end)
                    )
                    hotspots.append(hotspot)
        
        return hotspots
    
    def detect_high_fan_in(self, threshold: Optional[int] = None) -> List[Hotspot]:
        """Detect modules/functions with high fan-in."""
        if threshold is None:
            threshold = Config.HIGH_FAN_IN_THRESHOLD
        
        hotspots = []
        
        # Module-level
        for module_name, fan_in in self._module_fan_in.items():
            if fan_in > threshold:
                if module_name not in self.analyzer._module_info:
                    continue
                
                hotspot = Hotspot(
                    type='high_fan_in',
                    file_path=str(self.analyzer._module_info[module_name]['path']),
                    module_name=module_name,
                    severity_score=fan_in / threshold,
                    details={
                        'fan_in': fan_in,
                        'threshold': threshold,
                        'level': 'module'
                    }
                )
                hotspots.append(hotspot)
        
        # Function-level (simplified - using module-level as proxy)
        # TODO: Implement true function-level fan-in when call graph is more detailed
        
        return hotspots
    
    def detect_high_fan_out(self, threshold: Optional[int] = None) -> List[Hotspot]:
        """Detect modules/functions with high fan-out."""
        if threshold is None:
            threshold = Config.HIGH_FAN_OUT_THRESHOLD
        
        hotspots = []
        
        # Module-level
        for module_name, fan_out in self._module_fan_out.items():
            if fan_out > threshold:
                if module_name not in self.analyzer._module_info:
                    continue
                
                hotspot = Hotspot(
                    type='high_fan_out',
                    file_path=str(self.analyzer._module_info[module_name]['path']),
                    module_name=module_name,
                    severity_score=fan_out / threshold,
                    details={
                        'fan_out': fan_out,
                        'threshold': threshold,
                        'level': 'module'
                    }
                )
                hotspots.append(hotspot)
        
        return hotspots
    
    def detect_all_hotspots(
        self,
        filter_area: Optional[str] = None,
        top_n: Optional[int] = None
    ) -> List[Hotspot]:
        """Detect all hotspots and return combined, ranked list.
        
        Args:
            filter_area: Optional directory path or package name (prefix match)
            top_n: Optional limit on number of hotspots to return
            
        Returns:
            List of hotspots sorted by severity score
        """
        all_hotspots = []
        
        # Collect all hotspots
        all_hotspots.extend(self.detect_large_files())
        all_hotspots.extend(self.detect_large_methods())
        all_hotspots.extend(self.detect_high_fan_in())
        all_hotspots.extend(self.detect_high_fan_out())
        
        # Apply area filter if specified
        if filter_area:
            all_hotspots = self._filter_by_area(all_hotspots, filter_area)
        
        # Sort by severity score (descending)
        all_hotspots.sort(key=lambda x: x.severity_score, reverse=True)
        
        # Limit to top N if specified
        if top_n:
            all_hotspots = all_hotspots[:top_n]
        
        return all_hotspots
    
    def _filter_by_area(self, hotspots: List[Hotspot], area: str) -> List[Hotspot]:
        """Filter hotspots by area (directory path or package name).
        
        Args:
            hotspots: List of hotspots to filter
            area: Directory path or package name (prefix match)
            
        Returns:
            Filtered list of hotspots
        """
        filtered = []
        
        for hotspot in hotspots:
            # Check if file path starts with area
            if hotspot.file_path.startswith(area.replace('.', '/')):
                filtered.append(hotspot)
            # Check if module name starts with area
            elif hotspot.module_name.startswith(area):
                filtered.append(hotspot)
        
        return filtered
    
    def get_code_snippet(self, file_path: str, line_start: int, line_end: int, context_lines: int = 5) -> str:
        """Get code snippet with surrounding context.
        
        Args:
            file_path: Path to file (relative to repo root)
            line_start: Start line number
            line_end: End line number
            context_lines: Number of context lines before and after
            
        Returns:
            Code snippet as string
        """
        full_path = self.analyzer.repo_path / file_path
        
        try:
            with open(full_path, 'r', encoding='utf-8') as f:
                lines = f.readlines()
            
            # Adjust line numbers (1-indexed to 0-indexed)
            start_idx = max(0, line_start - 1 - context_lines)
            end_idx = min(len(lines), line_end + context_lines)
            
            snippet_lines = lines[start_idx:end_idx]
            return ''.join(snippet_lines)
        except (IOError, UnicodeDecodeError):
            return ""
    
    def generate_explanations(self, hotspots: List[Hotspot]) -> List[Hotspot]:
        """Generate explanations for hotspots using LLM or templates.
        
        Args:
            hotspots: List of hotspots to explain
            
        Returns:
            List of hotspots with explanations added
        """
        from rispec.llm_explainer import LLMExplainer
        
        explainer = LLMExplainer()
        
        for hotspot in hotspots:
            # Get code snippet if available
            code_snippet = ""
            if hotspot.line_range:
                code_snippet = self.get_code_snippet(
                    hotspot.file_path,
                    hotspot.line_range[0],
                    hotspot.line_range[1]
                )
            
            # Generate explanation
            hotspot.explanation = explainer.explain_hotspot(
                hotspot=hotspot,
                code_snippet=code_snippet
            )
        
        return hotspots

