"""LLM-based explanation generation for code hotspots."""

from typing import Optional
from rispec.config import Config
from rispec.legacy_patterns import Hotspot


class LLMExplainer:
    """Generates explanations for code hotspots using LLM or templates."""
    
    def __init__(self):
        """Initialize explainer."""
        self.use_llm = bool(Config.OPENAI_API_KEY)
    
    def explain_hotspot(self, hotspot: Hotspot, code_snippet: str = "") -> str:
        """Generate explanation for a hotspot.
        
        Args:
            hotspot: The hotspot to explain
            code_snippet: Optional code snippet for context
            
        Returns:
            Explanation string
        """
        if self.use_llm:
            return self._explain_with_llm(hotspot, code_snippet)
        else:
            return self._explain_with_template(hotspot)
    
    def _explain_with_llm(self, hotspot: Hotspot, code_snippet: str) -> str:
        """Generate explanation using LLM."""
        try:
            from openai import OpenAI
            
            client = OpenAI(api_key=Config.OPENAI_API_KEY)
            
            prompt = self._build_prompt(hotspot, code_snippet)
            
            response = client.chat.completions.create(
                model=Config.OPENAI_MODEL,
                messages=[
                    {
                        "role": "system",
                        "content": "You are a senior software engineer analyzing legacy code patterns. Provide concise, actionable explanations of why code hotspots are risky."
                    },
                    {
                        "role": "user",
                        "content": prompt
                    }
                ],
                max_tokens=200,
                temperature=0.3
            )
            
            return response.choices[0].message.content.strip()
        except Exception as e:
            # Fallback to template on error
            return self._explain_with_template(hotspot)
    
    def _build_prompt(self, hotspot: Hotspot, code_snippet: str) -> str:
        """Build prompt for LLM."""
        prompt_parts = [
            f"Analyze this code hotspot:",
            f"Type: {hotspot.type}",
            f"File: {hotspot.file_path}",
            f"Module: {hotspot.module_name}",
            f"Severity Score: {hotspot.severity_score:.2f}",
        ]
        
        if hotspot.details:
            prompt_parts.append(f"Details: {hotspot.details}")
        
        if hotspot.line_range:
            prompt_parts.append(f"Lines: {hotspot.line_range[0]}-{hotspot.line_range[1]}")
        
        if code_snippet:
            prompt_parts.append(f"\nCode snippet:\n```python\n{code_snippet}\n```")
        
        prompt_parts.append(
            "\nProvide a brief explanation (2-3 sentences) of why this is a risky code pattern "
            "and what potential issues it might cause."
        )
        
        return "\n".join(prompt_parts)
    
    def _explain_with_template(self, hotspot: Hotspot) -> str:
        """Generate explanation using template."""
        templates = {
            'large_file': (
                f"This file has {hotspot.details.get('line_count', 0)} lines of code, "
                f"exceeding the threshold of {hotspot.details.get('threshold', 0)} lines. "
                "Large files are harder to understand, test, and maintain. Consider splitting "
                "into smaller, focused modules."
            ),
            'large_method': (
                f"The method '{hotspot.details.get('method_name', 'unknown')}' has "
                f"{hotspot.details.get('line_count', 0)} lines, exceeding the threshold of "
                f"{hotspot.details.get('threshold', 0)} lines. Large methods violate the "
                "Single Responsibility Principle and are difficult to test and debug. "
                "Consider extracting functionality into smaller, focused functions."
            ),
            'high_fan_in': (
                f"This module has a fan-in of {hotspot.details.get('fan_in', 0)}, meaning "
                f"{hotspot.details.get('fan_in', 0)} other modules depend on it. High fan-in "
                "indicates tight coupling and makes the module difficult to change without "
                "affecting many dependent modules. Consider using interfaces or dependency "
                "inversion to reduce coupling."
            ),
            'high_fan_out': (
                f"This module has a fan-out of {hotspot.details.get('fan_out', 0)}, meaning "
                f"it depends on {hotspot.details.get('fan_out', 0)} other modules. High fan-out "
                "indicates the module may be doing too much or has too many responsibilities. "
                "Consider refactoring to reduce dependencies and improve cohesion."
            )
        }
        
        template = templates.get(hotspot.type, "This code pattern may indicate technical debt.")
        return template

