"""Tests for CLI commands."""

import pytest
from click.testing import CliRunner
from pathlib import Path
import tempfile
import shutil
import json
import os
from rispec.cli import cli
from rispec.config import Config


@pytest.fixture
def sample_repo():
    """Create a temporary sample repository for testing."""
    repo_dir = tempfile.mkdtemp()
    repo_path = Path(repo_dir)
    
    # Create sample Python files
    (repo_path / "main.py").write_text("""
import os
from utils.helper import helper_function

def main():
    helper_function()
    print("Hello")
""")
    
    (repo_path / "utils").mkdir()
    (repo_path / "utils" / "__init__.py").write_text("")
    (repo_path / "utils" / "helper.py").write_text("""
import os

def helper_function():
    return "help"
""")
    
    yield repo_path
    
    # Cleanup
    shutil.rmtree(repo_dir)


@pytest.fixture
def runner():
    """Create a CLI runner for testing."""
    return CliRunner()


def test_cli_help(runner):
    """Test CLI help command."""
    result = runner.invoke(cli, ['--help'])
    assert result.exit_code == 0
    assert "RiSpec" in result.output
    assert "analyze" in result.output
    assert "config" in result.output


def test_cli_version(runner):
    """Test CLI version command."""
    result = runner.invoke(cli, ['--version'])
    assert result.exit_code == 0
    assert "0.1.0" in result.output


def test_analyze_command_help(runner):
    """Test analyze command help."""
    result = runner.invoke(cli, ['analyze', '--help'])
    assert result.exit_code == 0
    assert "REPO_PATH" in result.output
    assert "--top-n" in result.output
    assert "--json" in result.output


def test_analyze_command_success(runner, sample_repo):
    """Test analyze command with valid repository."""
    result = runner.invoke(cli, ['analyze', str(sample_repo)])
    
    assert result.exit_code == 0
    assert "Indexing repository" in result.output or "Indexed" in result.output
    assert "Repository Analysis Summary" in result.output
    assert "Total source files" in result.output
    assert "Total modules" in result.output


def test_analyze_command_with_top_n(runner, sample_repo):
    """Test analyze command with --top-n option."""
    result = runner.invoke(cli, ['analyze', str(sample_repo), '--top-n', '5'])
    
    assert result.exit_code == 0
    assert "Repository Analysis Summary" in result.output


def test_analyze_command_json_output(runner, sample_repo):
    """Test analyze command with --json flag."""
    result = runner.invoke(cli, ['analyze', str(sample_repo), '--json'])
    
    assert result.exit_code == 0
    # Try to parse as JSON
    output_lines = result.output.strip().split('\n')
    # Find JSON output (skip stderr messages)
    json_start = None
    for i, line in enumerate(output_lines):
        if line.strip().startswith('{'):
            json_start = i
            break
    
    if json_start is not None:
        json_output = '\n'.join(output_lines[json_start:])
        data = json.loads(json_output)
        assert "total_files" in data
        assert "total_modules" in data
        assert "top_referenced_modules" in data
        assert "key_modules" in data


def test_analyze_command_nonexistent_path(runner):
    """Test analyze command with nonexistent path."""
    result = runner.invoke(cli, ['analyze', '/nonexistent/path/12345'])
    
    assert result.exit_code != 0
    assert "Error" in result.output or "does not exist" in result.output


def test_analyze_command_invalid_path_type(runner):
    """Test analyze command with invalid path."""
    result = runner.invoke(cli, ['analyze', '--top-n', '5'])
    
    # Should fail because repo_path is required
    assert result.exit_code != 0


def test_config_command(runner):
    """Test config command."""
    result = runner.invoke(cli, ['config'])
    
    assert result.exit_code == 0
    assert "RiSpec Configuration" in result.output
    assert "Data directory" in result.output
    assert "Max repo LOC" in result.output
    assert "Indexing timeout" in result.output
    assert "Max patch size" in result.output
    assert "OpenAI model" in result.output
    assert "API key configured" in result.output
    assert "interactive" in result.output.lower()  # Should mention interactive option


def test_config_command_output_format(runner):
    """Test config command output format."""
    result = runner.invoke(cli, ['config'])
    
    assert result.exit_code == 0
    # Check that separator line is present
    assert "=" * 50 in result.output
    # Check that all expected config values are shown
    assert str(Config.DATA_DIR) in result.output
    # MAX_REPO_LOC is formatted with commas, so check for formatted version
    assert "1,000,000" in result.output or str(Config.MAX_REPO_LOC) in result.output
    assert str(Config.INDEXING_TIMEOUT_SECONDS) in result.output
    assert str(Config.MAX_PATCH_SIZE_LINES) in result.output
    assert Config.OPENAI_MODEL in result.output


def test_config_command_interactive_flag(runner):
    """Test config command with --interactive flag shows menu."""
    # Use input to select exit option (4)
    result = runner.invoke(cli, ['config', '--interactive'], input='4\n')
    
    assert result.exit_code == 0
    assert "Interactive Configuration" in result.output
    assert "Set OpenAI API Key" in result.output
    assert "Select OpenAI Model" in result.output
    assert "Exit" in result.output


def test_config_command_interactive_model_selection(runner):
    """Test interactive model selection."""
    # Select option 2 (model selection), then select model 1, then exit (4)
    result = runner.invoke(
        cli, 
        ['config', '--interactive'], 
        input='2\n1\n4\n'
    )
    
    assert result.exit_code == 0
    assert "OpenAI Model Selection" in result.output
    assert "Available models:" in result.output
    assert "gpt-4-turbo-preview" in result.output or "gpt-4" in result.output


def test_analyze_command_empty_repo(runner):
    """Test analyze command with empty repository."""
    # Create an empty temporary directory
    empty_dir = tempfile.mkdtemp()
    try:
        result = runner.invoke(cli, ['analyze', empty_dir])
        
        # Should succeed but with 0 files
        assert result.exit_code == 0
        assert "0 files" in result.output or "0 modules" in result.output
    finally:
        shutil.rmtree(empty_dir)


def test_analyze_command_stderr_messages(runner, sample_repo):
    """Test that progress messages go to stderr."""
    result = runner.invoke(cli, ['analyze', str(sample_repo)])
    
    assert result.exit_code == 0
    # The indexing message should be in output (Click's CliRunner captures both)
    # But we can check that the summary is in the output
    assert "Repository Analysis Summary" in result.output


def test_analyze_command_combined_options(runner, sample_repo):
    """Test analyze command with multiple options."""
    result = runner.invoke(cli, [
        'analyze',
        str(sample_repo),
        '--top-n', '3',
        '--json'
    ])
    
    assert result.exit_code == 0
    # Should output JSON
    try:
        output_lines = result.output.strip().split('\n')
        json_start = None
        for i, line in enumerate(output_lines):
            if line.strip().startswith('{'):
                json_start = i
                break
        
        if json_start is not None:
            json_output = '\n'.join(output_lines[json_start:])
            data = json.loads(json_output)
            # Check that top_n was respected (should have at most 3 top referenced)
            assert len(data.get('top_referenced_modules', [])) <= 3
    except json.JSONDecodeError:
        # If JSON parsing fails, that's okay for this test
        pass


def test_hotspots_command(runner, sample_repo):
    """Test hotspots command."""
    result = runner.invoke(cli, ['hotspots', str(sample_repo)])
    
    assert result.exit_code == 0
    assert "Detecting hotspots" in result.output or "hotspot" in result.output.lower()


def test_hotspots_command_json(runner, sample_repo):
    """Test hotspots command with JSON output."""
    result = runner.invoke(cli, ['hotspots', str(sample_repo), '--json'])
    
    assert result.exit_code == 0
    # Try to parse as JSON
    output_lines = result.output.strip().split('\n')
    json_start = None
    for i, line in enumerate(output_lines):
        if line.strip().startswith('[') or line.strip().startswith('{'):
            json_start = i
            break
    
    if json_start is not None:
        json_output = '\n'.join(output_lines[json_start:])
        data = json.loads(json_output)
        assert isinstance(data, list) or isinstance(data, dict)


def test_hotspots_command_with_area_filter(runner, sample_repo):
    """Test hotspots command with area filter."""
    result = runner.invoke(cli, ['hotspots', str(sample_repo), '--area', 'utils'])
    
    assert result.exit_code == 0


def test_analyze_command_with_hotspots(runner, sample_repo):
    """Test analyze command with --hotspots option."""
    result = runner.invoke(cli, ['analyze', str(sample_repo), '--hotspots'])
    
    assert result.exit_code == 0
    # Should include both summary and hotspots
    assert "Repository Analysis Summary" in result.output or "hotspot" in result.output.lower()


def test_analyze_command_hotspots_only(runner, sample_repo):
    """Test analyze command with --hotspots-only option."""
    result = runner.invoke(cli, ['analyze', str(sample_repo), '--hotspots-only'])
    
    assert result.exit_code == 0
    # Should only show hotspots, not summary
    assert "Repository Analysis Summary" not in result.output or "hotspot" in result.output.lower()

