"""Tests for context manager."""

import pytest
import json
import tempfile
import shutil
from pathlib import Path
from rispec.repository import RepositoryAnalyzer
from rispec.context_manager import ContextManager, ContextEntry, ExplorationLog, TaskSession
from rispec.config import Config


@pytest.fixture
def sample_repo(tmp_path):
    """Create a sample repository structure."""
    repo = tmp_path / "test_repo"
    repo.mkdir()
    
    # Create sample Python files
    (repo / "main.py").write_text("""
import utils
from models import User

def main():
    user = User()
    utils.process(user)
""")
    
    (repo / "utils.py").write_text("""
def process(obj):
    pass
""")
    
    (repo / "models.py").write_text("""
class User:
    def __init__(self):
        pass
""")
    
    (repo / "config.py").write_text("""
SETTINGS = {}
""")
    
    return repo


@pytest.fixture
def analyzer(sample_repo):
    """Create and index a repository analyzer."""
    analyzer = RepositoryAnalyzer(sample_repo)
    analyzer.index()
    return analyzer


def test_context_manager_initialization(analyzer):
    """Test context manager initialization."""
    manager = ContextManager(analyzer, "task1", "Add user authentication")
    
    assert manager.task_id == "task1"
    assert manager.description == "Add user authentication"
    # Initial context may be empty if keywords don't match, which is valid
    assert isinstance(manager.session.context_files, list)
    assert manager.session.current_state == 'ready'


def test_initial_context_identification(analyzer):
    """Test initial context identification based on keywords."""
    manager = ContextManager(analyzer, "task1", "Update user model")
    
    # Should identify files with "user" or "model" in path/name
    context_files = manager.get_context_files()
    assert any("user" in f.lower() or "model" in f.lower() for f in context_files)


def test_keyword_extraction(tmp_path):
    """Test keyword extraction from description."""
    # Create a minimal repo for analyzer
    repo = tmp_path / "test_repo"
    repo.mkdir()
    (repo / "dummy.py").write_text("pass")
    
    analyzer = RepositoryAnalyzer(repo)
    analyzer.index()
    
    manager = ContextManager(analyzer, "task1", "Add user authentication feature")
    keywords = manager._extract_keywords("Add user authentication feature")
    
    assert "user" in keywords
    assert "authentication" in keywords
    assert "feature" in keywords
    # Note: "add" might be included if not in stop words, which is fine


def test_symbol_extraction(tmp_path):
    """Test symbol extraction from description."""
    # Create a minimal repo for analyzer
    repo = tmp_path / "test_repo"
    repo.mkdir()
    (repo / "dummy.py").write_text("pass")
    
    analyzer = RepositoryAnalyzer(repo)
    analyzer.index()
    
    manager = ContextManager(analyzer, "task1", "Update User class and process function")
    symbols = manager._extract_symbols("Update User class and process function")
    
    assert "User" in symbols
    assert "process" in symbols


def test_file_symbol_extraction(analyzer):
    """Test symbol extraction from files."""
    manager = ContextManager(analyzer, "task1", "Test")
    
    models_file = analyzer.repo_path / "models.py"
    symbols = manager._extract_file_symbols(models_file)
    
    assert "User" in symbols


def test_add_to_context(analyzer):
    """Test adding files to context."""
    manager = ContextManager(analyzer, "task1", "Test")
    initial_count = len(manager.session.context_files)
    
    manager._add_to_context("new_file.py", "Test reason", "manual")
    
    assert len(manager.session.context_files) == initial_count + 1
    assert manager.session.context_files[-1].path == "new_file.py"
    assert manager.session.context_files[-1].reason == "Test reason"
    assert manager.session.context_files[-1].source == "manual"


def test_exclude_file(analyzer):
    """Test excluding files from context."""
    manager = ContextManager(analyzer, "task1", "Test")
    
    # Add a file first
    manager._add_to_context("test_file.py", "Test", "initial")
    assert "test_file.py" in manager.get_context_files()
    
    # Exclude it
    manager.exclude("test_file.py", "Not needed")
    
    assert "test_file.py" not in manager.get_context_files()
    assert "test_file.py" in manager.session.excluded_files


def test_force_include(analyzer):
    """Test force including files."""
    manager = ContextManager(analyzer, "task1", "Test")
    initial_count = len(manager.session.context_files)
    
    manager.force_include("config.py", "Needed for config")
    
    assert len(manager.session.context_files) > initial_count
    assert any(entry.path == "config.py" and entry.source == "manual" 
               for entry in manager.session.context_files)


def test_expand_for_symbols(analyzer):
    """Test context expansion for symbols."""
    manager = ContextManager(analyzer, "task1", "Test")
    initial_count = len(manager.session.context_files)
    
    manager.expand_context(symbols=["User", "process"])
    
    # Should have added files containing these symbols
    assert len(manager.session.context_files) >= initial_count


def test_exploration_logging(analyzer):
    """Test exploration logging."""
    manager = ContextManager(analyzer, "task1", "Test")
    
    initial_log_count = len(manager.session.exploration_log)
    
    manager._add_to_context("test.py", "Test", "manual")
    
    assert len(manager.session.exploration_log) > initial_log_count
    assert manager.session.exploration_log[-1].action == "file_added"


def test_save_and_load_session(analyzer, tmp_path):
    """Test saving and loading sessions."""
    Config.DATA_DIR = tmp_path / "data"
    Config.ensure_data_dir()
    
    # Create and save session
    manager1 = ContextManager(analyzer, "task_save", "Test description")
    manager1.force_include("test.py", "Test")
    manager1.save_session()
    
    # Load session
    manager2 = ContextManager.load_session(analyzer, "task_save")
    
    assert manager2.task_id == "task_save"
    assert manager2.description == "Test description"
    assert len(manager2.session.context_files) == len(manager1.session.context_files)
    assert any(entry.path == "test.py" for entry in manager2.session.context_files)


def test_load_nonexistent_session(analyzer):
    """Test loading non-existent session raises error."""
    with pytest.raises(FileNotFoundError):
        ContextManager.load_session(analyzer, "nonexistent_task")


def test_get_context_files(analyzer):
    """Test getting context files list."""
    manager = ContextManager(analyzer, "task1", "Test")
    
    context_files = manager.get_context_files()
    
    assert isinstance(context_files, list)
    assert all(isinstance(f, str) for f in context_files)


def test_get_exploration_log(analyzer):
    """Test getting exploration log."""
    manager = ContextManager(analyzer, "task1", "Test")
    
    log = manager.get_exploration_log()
    
    assert isinstance(log, list)
    assert len(log) > 0
    assert all(isinstance(entry, dict) for entry in log)


def test_expand_from_llm_suggestion(analyzer):
    """Test expanding context from LLM suggestion."""
    manager = ContextManager(analyzer, "task1", "Test")
    initial_count = len(manager.session.context_files)
    
    suggestion = "You should include utils.py and config.py"
    manager._expand_from_llm_suggestion(suggestion)
    
    # Should have added files mentioned in suggestion
    context_files = manager.get_context_files()
    assert len(context_files) >= initial_count


def test_detect_missing_information_no_api_key(analyzer, monkeypatch):
    """Test missing information detection without API key."""
    monkeypatch.setattr(Config, "OPENAI_API_KEY", "")
    
    manager = ContextManager(analyzer, "task1", "Test")
    result = manager.detect_missing_information()
    
    assert result is None


def test_exclude_directory(analyzer):
    """Test excluding a directory."""
    manager = ContextManager(analyzer, "task1", "Test")
    
    # Add some files first
    manager._add_to_context("dir1/file1.py", "Test", "initial")
    manager._add_to_context("dir1/file2.py", "Test", "initial")
    
    # Exclude directory
    manager.exclude("dir1/", "Not needed")
    
    # Files in directory should be excluded
    context_files = manager.get_context_files()
    assert "dir1/file1.py" not in context_files
    assert "dir1/file2.py" not in context_files


def test_force_include_directory(analyzer):
    """Test force including a directory."""
    manager = ContextManager(analyzer, "task1", "Test")
    initial_count = len(manager.session.context_files)
    
    # Force include directory (should include all .py files in it)
    manager.force_include("tests/", "Need test files")
    
    # Should have added files
    assert len(manager.session.context_files) >= initial_count


def test_context_entry_duplicate_prevention(analyzer):
    """Test that duplicate files are not added to context."""
    manager = ContextManager(analyzer, "task1", "Test")
    
    manager._add_to_context("test.py", "First", "initial")
    initial_count = len(manager.session.context_files)
    
    # Try to add same file again
    manager._add_to_context("test.py", "Second", "manual")
    
    # Should not have added duplicate
    assert len(manager.session.context_files) == initial_count


def test_context_expansion_state(analyzer):
    """Test context expansion state changes."""
    manager = ContextManager(analyzer, "task1", "Test")
    
    assert manager.session.current_state == 'ready'
    
    # Expansion should change state temporarily
    manager.expand_context(symbols=["User"])
    
    assert manager.session.current_state == 'ready'


def test_session_serialization(analyzer, tmp_path):
    """Test session serialization to JSON."""
    Config.DATA_DIR = tmp_path / "data"
    Config.ensure_data_dir()
    
    manager = ContextManager(analyzer, "task_serial", "Test")
    manager.force_include("test.py", "Test")
    manager.exclude("excluded.py", "Not needed")
    manager.save_session()
    
    # Verify file exists and is valid JSON
    session_file = Config.DATA_DIR / "task_task_serial.json"
    assert session_file.exists()
    
    with open(session_file, 'r') as f:
        data = json.load(f)
    
    assert data['task_id'] == "task_serial"
    assert data['description'] == "Test"
    assert len(data['context_files']) > 0
    assert "excluded.py" in data['excluded_files']

