"""Tests for legacy pattern detection."""

import pytest
from pathlib import Path
import tempfile
import shutil
from rispec.repository import RepositoryAnalyzer
from rispec.legacy_patterns import LegacyPatternDetector, Hotspot, MethodInfo
from rispec.config import Config


@pytest.fixture
def sample_repo_with_hotspots():
    """Create a temporary repository with various hotspots."""
    repo_dir = tempfile.mkdtemp()
    repo_path = Path(repo_dir)
    
    # Create a large file (exceeds threshold)
    large_file_content = "\n".join([f"# Line {i}" for i in range(1, 1200)]) + "\n"
    (repo_path / "large_file.py").write_text(large_file_content)
    
    # Create a file with large method
    large_method_content = """
def small_function():
    return 1

def very_large_method():
    # This method has many lines
""" + "\n".join([f"    x = {i}" for i in range(1, 150)]) + """
    return x

def another_small_function():
    return 2
"""
    (repo_path / "large_method.py").write_text(large_method_content)
    
    # Create files with dependencies for fan-in/fan-out testing
    (repo_path / "popular_module.py").write_text("""
def useful_function():
    return "useful"
""")
    
    (repo_path / "dependent1.py").write_text("""
from popular_module import useful_function
""")
    
    (repo_path / "dependent2.py").write_text("""
from popular_module import useful_function
""")
    
    (repo_path / "dependent3.py").write_text("""
from popular_module import useful_function
""")
    
    (repo_path / "high_fan_out.py").write_text("""
import os
import sys
import json
import re
import ast
import pathlib
import tempfile
import shutil
import pytest
import click
from typing import Dict, List, Set, Optional
from collections import defaultdict
from dataclasses import dataclass
""")
    
    yield repo_path
    
    # Cleanup
    shutil.rmtree(repo_dir)


def test_legacy_pattern_detector_initialization(sample_repo_with_hotspots):
    """Test LegacyPatternDetector initialization."""
    analyzer = RepositoryAnalyzer(sample_repo_with_hotspots)
    analyzer.index()
    
    detector = LegacyPatternDetector(analyzer)
    assert detector.analyzer == analyzer
    assert len(detector._method_info) > 0


def test_detect_large_files(sample_repo_with_hotspots):
    """Test detection of large files."""
    analyzer = RepositoryAnalyzer(sample_repo_with_hotspots)
    analyzer.index()
    
    detector = LegacyPatternDetector(analyzer)
    hotspots = detector.detect_large_files()
    
    # Should detect large_file.py
    large_file_hotspots = [h for h in hotspots if 'large_file' in h.file_path]
    assert len(large_file_hotspots) > 0
    assert large_file_hotspots[0].type == 'large_file'
    assert large_file_hotspots[0].severity_score > 1.0


def test_detect_large_methods(sample_repo_with_hotspots):
    """Test detection of large methods."""
    analyzer = RepositoryAnalyzer(sample_repo_with_hotspots)
    analyzer.index()
    
    detector = LegacyPatternDetector(analyzer)
    hotspots = detector.detect_large_methods()
    
    # Should detect very_large_method
    large_method_hotspots = [h for h in hotspots if 'very_large_method' in str(h.details.get('method_name', ''))]
    assert len(large_method_hotspots) > 0
    assert large_method_hotspots[0].type == 'large_method'
    assert large_method_hotspots[0].line_range is not None


def test_detect_high_fan_in(sample_repo_with_hotspots):
    """Test detection of high fan-in."""
    analyzer = RepositoryAnalyzer(sample_repo_with_hotspots)
    analyzer.index()
    
    detector = LegacyPatternDetector(analyzer)
    hotspots = detector.detect_high_fan_in(threshold=2)  # Lower threshold for test
    
    # Should detect popular_module (has 3 dependents)
    popular_hotspots = [h for h in hotspots if 'popular_module' in h.module_name]
    assert len(popular_hotspots) > 0
    assert popular_hotspots[0].type == 'high_fan_in'


def test_detect_high_fan_out(sample_repo_with_hotspots):
    """Test detection of high fan-out."""
    analyzer = RepositoryAnalyzer(sample_repo_with_hotspots)
    analyzer.index()
    
    detector = LegacyPatternDetector(analyzer)
    hotspots = detector.detect_high_fan_out(threshold=5)  # Lower threshold for test
    
    # Should detect high_fan_out module
    fan_out_hotspots = [h for h in hotspots if 'high_fan_out' in h.module_name]
    assert len(fan_out_hotspots) > 0
    assert fan_out_hotspots[0].type == 'high_fan_out'


def test_detect_all_hotspots(sample_repo_with_hotspots):
    """Test detect_all_hotspots method."""
    analyzer = RepositoryAnalyzer(sample_repo_with_hotspots)
    analyzer.index()
    
    detector = LegacyPatternDetector(analyzer)
    hotspots = detector.detect_all_hotspots()
    
    assert len(hotspots) > 0
    # Should be sorted by severity score
    scores = [h.severity_score for h in hotspots]
    assert scores == sorted(scores, reverse=True)


def test_filter_by_area(sample_repo_with_hotspots):
    """Test filtering hotspots by area."""
    analyzer = RepositoryAnalyzer(sample_repo_with_hotspots)
    analyzer.index()
    
    detector = LegacyPatternDetector(analyzer)
    all_hotspots = detector.detect_all_hotspots()
    
    # Filter by file path prefix
    filtered = detector._filter_by_area(all_hotspots, "large")
    assert len(filtered) <= len(all_hotspots)
    assert all("large" in h.file_path or "large" in h.module_name for h in filtered)


def test_get_code_snippet(sample_repo_with_hotspots):
    """Test getting code snippet."""
    analyzer = RepositoryAnalyzer(sample_repo_with_hotspots)
    analyzer.index()
    
    detector = LegacyPatternDetector(analyzer)
    hotspots = detector.detect_large_methods()
    
    if hotspots:
        hotspot = hotspots[0]
        snippet = detector.get_code_snippet(
            hotspot.file_path,
            hotspot.line_range[0],
            hotspot.line_range[1],
            context_lines=2
        )
        assert isinstance(snippet, str)
        assert len(snippet) > 0


def test_generate_explanations_template(sample_repo_with_hotspots):
    """Test explanation generation with templates (no API key)."""
    # Temporarily clear API key
    original_key = Config.OPENAI_API_KEY
    Config.OPENAI_API_KEY = ""
    
    try:
        analyzer = RepositoryAnalyzer(sample_repo_with_hotspots)
        analyzer.index()
        
        detector = LegacyPatternDetector(analyzer)
        hotspots = detector.detect_large_files()[:1]  # Just one for testing
        
        detector.generate_explanations(hotspots)
        
        assert hotspots[0].explanation is not None
        assert len(hotspots[0].explanation) > 0
        assert "large file" in hotspots[0].explanation.lower() or "lines" in hotspots[0].explanation.lower()
    finally:
        Config.OPENAI_API_KEY = original_key


def test_method_info_extraction(sample_repo_with_hotspots):
    """Test method information extraction."""
    analyzer = RepositoryAnalyzer(sample_repo_with_hotspots)
    analyzer.index()
    
    detector = LegacyPatternDetector(analyzer)
    
    # Check that methods were extracted
    assert len(detector._method_info) > 0
    
    # Check that large_method.py has methods
    for module_name, methods in detector._method_info.items():
        if 'large_method' in module_name:
            assert len(methods) > 0
            # Should have very_large_method
            method_names = [m.name for m in methods]
            assert 'very_large_method' in method_names


def test_hotspot_severity_scoring(sample_repo_with_hotspots):
    """Test that severity scores are calculated correctly."""
    analyzer = RepositoryAnalyzer(sample_repo_with_hotspots)
    analyzer.index()
    
    detector = LegacyPatternDetector(analyzer)
    hotspots = detector.detect_all_hotspots()
    
    for hotspot in hotspots:
        assert hotspot.severity_score > 0
        assert isinstance(hotspot.severity_score, (int, float))


def test_detect_all_hotspots_with_top_n(sample_repo_with_hotspots):
    """Test detect_all_hotspots with top_n limit."""
    analyzer = RepositoryAnalyzer(sample_repo_with_hotspots)
    analyzer.index()
    
    detector = LegacyPatternDetector(analyzer)
    hotspots = detector.detect_all_hotspots(top_n=3)
    
    assert len(hotspots) <= 3

