"""Tests for repository analysis."""

import pytest
from pathlib import Path
import tempfile
import shutil
import time
from rispec.repository import RepositoryAnalyzer
from rispec.config import Config


@pytest.fixture
def sample_repo():
    """Create a temporary sample repository for testing."""
    repo_dir = tempfile.mkdtemp()
    repo_path = Path(repo_dir)
    
    # Create sample Python files
    (repo_path / "main.py").write_text("""
import os
import sys
from utils.helper import helper_function

def main():
    helper_function()
    print("Hello")
""")
    
    (repo_path / "utils").mkdir()
    (repo_path / "utils" / "__init__.py").write_text("")
    (repo_path / "utils" / "helper.py").write_text("""
import os

def helper_function():
    return "help"
""")
    
    (repo_path / "tests").mkdir()
    (repo_path / "tests" / "test_main.py").write_text("""
import pytest
from main import main

def test_main():
    assert True
""")
    
    yield repo_path
    
    # Cleanup
    shutil.rmtree(repo_dir)


def test_repository_analyzer_initialization(sample_repo):
    """Test RepositoryAnalyzer initialization."""
    analyzer = RepositoryAnalyzer(sample_repo)
    assert analyzer.repo_path == Path(sample_repo).resolve()
    assert not analyzer.is_indexed()


def test_repository_analyzer_nonexistent_path():
    """Test RepositoryAnalyzer with nonexistent path."""
    with pytest.raises(ValueError, match="does not exist"):
        RepositoryAnalyzer(Path("/nonexistent/path"))


def test_find_source_files(sample_repo):
    """Test finding source files."""
    analyzer = RepositoryAnalyzer(sample_repo)
    analyzer.index()
    
    source_files = analyzer._source_files
    assert len(source_files) >= 3  # main.py, helper.py, test_main.py
    
    file_names = [f.name for f in source_files]
    assert "main.py" in file_names
    assert "helper.py" in file_names


def test_build_dependency_graph(sample_repo):
    """Test building dependency graph."""
    analyzer = RepositoryAnalyzer(sample_repo)
    analyzer.index()
    
    graph = analyzer.get_dependency_graph()
    assert len(graph) > 0
    
    # Check that main.py has imports
    main_module = None
    for module in graph.keys():
        if "main" in module:
            main_module = module
            break
    
    if main_module:
        assert len(graph[main_module]) > 0


def test_get_summary(sample_repo):
    """Test getting summary."""
    analyzer = RepositoryAnalyzer(sample_repo)
    analyzer.index()
    
    summary = analyzer.get_summary()
    assert "total_files" in summary
    assert "total_modules" in summary
    assert "top_referenced_modules" in summary
    assert "key_modules" in summary
    assert summary["total_files"] > 0
    assert summary["total_modules"] > 0


def test_get_summary_text(sample_repo):
    """Test getting text summary."""
    analyzer = RepositoryAnalyzer(sample_repo)
    analyzer.index()
    
    text = analyzer.get_summary_text()
    assert isinstance(text, str)
    assert "Repository Analysis Summary" in text
    assert "Total source files" in text


def test_get_summary_json(sample_repo):
    """Test getting JSON summary."""
    analyzer = RepositoryAnalyzer(sample_repo)
    analyzer.index()
    
    json_str = analyzer.get_summary_json()
    assert isinstance(json_str, str)
    # Should be valid JSON
    import json
    data = json.loads(json_str)
    assert "total_files" in data


def test_indexing_results(sample_repo):
    """Test indexing returns proper results."""
    analyzer = RepositoryAnalyzer(sample_repo)
    results = analyzer.index()
    
    assert results["indexed"] is True
    assert results["total_files"] > 0
    assert results["total_modules"] > 0
    assert analyzer.is_indexed()


def test_ignore_directories(sample_repo):
    """Test that ignored directories are not indexed."""
    # Create a file in a directory that should be ignored
    (sample_repo / "__pycache__").mkdir()
    (sample_repo / "__pycache__" / "test.pyc").write_text("")
    
    analyzer = RepositoryAnalyzer(sample_repo)
    analyzer.index()
    
    # __pycache__ files should not be in source files
    cache_files = [f for f in analyzer._source_files if "__pycache__" in str(f)]
    assert len(cache_files) == 0


def test_indexing_time_measurement(sample_repo):
    """Test that indexing time is measured."""
    analyzer = RepositoryAnalyzer(sample_repo)
    results = analyzer.index()
    
    assert "indexing_time_seconds" in results
    assert isinstance(results["indexing_time_seconds"], (int, float))
    assert results["indexing_time_seconds"] >= 0


def test_indexing_stats(sample_repo):
    """Test get_indexing_stats method."""
    analyzer = RepositoryAnalyzer(sample_repo)
    analyzer.index()
    
    stats = analyzer.get_indexing_stats()
    assert "indexing_time_seconds" in stats
    assert "total_loc" in stats
    assert "timeout_warning" in stats
    assert "exceeds_loc_threshold" in stats
    assert isinstance(stats["total_loc"], int)
    assert stats["total_loc"] > 0


def test_loc_pre_check(sample_repo):
    """Test that LOC is calculated before indexing."""
    analyzer = RepositoryAnalyzer(sample_repo)
    analyzer.index()
    
    stats = analyzer.get_indexing_stats()
    assert stats["total_loc"] is not None
    assert stats["total_loc"] > 0


def test_loc_threshold_warning(sample_repo):
    """Test LOC threshold warning detection."""
    # Temporarily set a very low threshold
    original_threshold = Config.MAX_REPO_LOC
    Config.MAX_REPO_LOC = 10  # Very low threshold
    
    try:
        analyzer = RepositoryAnalyzer(sample_repo)
        analyzer.index()
        
        stats = analyzer.get_indexing_stats()
        # Should detect that LOC exceeds threshold
        assert stats["exceeds_loc_threshold"] is True or stats["exceeds_loc_threshold"] is False
    finally:
        Config.MAX_REPO_LOC = original_threshold


def test_timeout_warning(sample_repo):
    """Test timeout warning detection."""
    # Temporarily set a very low timeout
    original_timeout = Config.INDEXING_TIMEOUT_SECONDS
    Config.INDEXING_TIMEOUT_SECONDS = 0.001  # Very low timeout
    
    try:
        analyzer = RepositoryAnalyzer(sample_repo)
        # Add a small delay to ensure timeout
        time.sleep(0.01)
        results = analyzer.index()
        
        # Should have timeout warning if indexing took too long
        # Note: This might not always trigger on fast systems
        assert "timeout_warning" in results
    finally:
        Config.INDEXING_TIMEOUT_SECONDS = original_timeout


def test_indexing_results_include_timing(sample_repo):
    """Test that index() results include timing information."""
    analyzer = RepositoryAnalyzer(sample_repo)
    results = analyzer.index()
    
    assert "indexing_time_seconds" in results
    assert "total_loc" in results
    assert "timeout_warning" in results
    assert results["indexed"] is True

