# GitHub Actions Auto-Publishing Setup

This guide explains how the automated PyPI publishing workflow is configured for the ZeroProof Python SDK.

## Overview

The workflow automatically:
1. ✅ **Bumps the patch version** (e.g., 0.2.0 → 0.2.1)
2. ✅ **Updates version** in `pyproject.toml`, `setup.py`, and `zeroproof/__init__.py`
3. ✅ **Commits changes** to main with `[skip ci]` to prevent loops
4. ✅ **Creates and pushes git tag** (e.g., `v0.2.1`)
5. ✅ **Builds package** with `uv build`
6. ✅ **Publishes to PyPI** with `uv publish`
7. ✅ **Creates GitHub Release** with auto-generated notes

## Prerequisites

### 1. GitHub Repository Permissions

Go to **Settings → Actions → General → Workflow permissions** and ensure:
- ✅ **"Read and write permissions"** is selected
- ✅ **"Allow GitHub Actions to create and approve pull requests"** is checked

### 2. GitHub Secrets

The workflow requires one secret:

| Secret Name | Description | Example |
|-------------|-------------|---------|
| `UV_PUBLISH_TOKEN` | Your PyPI API token | `pypi-AgEIcHlwaS5vcmcC...` |

**✅ You've already configured this!**

To verify or update:
1. Go to **Settings → Secrets and variables → Actions**
2. Check that `UV_PUBLISH_TOKEN` exists
3. The token should start with `pypi-`

### 3. Get a PyPI Token (if needed)

If you need to create or regenerate your PyPI token:

1. Log in to https://pypi.org
2. Go to **Account Settings → API tokens**
3. Click **"Add API token"**
4. Set scope to **"Entire account"** or **"Project: zeroproof"** (if package exists)
5. Copy the token (starts with `pypi-`)
6. Add as `UV_PUBLISH_TOKEN` in GitHub Secrets

## How It Works

### Trigger Conditions

The workflow runs when:
- ✅ Code is pushed to `main` branch
- ✅ Changes are in the `python-sdk/` directory
- ✅ Manual trigger via GitHub UI (workflow_dispatch)

### What Gets Updated

Every time the workflow runs, it updates these three files:

```
python-sdk/
├── pyproject.toml          # version = "0.2.1"
├── setup.py                # version="0.2.1"
└── zeroproof/__init__.py   # __version__ = "0.2.1"
```

### Version Increment Logic

- **Current**: `0.2.0`
- **After 1st merge**: `0.2.1`
- **After 2nd merge**: `0.2.2`
- **After 3rd merge**: `0.2.3`
- And so on...

The patch version increments by 1 on every merge to main.

## Usage

### Normal Development Flow

1. **Make changes** to the Python SDK in `python-sdk/`
2. **Create a PR** and get it reviewed
3. **Merge to main** - The workflow automatically:
   - Bumps version from 0.2.0 → 0.2.1
   - Publishes to PyPI
   - Creates git tag `v0.2.1`
   - Creates GitHub release
4. **Users can install** with: `pip install --upgrade zeroproof`

### Manual Trigger

To manually run the workflow:

1. Go to **Actions → Publish Python SDK to PyPI**
2. Click **"Run workflow"**
3. Select branch: `main`
4. Click **"Run workflow"**

## Files Created

### `.github/workflows/publish-pypi.yml`
Main workflow file that orchestrates the entire publishing process.

### `python-sdk/bump_version.py`
Python script that:
- Reads current version from `pyproject.toml`
- Increments patch version by 1
- Updates all three version files
- Outputs new version for GitHub Actions

## Monitoring

### Check Workflow Status

1. Go to **Actions** tab in GitHub
2. Look for **"Publish Python SDK to PyPI"** workflows
3. Green ✅ = successful publish
4. Red ❌ = failed (check logs)

### Verify Publication

After workflow completes:
1. **PyPI**: Check https://pypi.org/project/zeroproof/
2. **Git Tags**: Run `git fetch --tags && git tag -l`
3. **Releases**: Check https://github.com/jacobweiss2305/zeroproof/releases

### Installation Test

```bash
# Install latest version
pip install --upgrade zeroproof

# Verify version
python -c "import zeroproof; print(zeroproof.__version__)"
```

## Troubleshooting

### ❌ "File already exists" on PyPI

**Problem**: Trying to publish a version that already exists on PyPI.

**Solution**: This shouldn't happen with auto-bumping, but if it does:
1. Check current version on PyPI
2. Manually update version in all three files to be higher
3. Commit and push to main

### ❌ Permission Denied (Git Push)

**Problem**: Workflow can't push commits or tags.

**Solution**: 
1. Go to **Settings → Actions → General**
2. Set **"Read and write permissions"**
3. Check **"Allow GitHub Actions to create and approve pull requests"**

### ❌ Authentication Failed (PyPI)

**Problem**: PyPI token is invalid or expired.

**Solution**:
1. Generate new token at https://pypi.org/manage/account/token/
2. Update `UV_PUBLISH_TOKEN` secret in GitHub
3. Re-run the workflow

### ❌ Infinite Loop

**Problem**: Workflow keeps triggering itself.

**Solution**: The workflow includes `[skip ci]` in commit messages to prevent this. If loops occur:
1. Check that the path filter excludes `bump_version.py`
2. Verify commit message includes `[skip ci]`

## Manual Version Management

If you need to skip the auto-bump or set a specific version:

### Option 1: Update Manually Before Merge

```bash
cd python-sdk

# Update version to what you want
# Edit pyproject.toml, setup.py, zeroproof/__init__.py

# Commit with [skip ci] to prevent auto-bump
git commit -m "chore: set version to 0.3.0 [skip ci]"
git push
```

### Option 2: Major/Minor Version Bumps

For major (1.0.0) or minor (0.3.0) version bumps:

```bash
cd python-sdk

# Manually edit all three files:
# - pyproject.toml: version = "0.3.0"
# - setup.py: version="0.3.0"
# - zeroproof/__init__.py: __version__ = "0.3.0"

git add pyproject.toml setup.py zeroproof/__init__.py
git commit -m "chore: bump to v0.3.0 [skip ci]"
git push origin main

# Then merge other changes normally
# Auto-bumping will continue from 0.3.0 → 0.3.1 → 0.3.2...
```

## Security Notes

🔒 **Never commit tokens**:
- The `.env` file with `UV_PUBLISH_TOKEN` should never be in git
- Always use GitHub Secrets for sensitive values
- The `.gitignore` already excludes `.env`

🔒 **Token Permissions**:
- Use project-scoped tokens when possible
- Rotate tokens regularly
- Revoke old tokens after updating

## Fast Iteration Benefits

With this setup:
- ✅ **Zero manual steps** for version management
- ✅ **Every merge is a release**
- ✅ **Automatic changelog** via releases
- ✅ **Traceable history** with git tags
- ✅ **Fast feedback** - merge and it's live in ~2 minutes

## Need Help?

- **Workflow logs**: Check GitHub Actions tab
- **PyPI status**: https://status.python.org/
- **uv docs**: https://docs.astral.sh/uv/

---

**Last Updated**: January 2025
