#!/usr/bin/env python3
"""
Script to automatically bump the patch version of the zeroproof package.
Updates version in pyproject.toml, setup.py, and zeroproof/__init__.py
"""

import re
import sys
from pathlib import Path


def bump_patch_version(version: str) -> str:
    """Increment the patch version by 0.1"""
    parts = version.split('.')
    if len(parts) != 3:
        raise ValueError(f"Invalid version format: {version}")
    
    major, minor, patch = parts
    # Increment patch version by 0.1
    new_patch = str(int(patch) + 1)
    return f"{major}.{minor}.{new_patch}"


def update_pyproject_toml(file_path: Path, new_version: str) -> None:
    """Update version in pyproject.toml"""
    content = file_path.read_text(encoding='utf-8')
    updated = re.sub(
        r'version = "[^"]+"',
        f'version = "{new_version}"',
        content
    )
    file_path.write_text(updated, encoding='utf-8')
    print(f"✓ Updated {file_path}")


def update_setup_py(file_path: Path, new_version: str) -> None:
    """Update version in setup.py"""
    content = file_path.read_text(encoding='utf-8')
    updated = re.sub(
        r'version="[^"]+"',
        f'version="{new_version}"',
        content
    )
    file_path.write_text(updated, encoding='utf-8')
    print(f"✓ Updated {file_path}")


def update_init_py(file_path: Path, new_version: str) -> None:
    """Update version in __init__.py"""
    content = file_path.read_text(encoding='utf-8')
    updated = re.sub(
        r'__version__ = "[^"]+"',
        f'__version__ = "{new_version}"',
        content
    )
    file_path.write_text(updated, encoding='utf-8')
    print(f"✓ Updated {file_path}")


def get_current_version(pyproject_path: Path) -> str:
    """Extract current version from pyproject.toml"""
    content = pyproject_path.read_text(encoding='utf-8')
    match = re.search(r'version = "([^"]+)"', content)
    if not match:
        raise ValueError("Could not find version in pyproject.toml")
    return match.group(1)


def main():
    # Get the directory where this script is located (python-sdk/)
    script_dir = Path(__file__).parent
    
    # Define file paths
    pyproject_path = script_dir / "pyproject.toml"
    setup_path = script_dir / "setup.py"
    init_path = script_dir / "zeroproof" / "__init__.py"
    
    # Verify all files exist
    for path in [pyproject_path, setup_path, init_path]:
        if not path.exists():
            print(f"Error: {path} not found!", file=sys.stderr)
            sys.exit(1)
    
    # Get current version
    current_version = get_current_version(pyproject_path)
    print(f"Current version: {current_version}")
    
    # Bump version
    new_version = bump_patch_version(current_version)
    print(f"New version: {new_version}")
    
    # Update all files
    update_pyproject_toml(pyproject_path, new_version)
    update_setup_py(setup_path, new_version)
    update_init_py(init_path, new_version)
    
    print(f"\n✓ Successfully bumped version from {current_version} to {new_version}")
    
    # Output new version for GitHub Actions
    print(f"\n::set-output name=version::{new_version}")
    # Also write to GITHUB_OUTPUT for newer GitHub Actions
    github_output = Path(os.environ.get('GITHUB_OUTPUT', '/dev/null'))
    if github_output.exists() or os.environ.get('GITHUB_OUTPUT'):
        with open(github_output, 'a') as f:
            f.write(f"version={new_version}\n")


if __name__ == "__main__":
    import os
    main()
