"""Generated by Django 4.2.4 on 2023-08-21 02:15."""

from types import MappingProxyType

import django.db.models.deletion
import pycountry
from comicbox.identifiers import parse_identifier
from django.db import migrations, models

MIGRATE_GTIN = True

FIELDS_TABLE_MAP = MappingProxyType(
    {
        "age_rating": "AgeRating",
        "country": "Country",
        "language": "Language",
        "scan_info": "ScanInfo",
        "original_format": "OriginalFormat",
    }
)
SEARCH_INDEX_UUID = "SI"
_ALPHA_2_LEN = 2


def _migrate_comments(apps, _schema_editor):
    comic_model = apps.get_model("codex", "comic")

    comics = comic_model.objects.exclude(comments="").filter(summary="")
    for comic in comics:
        comic.summary = comic.comments

    comic_model.objects.bulk_update(comics, ("summary",))


def _migrate_reading_direction(apps, _schema_editor):
    comic_model = apps.get_model("codex", "comic")

    comics = comic_model.objects.filter(read_ltr=False)
    for comic in comics:
        comic.reading_direction = "rtl"

    comic_model.objects.bulk_update(comics, ("reading_direction",))


def _get_pycountry_alpha_2(val, lookup) -> str:
    if val and len(val) > _ALPHA_2_LEN and (obj := lookup.lookup(val)):
        val = obj.alpha_2
    return val


def _create_new_rows(comic_model, model, field_name, model_name):
    """Create new model rows."""
    null_filter = {f"{field_name}__in": (None, "")}
    names = (
        comic_model.objects.exclude(**null_filter)
        .values_list(field_name, flat=True)
        .distinct()
    )
    names = sorted(names)
    names_map = {}
    objs = []
    for old_name in names:
        if field_name == "country":
            new_name = _get_pycountry_alpha_2(old_name, pycountry.countries)
        elif field_name == "language":
            new_name = _get_pycountry_alpha_2(old_name, pycountry.languages)
        else:
            new_name = old_name
        names_map[old_name] = new_name
        obj = model(name=new_name)
        objs.append(obj)

    if objs:
        model.objects.bulk_create(objs)
        print(f"\tCreated {len(objs)} {model_name}s")
    return names_map


def _link_rows_to_comic(comic_model, model, field_name, model_name, names_map):
    """Link rows to comics."""
    update_comics = []
    new_field_name = f"new_{field_name}"
    for old_name, new_name in names_map.items():
        named_filter = {field_name: old_name}
        comics = comic_model.objects.filter(**named_filter)
        for comic in comics:
            obj = model.objects.get(name=new_name)
            setattr(comic, new_field_name, obj)
            update_comics.append(comic)
    if update_comics:
        comic_model.objects.bulk_update(update_comics, [new_field_name])
        print(f"\tLinked {len(update_comics)} comics to {len(names_map)} {model_name}s")


def _migrate_fields_to_tables(apps, _schema_editor):
    comic_model = apps.get_model("codex", "comic")
    print()
    for field_name, model_name in FIELDS_TABLE_MAP.items():
        model = apps.get_model("codex", model_name.lower())
        names_map = _create_new_rows(comic_model, model, field_name, model_name)
        _link_rows_to_comic(comic_model, model, field_name, model_name, names_map)


def _migrate_bookmark(apps, _schema_editor):
    bm_model = apps.get_model("codex", "bookmark")

    bookmarks = bm_model.objects.exclude(read_in_reverse=None, vertical=None)

    update_bookmarks = []
    for bm in bookmarks:
        if bm.read_in_reverse and not bm.vertical:
            bm.reading_direction = "rtl"
        elif not bm.read_in_reverse and bm.vertical:
            bm.reading_direction = "ttb"
            bm.two_pages = None
        elif bm.read_in_reverse and bm.vertical:
            bm.reading_direction = "btt"
            bm.two_pages = None
        else:
            bm.reading_direction = "ltr"
        update_bookmarks.append(bm)

    if update_bookmarks:
        bm_model.objects.bulk_update(
            update_bookmarks, ["reading_direction", "two_pages"]
        )
        print(f"\tMigrated {len(update_bookmarks)} Bookmarks to use reading_direction")


def _migrate_gtin_to_ids_scan(comics):
    identifiers = {}
    comic_identifiers = {}
    for comic in comics:
        try:
            identifier_type, code = parse_identifier(comic.gtin)
        except Exception:
            identifier_type = None
            code = None
        if not code:
            continue
        if identifier_type not in identifiers:
            identifiers[identifier_type] = set()
        identifiers[identifier_type].add(code)
        comic_identifiers[comic] = (identifier_type, code)
    return identifiers, comic_identifiers


def _migrate_gtin_to_ids_create_id_types(identifier_type_model, identifiers):
    create_identifier_types = []
    for name in identifiers:
        obj = identifier_type_model(name=name)
        create_identifier_types.append(obj)

    if create_identifier_types:
        identifier_type_model.objects.bulk_create(create_identifier_types)
        print(f"\tCreated {len(create_identifier_types)} IdentifierTypes")


def _migrate_gtin_to_ids_create_ids(
    identifier_model, identifier_type_model, identifiers
):
    create_identifiers = []
    for name, nsses in identifiers.items():
        identifier_type = identifier_type_model.objects.get(name=name)
        for nss in nsses:
            identifier = identifier_model(identifier_type=identifier_type, nss=nss)
            create_identifiers.append(identifier)

    if create_identifiers:
        identifier_model.objects.bulk_create(create_identifiers)
        print(f"\tCreated {len(create_identifiers)} Identifiers")


def _migrate_gtin_to_ids_link_comics(
    comic_model, identifier_model, comics, comic_identifiers
):
    through_model = comic_model.identifiers.through
    tms = []
    for comic in comics:
        name, nss = comic_identifiers[comic]
        identifier = identifier_model.objects.get(identifier_type__name=name, nss=nss)
        tm = through_model(comic_id=comic.pk, identifier_id=identifier.pk)
        tms.append(tm)

    if tms:
        through_model.objects.bulk_create(tms)
        print(f"\tLinked {len(comics)} Comics to Identifiers")


def _migrate_gtin_to_identifiers(apps, _schema_editor):
    """Migrate gtin to identifiers table."""
    if not MIGRATE_GTIN:
        return

    comic_model = apps.get_model("codex", "comic")
    comics = comic_model.objects.exclude(gtin="")

    identifiers, comic_identifiers = _migrate_gtin_to_ids_scan(comics)
    identifier_type_model = apps.get_model("codex", "identifiertype")
    _migrate_gtin_to_ids_create_id_types(identifier_type_model, identifiers)
    identifier_model = apps.get_model("codex", "identifier")
    _migrate_gtin_to_ids_create_ids(
        identifier_model, identifier_type_model, identifiers
    )
    _migrate_gtin_to_ids_link_comics(
        comic_model, identifier_model, comics, comic_identifiers
    )


def _migrate_volume_name(apps, _schema_editor):
    volume_model = apps.get_model("codex", "volume")

    all_volumes = volume_model.objects.all()

    update_volumes = []
    for volume in all_volumes:
        if volume.name == "":
            volume.name = None
            update_volumes.append(volume)

    if update_volumes:
        volume_model.objects.bulk_update(update_volumes, ["name"])
        print(f"\tMigrated {len(update_volumes)} volumes with empty names.")


def _clear_search_index_uuid(apps, _schema_editor):
    """Clear the search index uuid to force a search rebuild."""
    ts_model = apps.get_model("codex", "timestamp")
    ts_model.objects.filter(key=SEARCH_INDEX_UUID).update(version="")


class Migration(migrations.Migration):
    """Migrate database."""

    dependencies = [
        ("auth", "0012_alter_user_first_name_max_length"),
        ("codex", "0025_add_story_arc_number"),
    ]

    operations = [
        migrations.RenameModel(
            old_name="Creator",
            new_name="Contributor",
        ),
        migrations.RenameModel(
            old_name="CreatorPerson",
            new_name="ContributorPerson",
        ),
        migrations.RenameModel(
            old_name="CreatorRole",
            new_name="ContributorRole",
        ),
        migrations.RenameField(
            model_name="comic",
            old_name="creators",
            new_name="contributors",
        ),
        migrations.RenameField(
            model_name="comic",
            old_name="issue",
            new_name="issue_number",
        ),
        migrations.RunPython(
            _migrate_comments,
        ),
        migrations.RemoveField(
            model_name="comic",
            name="comments",
        ),
        migrations.RemoveField(
            model_name="comic",
            name="max_page",
        ),
        migrations.AddField(
            model_name="comic",
            name="monochrome",
            field=models.BooleanField(db_index=True, default=False),
        ),
        migrations.AddField(
            model_name="comic",
            name="review",
            field=models.TextField(default=""),
        ),
        migrations.AddField(
            model_name="comic",
            name="tagger",
            field=models.CharField(db_index=True, default="", max_length=128),
        ),
        migrations.AddField(
            model_name="comic",
            name="reading_direction",
            field=models.CharField(
                choices=[
                    ("ltr", "Ltr"),
                    ("rtl", "Rtl"),
                    ("ttb", "Ttb"),
                    ("btt", "Btt"),
                ],
                db_index=True,
                default="ltr",
                max_length=3,
            ),
        ),
        migrations.RunPython(
            _migrate_reading_direction,
        ),
        migrations.RemoveField(
            model_name="comic",
            name="read_ltr",
        ),
        migrations.CreateModel(
            name="Tagger",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                ("created_at", models.DateTimeField(auto_now_add=True)),
                ("updated_at", models.DateTimeField(auto_now=True)),
                ("name", models.CharField(db_index=True, max_length=128)),
            ],
            options={
                "abstract": False,
                "unique_together": {("name",)},
            },
        ),
        migrations.CreateModel(
            name="ScanInfo",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                ("created_at", models.DateTimeField(auto_now_add=True)),
                ("updated_at", models.DateTimeField(auto_now=True)),
                ("name", models.CharField(db_index=True, max_length=128)),
            ],
            options={
                "abstract": False,
                "unique_together": {("name",)},
            },
        ),
        migrations.CreateModel(
            name="OriginalFormat",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                ("created_at", models.DateTimeField(auto_now_add=True)),
                ("updated_at", models.DateTimeField(auto_now=True)),
                ("name", models.CharField(db_index=True, max_length=128)),
            ],
            options={
                "abstract": False,
                "unique_together": {("name",)},
            },
        ),
        migrations.CreateModel(
            name="Language",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                ("created_at", models.DateTimeField(auto_now_add=True)),
                ("updated_at", models.DateTimeField(auto_now=True)),
                ("name", models.CharField(db_index=True, max_length=128)),
            ],
            options={
                "abstract": False,
                "unique_together": {("name",)},
            },
        ),
        migrations.CreateModel(
            name="Country",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                ("created_at", models.DateTimeField(auto_now_add=True)),
                ("updated_at", models.DateTimeField(auto_now=True)),
                ("name", models.CharField(db_index=True, max_length=128)),
            ],
            options={
                "abstract": False,
                "unique_together": {("name",)},
            },
        ),
        migrations.CreateModel(
            name="AgeRating",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                ("created_at", models.DateTimeField(auto_now_add=True)),
                ("updated_at", models.DateTimeField(auto_now=True)),
                ("name", models.CharField(db_index=True, max_length=128)),
            ],
            options={
                "abstract": False,
                "unique_together": {("name",)},
            },
        ),
        migrations.AddField(
            model_name="comic",
            name="new_age_rating",
            field=models.ForeignKey(
                null=True,
                on_delete=django.db.models.deletion.CASCADE,
                to="codex.agerating",
            ),
        ),
        migrations.AddField(
            model_name="comic",
            name="new_country",
            field=models.ForeignKey(
                null=True,
                on_delete=django.db.models.deletion.CASCADE,
                to="codex.country",
            ),
        ),
        migrations.AddField(
            model_name="comic",
            name="new_language",
            field=models.ForeignKey(
                null=True,
                on_delete=django.db.models.deletion.CASCADE,
                to="codex.language",
            ),
        ),
        migrations.AddField(
            model_name="comic",
            name="new_original_format",
            field=models.ForeignKey(
                null=True,
                on_delete=django.db.models.deletion.CASCADE,
                to="codex.originalformat",
            ),
        ),
        migrations.AddField(
            model_name="comic",
            name="new_scan_info",
            field=models.ForeignKey(
                null=True,
                on_delete=django.db.models.deletion.CASCADE,
                to="codex.scaninfo",
            ),
        ),
        migrations.AddField(
            model_name="comic",
            name="tagger",
            field=models.ForeignKey(
                null=True,
                on_delete=django.db.models.deletion.CASCADE,
                to="codex.tagger",
            ),
        ),
        migrations.RunPython(
            _migrate_fields_to_tables,
        ),
        migrations.RemoveField(
            model_name="comic",
            name="age_rating",
        ),
        migrations.RemoveField(
            model_name="comic",
            name="country",
        ),
        migrations.RemoveField(
            model_name="comic",
            name="language",
        ),
        migrations.RemoveField(
            model_name="comic",
            name="original_format",
        ),
        migrations.RemoveField(
            model_name="comic",
            name="scan_info",
        ),
        migrations.RenameField(
            model_name="comic",
            old_name="new_age_rating",
            new_name="age_rating",
        ),
        migrations.RenameField(
            model_name="comic",
            old_name="new_country",
            new_name="country",
        ),
        migrations.RenameField(
            model_name="comic",
            old_name="new_language",
            new_name="language",
        ),
        migrations.RenameField(
            model_name="comic",
            old_name="new_original_format",
            new_name="original_format",
        ),
        migrations.RenameField(
            model_name="comic",
            old_name="new_scan_info",
            new_name="scan_info",
        ),
        migrations.AddField(
            model_name="bookmark",
            name="reading_direction",
            field=models.CharField(
                blank=True,
                choices=[
                    ("ltr", "Ltr"),
                    ("rtl", "Rtl"),
                    ("ttb", "Ttb"),
                    ("btt", "Btt"),
                ],
                default="",
                max_length=3,
            ),
        ),
        migrations.RunPython(_migrate_bookmark),
        migrations.RemoveField(
            model_name="bookmark",
            name="read_in_reverse",
        ),
        migrations.RemoveField(
            model_name="bookmark",
            name="vertical",
        ),
        migrations.AlterModelOptions(
            name="adminflag",
            options={"get_latest_by": "updated_at"},
        ),
        migrations.AlterModelOptions(
            name="agerating",
            options={"get_latest_by": "updated_at"},
        ),
        migrations.AlterModelOptions(
            name="bookmark",
            options={"get_latest_by": "updated_at"},
        ),
        migrations.AlterModelOptions(
            name="character",
            options={"get_latest_by": "updated_at"},
        ),
        migrations.AlterModelOptions(
            name="comic",
            options={"get_latest_by": "updated_at", "verbose_name": "Issue"},
        ),
        migrations.AlterModelOptions(
            name="contributor",
            options={"get_latest_by": "updated_at"},
        ),
        migrations.AlterModelOptions(
            name="contributorperson",
            options={"get_latest_by": "updated_at"},
        ),
        migrations.AlterModelOptions(
            name="contributorrole",
            options={"get_latest_by": "updated_at"},
        ),
        migrations.AlterModelOptions(
            name="country",
            options={"get_latest_by": "updated_at", "verbose_name_plural": "Countries"},
        ),
        migrations.AlterModelOptions(
            name="failedimport",
            options={"get_latest_by": "updated_at"},
        ),
        migrations.AlterModelOptions(
            name="folder",
            options={"get_latest_by": "updated_at"},
        ),
        migrations.AlterModelOptions(
            name="genre",
            options={"get_latest_by": "updated_at"},
        ),
        migrations.AlterModelOptions(
            name="imprint",
            options={"get_latest_by": "updated_at"},
        ),
        migrations.AlterModelOptions(
            name="language",
            options={"get_latest_by": "updated_at"},
        ),
        migrations.AlterModelOptions(
            name="library",
            options={"get_latest_by": "updated_at", "verbose_name_plural": "Libraries"},
        ),
        migrations.AlterModelOptions(
            name="location",
            options={"get_latest_by": "updated_at"},
        ),
        migrations.AlterModelOptions(
            name="originalformat",
            options={"get_latest_by": "updated_at"},
        ),
        migrations.AlterModelOptions(
            name="publisher",
            options={"get_latest_by": "updated_at"},
        ),
        migrations.AlterModelOptions(
            name="scaninfo",
            options={"get_latest_by": "updated_at"},
        ),
        migrations.AlterModelOptions(
            name="series",
            options={"get_latest_by": "updated_at", "verbose_name_plural": "Series"},
        ),
        migrations.AlterModelOptions(
            name="seriesgroup",
            options={"get_latest_by": "updated_at"},
        ),
        migrations.AlterModelOptions(
            name="storyarc",
            options={"get_latest_by": "updated_at"},
        ),
        migrations.AlterModelOptions(
            name="storyarcnumber",
            options={"get_latest_by": "updated_at"},
        ),
        migrations.AlterModelOptions(
            name="tag",
            options={"get_latest_by": "updated_at"},
        ),
        migrations.AlterModelOptions(
            name="tagger",
            options={"get_latest_by": "updated_at"},
        ),
        migrations.AlterModelOptions(
            name="team",
            options={"get_latest_by": "updated_at"},
        ),
        migrations.AlterModelOptions(
            name="timestamp",
            options={"get_latest_by": "updated_at"},
        ),
        migrations.AlterModelOptions(
            name="volume",
            options={"get_latest_by": "updated_at"},
        ),
        migrations.CreateModel(
            name="IdentifierType",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                ("created_at", models.DateTimeField(auto_now_add=True)),
                ("updated_at", models.DateTimeField(auto_now=True)),
                ("name", models.CharField(db_index=True, max_length=128)),
            ],
            options={
                "get_latest_by": "updated_at",
                "abstract": False,
                "unique_together": {("name",)},
            },
        ),
        migrations.CreateModel(
            name="Identifier",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                ("created_at", models.DateTimeField(auto_now_add=True)),
                ("updated_at", models.DateTimeField(auto_now=True)),
                ("nss", models.CharField(max_length=128)),
                ("url", models.URLField(default="")),
                (
                    "identifier_type",
                    models.ForeignKey(
                        null=True,
                        on_delete=django.db.models.deletion.CASCADE,
                        to="codex.identifiertype",
                    ),
                ),
            ],
            options={
                "get_latest_by": "updated_at",
                "abstract": False,
                "unique_together": {("identifier_type", "nss")},
            },
        ),
        migrations.AddField(
            model_name="comic",
            name="identifiers",
            field=models.ManyToManyField(to="codex.identifier"),
        ),
        migrations.RunPython(_migrate_gtin_to_identifiers),
        migrations.CreateModel(
            name="Story",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                ("created_at", models.DateTimeField(auto_now_add=True)),
                ("updated_at", models.DateTimeField(auto_now=True)),
                ("name", models.CharField(db_index=True, max_length=128)),
            ],
            options={
                "get_latest_by": "updated_at",
                "verbose_name_plural": "Stories",
                "abstract": False,
                "unique_together": {("name",)},
            },
        ),
        migrations.AddField(
            model_name="comic",
            name="stories",
            field=models.ManyToManyField(to="codex.story"),
        ),
        migrations.RemoveField(
            model_name="comic",
            name="gtin",
        ),
        migrations.RemoveField(
            model_name="comic",
            name="web",
        ),
        migrations.AlterField(
            model_name="volume",
            name="name",
            field=models.SmallIntegerField(db_index=True, default=None, null=True),
        ),
        migrations.RunPython(_migrate_volume_name),
        migrations.CreateModel(
            name="GroupAuth",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True,
                        primary_key=True,
                        serialize=False,
                        verbose_name="ID",
                    ),
                ),
                ("created_at", models.DateTimeField(auto_now_add=True)),
                ("updated_at", models.DateTimeField(auto_now=True)),
                ("exclude", models.BooleanField(db_index=True, default=False)),
                (
                    "group",
                    models.OneToOneField(
                        on_delete=django.db.models.deletion.CASCADE, to="auth.group"
                    ),
                ),
            ],
            options={
                "get_latest_by": "updated_at",
                "abstract": False,
                "unique_together": {("group",)},
            },
        ),
        migrations.AlterModelOptions(
            name="librarianstatus",
            options={
                "get_latest_by": "updated_at",
                "verbose_name_plural": "LibrarianStatuses",
            },
        ),
        migrations.RunPython(_clear_search_index_uuid),
    ]
