"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const fs = require("fs-extra");
const aws_entities_1 = require("../lib/aws-entities");
const default_1 = require("../lib/default");
const fact_tables_1 = require("./fact-tables");
async function main() {
    checkRegions(fact_tables_1.APPMESH_ECR_ACCOUNTS);
    checkRegions(fact_tables_1.DLC_REPOSITORY_ACCOUNTS);
    checkRegions(fact_tables_1.ELBV2_ACCOUNTS);
    checkRegions(fact_tables_1.FIREHOSE_CIDR_BLOCKS);
    checkRegions(fact_tables_1.ROUTE_53_BUCKET_WEBSITE_ZONE_IDS);
    checkRegionsSubMap(fact_tables_1.CLOUDWATCH_LAMBDA_INSIGHTS_ARNS);
    const lines = [
        "import { Fact, FactName } from './fact';",
        '',
        '/* eslint-disable quote-props */',
        '/* eslint-disable max-len */',
        '',
        '/**',
        ' * Built-in regional information, re-generated by `npm run build`.',
        ' *',
        ` * @generated ${new Date().toISOString()}`,
        ' */',
        'export class BuiltIns {',
        '  /**',
        '   * Registers all the built in regional data in the RegionInfo database.',
        '   */',
        '  public static register(): void {',
    ];
    const defaultMap = 'default';
    for (const region of aws_entities_1.AWS_REGIONS) {
        let partition = fact_tables_1.PARTITION_MAP[defaultMap].partition;
        let domainSuffix = fact_tables_1.PARTITION_MAP[defaultMap].domainSuffix;
        for (const key in fact_tables_1.PARTITION_MAP) {
            if (region.startsWith(key)) {
                partition = fact_tables_1.PARTITION_MAP[key].partition;
                domainSuffix = fact_tables_1.PARTITION_MAP[key].domainSuffix;
            }
        }
        registerFact(region, 'PARTITION', partition);
        registerFact(region, 'DOMAIN_SUFFIX', domainSuffix);
        registerFact(region, 'CDK_METADATA_RESOURCE_AVAILABLE', fact_tables_1.AWS_CDK_METADATA.has(region) ? 'YES' : 'NO');
        registerFact(region, 'S3_STATIC_WEBSITE_ENDPOINT', aws_entities_1.before(region, aws_entities_1.RULE_S3_WEBSITE_REGIONAL_SUBDOMAIN)
            ? `s3-website-${region}.${domainSuffix}`
            : `s3-website.${region}.${domainSuffix}`);
        registerFact(region, 'S3_STATIC_WEBSITE_ZONE_53_HOSTED_ZONE_ID', fact_tables_1.ROUTE_53_BUCKET_WEBSITE_ZONE_IDS[region] || '');
        registerFact(region, 'EBS_ENV_ENDPOINT_HOSTED_ZONE_ID', fact_tables_1.EBS_ENV_ENDPOINT_HOSTED_ZONE_IDS[region] || '');
        registerFact(region, 'ELBV2_ACCOUNT', fact_tables_1.ELBV2_ACCOUNTS[region]);
        registerFact(region, 'DLC_REPOSITORY_ACCOUNT', fact_tables_1.DLC_REPOSITORY_ACCOUNTS[region]);
        registerFact(region, 'APPMESH_ECR_ACCOUNT', fact_tables_1.APPMESH_ECR_ACCOUNTS[region]);
        const firehoseCidrBlock = fact_tables_1.FIREHOSE_CIDR_BLOCKS[region];
        if (firehoseCidrBlock) {
            registerFact(region, 'FIREHOSE_CIDR_BLOCK', `${fact_tables_1.FIREHOSE_CIDR_BLOCKS[region]}/27`);
        }
        const vpcEndpointServiceNamePrefix = `${domainSuffix.split('.').reverse().join('.')}.vpce`;
        registerFact(region, 'VPC_ENDPOINT_SERVICE_NAME_PREFIX', vpcEndpointServiceNamePrefix);
        for (const service of aws_entities_1.AWS_SERVICES) {
            registerFact(region, ['servicePrincipal', service], default_1.Default.servicePrincipal(service, region, domainSuffix));
        }
        for (const version in fact_tables_1.CLOUDWATCH_LAMBDA_INSIGHTS_ARNS) {
            for (const arch in fact_tables_1.CLOUDWATCH_LAMBDA_INSIGHTS_ARNS[version]) {
                registerFact(region, ['cloudwatchLambdaInsightsVersion', version, arch], fact_tables_1.CLOUDWATCH_LAMBDA_INSIGHTS_ARNS[version][arch][region]);
            }
        }
    }
    lines.push('  }');
    lines.push('');
    lines.push('  private constructor() {}');
    lines.push('}');
    await fs.writeFile(path.resolve(__dirname, '..', 'lib', 'built-ins.generated.ts'), lines.join('\n'));
    function registerFact(region, name, value) {
        const factName = typeof name === 'string' ? name : `${name[0]}(${name.slice(1).map(s => JSON.stringify(s)).join(', ')})`;
        lines.push(`    Fact.register({ region: ${JSON.stringify(region)}, name: FactName.${factName}, value: ${JSON.stringify(value)} });`);
    }
}
/**
 * Verifies that the provided map of region to fact does not contain an entry
 * for a region that was not registered in `AWS_REGIONS`.
 */
function checkRegions(map) {
    const allRegions = new Set(aws_entities_1.AWS_REGIONS);
    for (const region of Object.keys(map)) {
        if (!allRegions.has(region)) {
            throw new Error(`Un-registered region fact found: ${region}. Add to AWS_REGIONS list!`);
        }
    }
}
/**
 * Verifies that the provided map of <KEY> to region to fact does not contain an entry
 * for a region that was not registered in `AWS_REGIONS`.
 */
function checkRegionsSubMap(map) {
    const allRegions = new Set(aws_entities_1.AWS_REGIONS);
    for (const key of Object.keys(map)) {
        for (const subKey of Object.keys(map[key])) {
            for (const region of Object.keys(map[key][subKey])) {
                if (!allRegions.has(region)) {
                    throw new Error(`Un-registered region fact found: ${region}. Add to AWS_REGIONS list!`);
                }
            }
        }
    }
}
main().catch(e => {
    // eslint-disable-next-line no-console
    console.error(e);
    process.exit(-1);
});
//# sourceMappingURL=data:application/json;base64,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