from typing import Literal, Optional

from pydantic import BaseModel


class ResponseFunctionToolCall(BaseModel):
    """
    Example usage

    {
        "id": "fc_12345xyz",
        "call_id": "call_12345xyz",
        "type": "function_call",
        "name": "get_weather",
        "arguments": "{\"location\":\"Paris, France\"}"
    }
    """

    arguments: str
    """A JSON string of the arguments to pass to the function."""

    call_id: str
    """The unique ID of the function tool call generated by the model."""

    name: str
    """The name of the function to run."""

    type: Literal["function_call"] = "function_call"
    """The type of the function tool call. Always `function_call`."""

    id: Optional[str] = None
    """The unique ID of the function tool call."""

    status: Optional[Literal["in_progress", "completed", "incomplete"]] = None
    """The status of the item.

    One of `in_progress`, `completed`, or `incomplete`. Populated when items are
    returned via API.
    """


class ResponseFunctionToolCallResult(BaseModel):
    """
    Example usage

    {
        "type": "function_call_output",
        "call_id": tool_call.call_id,
        "output": str(result)
    }
    """

    type: Literal["function_call_output"] = "function_call_output"
    """The type of the function tool call result. Always `function_call_output`."""

    call_id: str
    """The unique ID of the function tool call."""

    output: str
    """The output of the function tool call as a string."""