from graphql_api.api import GraphQLAPI
from graphql_api.mapper import GraphQLMetaKey


class TestSchemaFiltering:
    def test_query_remove_invalid(self):
        api = GraphQLAPI()

        class Person:
            def __init__(self):
                self.name = ""

            @api.field(mutable=True)
            def update_name(self, name: str) -> "Person":
                self.name = name
                return self

        # noinspection PyUnusedLocal
        @api.type(is_root_type=True)
        class Root:
            @api.field
            def person(self) -> Person:
                return Person()

        executor = api.executor()

        test_query = """
            query PersonName {
                person {
                    updateName(name:"phil") {
                        name
                    }
                }
            }
        """

        result = executor.execute(test_query)
        assert result.errors
        assert "Cannot query field" in result.errors[0].message

    def test_mutation_return_query(self):
        """
        Mutation fields by default should return queries
        :return:
        """
        api = GraphQLAPI()

        class Person:
            def __init__(self):
                self._name = ""

            @api.field
            def name(self) -> str:
                return self._name

            @api.field(mutable=True)
            def update_name(self, name: str) -> "Person":
                self._name = name
                return self

        # noinspection PyUnusedLocal
        @api.type(is_root_type=True)
        class Root:
            @api.field
            def person(self) -> Person:
                return Person()

        executor = api.executor()

        test_query = """
            mutation PersonName {
                person {
                    updateName(name:"phil") {
                        name
                    }
                }
            }
        """

        result = executor.execute(test_query)
        assert not result.errors

        expected = {"person": {"updateName": {"name": "phil"}}}

        assert result.data == expected

    def test_keep_interface(self):
        api = GraphQLAPI()

        @api.type(interface=True)
        class Person:
            @api.field
            def name(self) -> str:
                pass

        class Employee(Person):
            def __init__(self):
                self._name = "Bob"

            @api.field
            def name(self) -> str:
                return self._name

            @api.field
            def department(self) -> str:
                return "Human Resources"

            @api.field(mutable=True)
            def set_name(self, name: str) -> str:
                self._name = name
                return name

        bob_employee = Employee()

        # noinspection PyUnusedLocal
        @api.type(is_root_type=True)
        class Root:
            @api.field
            def person(self) -> Person:
                return bob_employee

        executor = api.executor()

        test_query = """
            query PersonName {
                person {
                    name
                    ... on Employee {
                        department
                    }
                }
            }
        """

        test_mutation = """
            mutation SetPersonName {
                person {
                    ... on EmployeeMutable {
                        setName(name: "Tom")
                    }
                }
            }
        """

        result = executor.execute(test_query)

        expected = {"person": {"name": "Bob", "department": "Human Resources"}}

        expected_2 = {"person": {"name": "Tom", "department": "Human Resources"}}

        assert result.data == expected

        result = executor.execute(test_mutation)

        assert not result.errors

        result = executor.execute(test_query)

        assert result.data == expected_2

    def test_mutation_return_mutable_flag(self):
        api = GraphQLAPI()

        @api.type
        class Person:
            def __init__(self):
                self._name = ""

            @api.field
            def name(self) -> str:
                return self._name

            @api.field(mutable=True)
            def update_name(self, name: str) -> "Person":
                self._name = name
                return self

            @api.field({GraphQLMetaKey.resolve_to_mutable: True}, mutable=True)
            def update_name_mutable(self, name: str) -> "Person":
                self._name = name
                return self

        # noinspection PyUnusedLocal
        @api.type(is_root_type=True)
        class Root:
            @api.field
            def person(self) -> Person:
                return Person()

        executor = api.executor()

        test_query = """
                    mutation PersonName {
                        person {
                            updateName(name:"phil") {
                                name
                            }
                        }
                    }
                """

        result = executor.execute(test_query)
        assert not result.errors

        expected = {"person": {"updateName": {"name": "phil"}}}

        assert result.data == expected

        test_mutable_query = """
                    mutation PersonName {
                        person {
                            updateNameMutable(name:"tom") {
                                updateName(name:"phil") {
                                    name
                                }
                            }
                        }
                    }
                """

        result = executor.execute(test_mutable_query)
        assert not result.errors

        expected = {"person": {"updateNameMutable": {"updateName": {"name": "phil"}}}}

        assert result.data == expected

        test_invalid_query = """
                    mutation PersonName {
                        person {
                            updateName(name:"tom") {
                                updateName(name:"phil") {
                                    name
                                }
                            }
                        }
                    }
                """

        result = executor.execute(test_invalid_query)
        assert result.errors
        assert "Cannot query field 'updateName'" in result.errors[0].message

        test_invalid_mutable_query = """
                    mutation PersonName {
                        person {
                            updateNameMutable(name:"tom") {
                                name
                            }
                        }
                    }
                """

        result = executor.execute(test_invalid_mutable_query)
        assert result.errors
        assert "Cannot query field 'name'" in result.errors[0].message

    def test_filter_all_fields_removes_empty_type(self):
        """
        Test that when filtering removes all fields from a type,
        in strict mode the empty type is completely removed from the schema
        """
        from graphql_api.reduce import TagFilter
        from graphql_api.decorators import field

        class SecretData:
            @field({"tags": ["admin"]})
            def secret_value(self) -> str:
                return "secret"

            @field({"tags": ["admin"]})
            def another_secret(self) -> int:
                return 42

        class Root:
            @field
            def public_data(self) -> str:
                return "public"

            @field
            def secret_data(self) -> SecretData:
                return SecretData()

        # Create API with filter that removes admin fields using strict mode (preserve_transitive=False)
        filtered_api = GraphQLAPI(
            root_type=Root,
            filters=[TagFilter(tags=["admin"], preserve_transitive=False)],
        )
        executor = filtered_api.executor()

        # This query should work without errors
        test_query = """
            query GetPublicData {
                publicData
            }
        """

        result = executor.execute(test_query)
        assert not result.errors
        assert result.data == {"publicData": "public"}

        # SecretData should be completely removed in strict mode
        # The field referencing it should also be removed
        test_query_with_empty_type = """
            query GetSecretData {
                secretData {
                    secretValue
                }
            }
        """

        result = executor.execute(test_query_with_empty_type)
        assert result.errors
        assert "Cannot query field 'secretData'" in str(result.errors[0])

        # Verify schema structure - SecretData should be completely absent
        schema, _ = filtered_api.build_schema()
        type_map = schema.type_map
        assert "SecretData" not in type_map

        # Root should not have the secretData field
        assert schema.query_type is not None
        root_fields = schema.query_type.fields
        assert "publicData" in root_fields
        assert "secretData" not in root_fields

    def test_preserve_transitive_empty_types(self):
        """
        Test that preserve_transitive=True preserves object types that have some accessible fields
        when they are referenced by unfiltered types, even if some of their fields are filtered.
        Types with NO accessible fields cannot be preserved due to GraphQL constraints.
        """
        from graphql_api.reduce import TagFilter
        from graphql_api.decorators import field

        # Type that has some fields filtered but retains one accessible field
        class PartiallyFiltered:
            @field({"tags": ["admin"]})
            def admin_only_field(self) -> str:
                return "admin-only"

            @field  # This field remains accessible
            def public_field(self) -> str:
                return "public data"

        class HasValidFields:
            @field
            def public_field(self) -> str:
                return "public"

            @field
            def partial_ref(self) -> PartiallyFiltered:
                return PartiallyFiltered()

        class Root:
            @field
            def has_valid_fields(self) -> HasValidFields:
                return HasValidFields()

        # Test with preserve_transitive=True (default)
        preserve_api = GraphQLAPI(
            root_type=Root,
            filters=[TagFilter(tags=["admin"], preserve_transitive=True)],
        )

        # Build schema and verify PartiallyFiltered is preserved
        schema, _ = preserve_api.build_schema()
        type_map = schema.type_map

        assert "HasValidFields" in type_map
        assert (
            "PartiallyFiltered" in type_map
        )  # Should be preserved because it has accessible fields!

        # Verify the reference field is preserved
        from graphql import GraphQLObjectType

        has_valid_fields_type = type_map["HasValidFields"]
        assert isinstance(has_valid_fields_type, GraphQLObjectType)
        assert "publicField" in has_valid_fields_type.fields
        assert (
            "partialRef" in has_valid_fields_type.fields
        )  # Reference should be preserved

        # Verify PartiallyFiltered has one accessible field (admin field filtered out)
        partial_type = type_map["PartiallyFiltered"]
        assert isinstance(partial_type, GraphQLObjectType)
        assert len(partial_type.fields) == 1  # One accessible field
        assert "adminOnlyField" not in partial_type.fields  # Filtered field removed
        assert "publicField" in partial_type.fields  # Public field preserved

        # Compare with preserve_transitive=False - in this case, both modes should preserve the type
        # because it has accessible fields
        strict_api = GraphQLAPI(
            root_type=Root,
            filters=[TagFilter(tags=["admin"], preserve_transitive=False)],
        )

        strict_schema, _ = strict_api.build_schema()
        strict_type_map = strict_schema.type_map

        assert "HasValidFields" in strict_type_map
        assert (
            "PartiallyFiltered" in strict_type_map
        )  # Should also be preserved in strict mode

        # Verify the behavior is the same since the type has accessible fields
        strict_partial_type = strict_type_map["PartiallyFiltered"]
        assert isinstance(strict_partial_type, GraphQLObjectType)
        assert len(strict_partial_type.fields) == 1
        assert "publicField" in strict_partial_type.fields

    def test_preserve_transitive_vs_strict_difference(self):
        """
        Test the difference between preserve_transitive=True and preserve_transitive=False.
        preserve_transitive=True should preserve more types that are referenced
        but might not be directly needed.
        """
        from graphql_api.reduce import TagFilter
        from graphql_api.decorators import field

        # A type that would only be preserved with preserve_transitive=True
        class IndirectlyReferenced:
            @field
            def some_data(self) -> str:
                return "data"

        class DirectlyReferenced:
            @field
            def public_info(self) -> str:
                return "public"

            # This creates a transitive reference
            @field
            def indirect_ref(self) -> IndirectlyReferenced:
                return IndirectlyReferenced()

        class Root:
            @field
            def direct_ref(self) -> DirectlyReferenced:
                return DirectlyReferenced()

        # Test preserve_transitive=True (default)
        preserve_api = GraphQLAPI(root_type=Root)
        preserve_schema, _ = preserve_api.build_schema()
        preserve_type_map = preserve_schema.type_map

        # Both types should be preserved
        assert "DirectlyReferenced" in preserve_type_map
        assert "IndirectlyReferenced" in preserve_type_map

        # Test preserve_transitive=False
        strict_api = GraphQLAPI(
            root_type=Root,
            filters=[
                TagFilter(tags=[], preserve_transitive=False)
            ],  # Empty filter with strict behavior
        )
        strict_schema, _ = strict_api.build_schema()
        strict_type_map = strict_schema.type_map

        # Both types should also be preserved in this case since no filtering is applied
        # and both types have accessible fields
        assert "DirectlyReferenced" in strict_type_map
        assert "IndirectlyReferenced" in strict_type_map

    def test_default_filtering_behavior_is_preserve_transitive(self):
        """
        Test that the default filtering behavior preserves transitive dependencies.
        This test verifies that TagFilter defaults to preserve_transitive=True.
        """
        from graphql_api.reduce import TagFilter
        from graphql_api.decorators import field

        class ReferencedType:
            @field
            def some_field(self) -> str:
                return "data"

        class Root:
            @field
            def reference(self) -> ReferencedType:
                return ReferencedType()

        # Test default behavior (TagFilter should default to preserve_transitive=True)
        api_default = GraphQLAPI(root_type=Root, filters=[TagFilter(tags=[])])
        schema_default, _ = api_default.build_schema()

        # Test explicit preserve_transitive=True
        api_preserve = GraphQLAPI(
            root_type=Root, filters=[TagFilter(tags=[], preserve_transitive=True)]
        )
        schema_preserve, _ = api_preserve.build_schema()

        # Both should have the same types
        default_types = {
            name
            for name, type_obj in schema_default.type_map.items()
            if hasattr(type_obj, "fields") and not name.startswith("_")
        }
        preserve_types = {
            name
            for name, type_obj in schema_preserve.type_map.items()
            if hasattr(type_obj, "fields") and not name.startswith("_")
        }

        assert default_types == preserve_types
        assert "ReferencedType" in default_types
        assert "Root" in default_types

    def test_filter_response_enum_properties(self):
        """
        Test that the FilterResponse enum has correct properties
        """
        from graphql_api.reduce import FilterResponse

        # Test ALLOW - keep field, don't preserve transitive
        assert not FilterResponse.ALLOW.should_filter
        assert not FilterResponse.ALLOW.preserve_transitive

        # Test ALLOW_TRANSITIVE - keep field, preserve transitive
        assert not FilterResponse.ALLOW_TRANSITIVE.should_filter
        assert FilterResponse.ALLOW_TRANSITIVE.preserve_transitive

        # Test REMOVE - remove field, preserve transitive
        assert FilterResponse.REMOVE.should_filter
        assert FilterResponse.REMOVE.preserve_transitive

        # Test REMOVE_STRICT - remove field, don't preserve transitive
        assert FilterResponse.REMOVE_STRICT.should_filter
        assert not FilterResponse.REMOVE_STRICT.preserve_transitive

    def test_all_filter_response_behaviors(self):
        """
        Test all 4 FilterResponse enum values in a comprehensive scenario
        """
        from graphql_api.reduce import FilterResponse, TagFilter
        from graphql_api.decorators import field

        class AllBehaviorsFilter(TagFilter):
            def __init__(self):
                super().__init__(tags=[], preserve_transitive=True)

            def filter_field(self, name: str, meta: dict) -> FilterResponse:
                tags = meta.get("tags", [])

                if "allow" in tags:
                    return FilterResponse.ALLOW  # Keep field, don't preserve transitive
                elif "allow_transitive" in tags:
                    return (
                        FilterResponse.ALLOW_TRANSITIVE
                    )  # Keep field, preserve transitive
                elif "remove" in tags:
                    return FilterResponse.REMOVE  # Remove field, preserve transitive
                elif "remove_strict" in tags:
                    return (
                        FilterResponse.REMOVE_STRICT
                    )  # Remove field, don't preserve transitive
                else:
                    return FilterResponse.ALLOW_TRANSITIVE  # Default behavior

        class ReferencedType:
            @field({"tags": ["allow"]})
            def allow_field(self) -> str:
                return "allow"

            @field({"tags": ["allow_transitive"]})
            def allow_transitive_field(self) -> str:
                return "allow_transitive"

            @field({"tags": ["remove"]})
            def remove_field(self) -> str:
                return "remove"

            @field({"tags": ["remove_strict"]})
            def remove_strict_field(self) -> str:
                return "remove_strict"

        class Root:
            @field({"tags": ["allow"]})
            def allow_ref(self) -> ReferencedType:
                return ReferencedType()

            @field({"tags": ["allow_transitive"]})
            def allow_transitive_ref(self) -> ReferencedType:
                return ReferencedType()

            @field({"tags": ["remove"]})
            def remove_ref(self) -> ReferencedType:
                return ReferencedType()

            @field({"tags": ["remove_strict"]})
            def remove_strict_ref(self) -> ReferencedType:
                return ReferencedType()

        # Test with the comprehensive filter
        api = GraphQLAPI(root_type=Root, filters=[AllBehaviorsFilter()])
        schema, _ = api.build_schema()
        executor = api.executor()

        # Verify schema structure
        assert schema.query_type is not None
        root_fields = schema.query_type.fields

        # ALLOW and ALLOW_TRANSITIVE should keep the fields
        assert "allowRef" in root_fields
        assert "allowTransitiveRef" in root_fields

        # REMOVE and REMOVE_STRICT should remove the fields
        assert "removeRef" not in root_fields
        assert "removeStrictRef" not in root_fields

        # ReferencedType should still exist (preserved by ALLOW_TRANSITIVE and REMOVE behaviors)
        assert "ReferencedType" in schema.type_map

        from graphql import GraphQLObjectType

        referenced_type = schema.type_map["ReferencedType"]
        assert isinstance(referenced_type, GraphQLObjectType)

        # Only ALLOW and ALLOW_TRANSITIVE fields should remain in ReferencedType
        assert "allowField" in referenced_type.fields
        assert "allowTransitiveField" in referenced_type.fields
        assert "removeField" not in referenced_type.fields
        assert "removeStrictField" not in referenced_type.fields

        # Test queries work for allowed fields
        result = executor.execute(
            """
            query TestAllowed {
                allowRef {
                    allowField
                    allowTransitiveField
                }
                allowTransitiveRef {
                    allowField
                    allowTransitiveField
                }
            }
        """
        )

        assert not result.errors
        expected = {
            "allowRef": {
                "allowField": "allow",
                "allowTransitiveField": "allow_transitive",
            },
            "allowTransitiveRef": {
                "allowField": "allow",
                "allowTransitiveField": "allow_transitive",
            },
        }
        assert result.data == expected

    def test_custom_filter_without_preserve_transitive_attribute(self):
        """
        Test that custom filters without preserve_transitive attributes work correctly.
        This addresses the issue where the system was trying to access preserve_transitive
        on filter objects instead of determining behavior from FilterResponse values.
        """
        from graphql_api.reduce import GraphQLFilter, FilterResponse
        from graphql_api.decorators import field

        class CustomFilter(GraphQLFilter):
            """Custom filter without preserve_transitive attribute"""

            def filter_field(self, name: str, meta: dict) -> FilterResponse:
                tags = meta.get("tags", [])

                if "private" in tags:
                    return FilterResponse.REMOVE_STRICT  # Remove without preservation
                elif "admin" in tags:
                    return FilterResponse.REMOVE  # Remove with preservation
                else:
                    return FilterResponse.ALLOW_TRANSITIVE  # Keep with preservation

        class DataType:
            @field
            def public_field(self) -> str:
                return "public"

            @field({"tags": ["private"]})
            def private_field(self) -> str:
                return "private"

            @field({"tags": ["admin"]})
            def admin_field(self) -> str:
                return "admin"

        class Root:
            @field
            def data(self) -> DataType:
                return DataType()

        # This should work without errors (no preserve_transitive attribute access)
        api = GraphQLAPI(root_type=Root, filters=[CustomFilter()])
        schema, _ = api.build_schema()
        executor = api.executor()

        # Verify schema structure
        assert "DataType" in schema.type_map
        from graphql import GraphQLObjectType

        data_type = schema.type_map["DataType"]
        assert isinstance(data_type, GraphQLObjectType)

        # Should have public field but not private/admin fields
        assert "publicField" in data_type.fields
        assert "privateField" not in data_type.fields
        assert "adminField" not in data_type.fields

        # Test query execution
        result = executor.execute(
            """
            query TestQuery {
                data {
                    publicField
                }
            }
        """
        )

        assert not result.errors
        assert result.data == {"data": {"publicField": "public"}}

    def test_unused_mutable_types_filtered_out(self):
        """
        Test that mutable object types that are not used from the root mutation type are filtered out.
        Only mutable types that are actually reachable from the root should remain in the schema.

        This test verifies that:
        1. Unused mutable types (UnusedMutableType, AnotherUnusedMutableType) are not included in the schema
        2. Used mutable types (UsedMutableType) are correctly included
        3. The filtering works correctly even with the fixed field.type assignment
        4. Query fields are preserved on mutable types (unless marked with resolve_to_mutable: True)
        """
        from graphql_api.decorators import field

        class UsedMutableType:
            """This type will be referenced from the root mutation"""

            def __init__(self):
                self._value = "used"

            @field
            def value(self) -> str:
                return self._value

            @field(mutable=True)
            def update_value(self, new_value: str) -> "UsedMutableType":
                self._value = new_value
                return self

        class UnusedMutableType:
            """This type will NOT be referenced from the root mutation"""

            def __init__(self):
                self._data = "unused"

            @field
            def data(self) -> str:
                return self._data

            @field(mutable=True)
            def update_data(self, new_data: str) -> "UnusedMutableType":
                self._data = new_data
                return self

        class AnotherUnusedMutableType:
            """Another unused mutable type to make the test more comprehensive"""

            def __init__(self):
                self._info = "also unused"

            @field
            def info(self) -> str:
                return self._info

            @field(mutable=True)
            def update_info(self, new_info: str) -> "AnotherUnusedMutableType":
                self._info = new_info
                return self

        class Root:
            @field
            def used_object(self) -> UsedMutableType:
                return UsedMutableType()

            @field(mutable=True)
            def create_used_object(self, value: str) -> UsedMutableType:
                obj = UsedMutableType()
                obj._value = value
                return obj

            # Note: We deliberately don't reference UnusedMutableType or AnotherUnusedMutableType

        # Build the API
        api = GraphQLAPI(root_type=Root)
        schema, _ = api.build_schema()

        # Check that only used types are in the schema
        type_map = schema.type_map

        # UsedMutableType should be present (both in query and mutation forms)
        assert "UsedMutableType" in type_map
        assert "UsedMutableTypeMutable" in type_map

        # UnusedMutableType should NOT be present in either form
        assert "UnusedMutableType" not in type_map
        assert "UnusedMutableTypeMutable" not in type_map

        # AnotherUnusedMutableType should NOT be present in either form
        assert "AnotherUnusedMutableType" not in type_map
        assert "AnotherUnusedMutableTypeMutable" not in type_map

        # Verify the mutation schema only contains used types
        mutation_type = schema.mutation_type
        assert mutation_type is not None

        # Should have create_used_object mutation field
        assert "createUsedObject" in mutation_type.fields

        # Should be able to access the used object's mutable methods
        assert "usedObject" in mutation_type.fields

        # Test that mutation operations work correctly
        executor = api.executor()

        # Test creating and updating the used object
        result = executor.execute(
            """
            mutation TestMutation {
                createUsedObject(value: "test") {
                    updateValue(newValue: "updated") {
                        value
                    }
                }
            }
        """
        )

        assert not result.errors
        assert result.data == {
            "createUsedObject": {"updateValue": {"value": "updated"}}
        }

        # Test that we can't access unused types (they shouldn't exist in the schema)
        # This mutation should fail at schema build time, not execution time
        try:
            _ = executor.execute(
                """
                mutation BadMutation {
                    unusedObject {
                        updateData(newData: "test") {
                            data
                        }
                    }
                }
            """
            )
            # If we get here, the unused type wasn't properly filtered out
            assert False, "Unused mutable type should not be accessible in mutations"
        except Exception:
            # Expected - the field shouldn't exist
            pass

    def test_filter_mutable_fields(self):
        """
        Test filtering of mutable fields in both query and mutation contexts
        """
        from graphql_api.reduce import TagFilter
        from graphql_api.decorators import field

        class User:
            def __init__(self):
                self._name = "John"
                self._email = "john@example.com"
                self._admin_notes = "secret notes"

            @field
            def name(self) -> str:
                return self._name

            @field
            def email(self) -> str:
                return self._email

            @field({"tags": ["admin"]})
            def admin_notes(self) -> str:
                return self._admin_notes

            @field(mutable=True)
            def update_name(self, name: str) -> "User":
                self._name = name
                return self

            @field({"tags": ["admin"]}, mutable=True)
            def update_admin_notes(self, notes: str) -> "User":
                self._admin_notes = notes
                return self

            @field({"tags": ["admin"]}, mutable=True)
            def delete_user(self) -> bool:
                return True

        class Root:
            @field
            def user(self) -> User:
                return User()

        # Test with admin filter
        filtered_api = GraphQLAPI(root_type=Root, filters=[TagFilter(tags=["admin"])])
        executor = filtered_api.executor()

        # Query should work for non-admin fields
        query_test = """
            query GetUser {
                user {
                    name
                    email
                }
            }
        """
        result = executor.execute(query_test)
        assert not result.errors
        assert result.data == {"user": {"name": "John", "email": "john@example.com"}}

        # Query should fail for admin fields
        admin_query_test = """
            query GetAdminNotes {
                user {
                    adminNotes
                }
            }
        """
        result = executor.execute(admin_query_test)
        assert result.errors
        assert "Cannot query field 'adminNotes'" in str(result.errors[0])

        # Mutation should work for non-admin mutable fields
        mutation_test = """
            mutation UpdateName {
                user {
                    updateName(name: "Jane") {
                        name
                    }
                }
            }
        """
        result = executor.execute(mutation_test)
        assert not result.errors
        assert result.data == {"user": {"updateName": {"name": "Jane"}}}

        # Mutation should fail for admin mutable fields
        admin_mutation_test = """
            mutation UpdateAdminNotes {
                user {
                    updateAdminNotes(notes: "new notes") {
                        name
                    }
                }
            }
        """
        result = executor.execute(admin_mutation_test)
        assert result.errors
        assert "Cannot query field 'updateAdminNotes'" in str(result.errors[0])

    def test_filter_interface_fields(self):
        """
        Test filtering of fields on interfaces and their implementations
        """
        from graphql_api.reduce import TagFilter
        from graphql_api.decorators import field

        class Animal:
            @field
            def name(self) -> str:
                pass

            @field({"tags": ["vet"]})
            def medical_history(self) -> str:
                pass

        class Dog(Animal):
            def __init__(self):
                self._name = "Buddy"

            @field
            def name(self) -> str:
                return self._name

            @field({"tags": ["vet"]})
            def medical_history(self) -> str:
                return "Vaccinated"

            @field
            def breed(self) -> str:
                return "Golden Retriever"

            @field({"tags": ["vet"]})
            def vet_notes(self) -> str:
                return "Healthy dog"

        class Root:
            @field
            def dog(self) -> Dog:
                return Dog()

            @field({"tags": ["vet"]})
            def vet_data(self) -> str:
                return "Only for vets"

        # Test with vet filter
        filtered_api = GraphQLAPI(root_type=Root, filters=[TagFilter(tags=["vet"])])
        executor = filtered_api.executor()

        # Should work for non-vet fields
        test_query = """
            query GetAnimals {
                dog {
                    name
                    breed
                }
            }
        """
        result = executor.execute(test_query)
        assert not result.errors
        expected = {"dog": {"name": "Buddy", "breed": "Golden Retriever"}}
        assert result.data == expected

        # Should fail for vet fields
        vet_query = """
            query GetVetInfo {
                dog {
                    medicalHistory
                }
            }
        """
        result = executor.execute(vet_query)
        assert result.errors
        assert "Cannot query field 'medicalHistory'" in str(result.errors[0])

        # Should fail for vet root fields
        vet_root_query = """
            query GetVetData {
                vetData
            }
        """
        result = executor.execute(vet_root_query)
        assert result.errors
        assert "Cannot query field 'vetData'" in str(result.errors[0])

    def test_filter_nested_types(self):
        """
        Test filtering with deeply nested type structures
        """
        from graphql_api.reduce import TagFilter
        from graphql_api.decorators import field

        class Address:
            @field
            def street(self) -> str:
                return "123 Main St"

            @field({"tags": ["private"]})
            def apartment_number(self) -> str:
                return "Apt 4B"

        class ContactInfo:
            @field
            def email(self) -> str:
                return "user@example.com"

            @field({"tags": ["private"]})
            def phone(self) -> str:
                return "555-0123"

            @field
            def address(self) -> Address:
                return Address()

        class Profile:
            @field
            def bio(self) -> str:
                return "Software developer"

            @field({"tags": ["private"]})
            def salary(self) -> int:
                return 75000

            @field
            def contact(self) -> ContactInfo:
                return ContactInfo()

        class User:
            @field
            def username(self) -> str:
                return "johndoe"

            @field
            def profile(self) -> Profile:
                return Profile()

        class Root:
            @field
            def user(self) -> User:
                return User()

        # Test with private filter
        filtered_api = GraphQLAPI(root_type=Root, filters=[TagFilter(tags=["private"])])
        executor = filtered_api.executor()

        # Should work for non-private nested fields
        test_query = """
            query GetUserData {
                user {
                    username
                    profile {
                        bio
                        contact {
                            email
                            address {
                                street
                            }
                        }
                    }
                }
            }
        """
        result = executor.execute(test_query)
        assert not result.errors
        expected = {
            "user": {
                "username": "johndoe",
                "profile": {
                    "bio": "Software developer",
                    "contact": {
                        "email": "user@example.com",
                        "address": {"street": "123 Main St"},
                    },
                },
            }
        }
        assert result.data == expected

        # Should fail for private fields at any level
        private_query = """
            query GetPrivateData {
                user {
                    profile {
                        salary
                    }
                }
            }
        """
        result = executor.execute(private_query)
        assert result.errors
        assert "Cannot query field 'salary'" in str(result.errors[0])

    def test_filter_list_and_optional_fields(self):
        """
        Test filtering with list and optional field types
        """
        from graphql_api.reduce import TagFilter
        from graphql_api.decorators import field
        from typing import List, Optional

        class Tag:
            def __init__(self, name: str):
                self._name = name

            @field
            def name(self) -> str:
                return self._name

            @field({"tags": ["internal"]})
            def internal_id(self) -> int:
                return 123

        class Post:
            @field
            def title(self) -> str:
                return "My Post"

            @field
            def tags(self) -> List[Tag]:
                return [Tag("python"), Tag("graphql")]

            @field({"tags": ["internal"]})
            def internal_tags(self) -> Optional[List[Tag]]:
                return [Tag("draft")]

            @field
            def optional_summary(self) -> Optional[str]:
                return None

            @field({"tags": ["admin"]})
            def admin_notes(self) -> Optional[str]:
                return "Admin only note"

        class Root:
            @field
            def post(self) -> Post:
                return Post()

            @field({"tags": ["internal"]})
            def internal_posts(self) -> List[Post]:
                return [Post()]

        # Test with internal and admin filters
        filtered_api = GraphQLAPI(
            root_type=Root, filters=[TagFilter(tags=["internal", "admin"])]
        )
        executor = filtered_api.executor()

        # Should work for non-filtered list/optional fields
        test_query = """
            query GetPost {
                post {
                    title
                    tags {
                        name
                    }
                    optionalSummary
                }
            }
        """
        result = executor.execute(test_query)
        assert not result.errors
        expected = {
            "post": {
                "title": "My Post",
                "tags": [{"name": "python"}, {"name": "graphql"}],
                "optionalSummary": None,
            }
        }
        assert result.data == expected

        # Should fail for filtered fields
        internal_query = """
            query GetInternalData {
                post {
                    internalTags {
                        name
                    }
                }
            }
        """
        result = executor.execute(internal_query)
        assert result.errors
        assert "Cannot query field 'internalTags'" in str(result.errors[0])

    def test_filter_union_types(self):
        """
        Test filtering with union types
        """
        from graphql_api.reduce import TagFilter
        from graphql_api.decorators import field
        from typing import Union

        class PublicContent:
            @field
            def title(self) -> str:
                return "Public Content"

            @field
            def content(self) -> str:
                return "This is public"

        class PrivateContent:
            @field({"tags": ["admin"]})
            def title(self) -> str:
                return "Private Content"

            @field({"tags": ["admin"]})
            def secret_data(self) -> str:
                return "Secret information"

        class MixedContent:
            @field
            def public_field(self) -> str:
                return "Public"

            @field({"tags": ["admin"]})
            def private_field(self) -> str:
                return "Private"

        class Root:
            @field
            def content(self) -> Union[PublicContent, PrivateContent, MixedContent]:
                return PublicContent()

            @field
            def mixed_content(self) -> Union[PublicContent, MixedContent]:
                return MixedContent()

        # Test with admin filter
        filtered_api = GraphQLAPI(root_type=Root, filters=[TagFilter(tags=["admin"])])
        executor = filtered_api.executor()

        # Should work for public union types
        test_query = """
            query GetContent {
                content {
                    ... on PublicContent {
                        title
                        content
                    }
                }
                mixedContent {
                    ... on PublicContent {
                        title
                        content
                    }
                    ... on MixedContent {
                        publicField
                    }
                }
            }
        """
        result = executor.execute(test_query)
        assert not result.errors
        expected = {
            "content": {"title": "Public Content", "content": "This is public"},
            "mixedContent": {"publicField": "Public"},
        }
        assert result.data == expected

        # PrivateContent should be removed from union since all its fields are filtered
        # This should still work but PrivateContent won't be available
        private_query = """
            query GetPrivateContent {
                content {
                    ... on PrivateContent {
                        title
                    }
                }
            }
        """
        result = executor.execute(private_query)
        # This should execute without errors but return no data for PrivateContent
        assert not result.errors
        assert result.data == {"content": {}}

    def test_filter_multiple_criteria(self):
        """
        Test filtering with multiple filter criteria
        """
        from graphql_api.reduce import TagFilter, FilterResponse
        from graphql_api.decorators import field

        class CustomFilter(TagFilter):
            def filter_field(self, name: str, meta: dict) -> FilterResponse:
                # Filter out fields with 'admin' tag OR fields starting with 'internal_'
                parent_response = super().filter_field(name, meta)
                if parent_response.should_filter:
                    return parent_response

                # Filter fields starting with 'internal_'
                if name.startswith("internal_"):
                    return (
                        FilterResponse.REMOVE
                        if self.preserve_transitive
                        else FilterResponse.REMOVE_STRICT
                    )
                else:
                    return FilterResponse.ALLOW_TRANSITIVE

        class Data:
            @field
            def public_data(self) -> str:
                return "public"

            @field({"tags": ["admin"]})
            def admin_data(self) -> str:
                return "admin only"

            @field
            def internal_data(self) -> str:
                return "internal"

            @field({"tags": ["user"]})
            def internal_user_data(self) -> str:
                return "internal user data"

        class Root:
            @field
            def data(self) -> Data:
                return Data()

        # Test with custom filter
        filtered_api = GraphQLAPI(
            root_type=Root, filters=[CustomFilter(tags=["admin"])]
        )
        executor = filtered_api.executor()

        # Should be able to query public data
        test_query = """
            query GetData {
                data {
                    publicData
                }
            }
        """

        result = executor.execute(test_query)
        assert not result.errors
        assert result.data == {"data": {"publicData": "public"}}

        # Should NOT be able to query admin data (filtered by tag)
        admin_query = """
            query GetAdminData {
                data {
                    adminData
                }
            }
        """

        result = executor.execute(admin_query)
        assert result.errors
        assert "Cannot query field 'adminData'" in str(result.errors[0])

        # Should NOT be able to query internal data (filtered by name)
        internal_query = """
            query GetInternalData {
                data {
                    internalData
                }
            }
        """

        result = executor.execute(internal_query)
        assert result.errors
        assert "Cannot query field 'internalData'" in str(result.errors[0])

        # Should NOT be able to query internal user data (filtered by name, even though tag is not in filter)
        internal_user_query = """
            query GetInternalUserData {
                data {
                    internalUserData
                }
            }
        """

        result = executor.execute(internal_user_query)
        assert result.errors
        assert "Cannot query field 'internalUserData'" in str(result.errors[0])

    def test_filter_empty_mutation_type(self):
        """
        Test that filtering can remove all mutable fields leaving empty mutation type
        """
        from graphql_api.reduce import TagFilter
        from graphql_api.decorators import field

        class User:
            def __init__(self):
                self._name = "John"

            @field
            def name(self) -> str:
                return self._name

            @field({"tags": ["admin"]}, mutable=True)
            def update_name(self, name: str) -> "User":
                self._name = name
                return self

            @field({"tags": ["admin"]}, mutable=True)
            def delete_user(self) -> bool:
                return True

        class Root:
            @field
            def user(self) -> User:
                return User()

            @field({"tags": ["admin"]}, mutable=True)
            def create_user(self, name: str) -> User:
                return User()

        # Filter out all admin fields
        filtered_api = GraphQLAPI(root_type=Root, filters=[TagFilter(tags=["admin"])])
        executor = filtered_api.executor()

        # Query should still work
        test_query = """
            query GetUser {
                user {
                    name
                }
            }
        """
        result = executor.execute(test_query)
        assert not result.errors
        assert result.data == {"user": {"name": "John"}}

        # Mutation should fail since all mutable fields are filtered
        mutation_query = """
            mutation UpdateUser {
                user {
                    updateName(name: "Jane") {
                        name
                    }
                }
            }
        """
        result = executor.execute(mutation_query)
        assert result.errors
        # Should fail because updateName field is filtered out

    def test_recursive_object_type_preservation(self):
        """
        Test that object types on fields of unfiltered objects (recursive)
        are still left on the schema, even if the referenced types would
        otherwise be filtered out, as long as the referenced types have at least one accessible field.
        Types with NO accessible fields cannot be preserved due to GraphQL constraints.
        """
        from graphql_api.reduce import TagFilter
        from graphql_api.decorators import field

        # Deepest nested type - has one accessible field after filtering
        class DeepConfig:
            @field({"tags": ["admin"]})
            def secret_key(self) -> str:
                return "secret"

            @field  # This field will remain accessible
            def public_setting(self) -> str:
                return "public setting"

        # Middle type - references DeepConfig, has unfiltered fields
        class MiddleConfig:
            @field
            def public_setting(self) -> str:
                return "public"

            @field
            def deep_config(self) -> DeepConfig:
                return DeepConfig()

        # Root type - references MiddleConfig and has unfiltered fields
        class AppConfig:
            @field
            def app_name(self) -> str:
                return "MyApp"

            @field
            def middle_config(self) -> MiddleConfig:
                return MiddleConfig()

        class Root:
            @field
            def config(self) -> AppConfig:
                return AppConfig()

        # Create filtered API that removes admin fields using preserve_transitive=True (default)
        filtered_api = GraphQLAPI(
            root_type=Root,
            filters=[TagFilter(tags=["admin"], preserve_transitive=True)],
        )

        # Build the schema to check its structure
        schema, _ = filtered_api.build_schema()
        type_map = schema.type_map

        # All object types should be preserved because they're reachable
        # from unfiltered fields and DeepConfig has at least one accessible field
        assert "AppConfig" in type_map
        assert "MiddleConfig" in type_map
        assert "DeepConfig" in type_map  # This should be preserved!

        # Verify that the fields referencing these types are preserved
        assert schema.query_type is not None
        root_fields = schema.query_type.fields
        assert "config" in root_fields

        from graphql import GraphQLObjectType

        app_config_type = type_map["AppConfig"]
        assert isinstance(app_config_type, GraphQLObjectType)
        assert "appName" in app_config_type.fields
        assert "middleConfig" in app_config_type.fields

        middle_config_type = type_map["MiddleConfig"]
        assert isinstance(middle_config_type, GraphQLObjectType)
        assert "publicSetting" in middle_config_type.fields
        assert "deepConfig" in middle_config_type.fields  # This should be preserved!

        # DeepConfig should exist with accessible fields (admin field filtered out)
        deep_config_type = type_map["DeepConfig"]
        assert isinstance(deep_config_type, GraphQLObjectType)
        # The admin field should be removed from DeepConfig
        assert "secretKey" not in deep_config_type.fields
        # But the public field should remain
        assert "publicSetting" in deep_config_type.fields
        assert len(deep_config_type.fields) == 1

        # Test that queries work for the preserved structure
        executor = filtered_api.executor()
        result = executor.execute(
            """
            query GetConfig {
                config {
                    appName
                    middleConfig {
                        publicSetting
                        deepConfig {
                            publicSetting
                        }
                    }
                }
            }
        """
        )

        assert not result.errors
        expected = {
            "config": {
                "appName": "MyApp",
                "middleConfig": {
                    "publicSetting": "public",
                    "deepConfig": {"publicSetting": "public setting"},
                },
            }
        }
        assert result.data == expected
