from ctypes import (
    POINTER,
    Structure,
    c_char,
    c_int,
    c_uint8,
    c_uint16,
    c_uint32,
    c_ushort,
)

# Constants matching C implementation
WINDOW_SIZE = 16
MAX_PACKET_SIZE = 128
TIMEOUT_MS = 500
MAX_RETRIES = 3
BUFFER_SIZE = 256
# Packet flags
SYN = 0x01
ACK = 0x02
FIN = 0x04


# First define the network structures
class in_addr(Structure):
    _fields_ = [("s_addr", c_uint32)]


class sockaddr_in(Structure):
    _fields_ = [
        ("sin_family", c_ushort),
        ("sin_port", c_uint16),
        ("sin_addr", in_addr),
        ("sin_zero", c_char * 8),
    ]


class RUDPHeader(Structure):
    _fields_ = [
        ("flag", c_uint8),
        ("seq_num", c_uint8),
        ("ack_num", c_uint8),
        ("rwnd", c_uint8),
        ("data_len", c_uint16),
    ]


class RUDPPacket(Structure):
    _fields_ = [("header", RUDPHeader), ("data", c_char * MAX_PACKET_SIZE)]


class WindowSlot(Structure):
    _fields_ = [
        ("packet", RUDPPacket),
        ("send_time", c_uint32),
        ("retries", c_uint8),
    ]


class SenderWnd(Structure):
    _fields_ = [
        ("packets_buffer", WindowSlot * BUFFER_SIZE),
        ("base_seq", c_uint8),
        ("next_seq", c_uint8),
        ("unacked_packet_count", c_uint8),
        ("peer_rwnd", c_uint8),
    ]

    def __init__(
        self, base_seq: int, next_seq: int, unacked_packet_count: int, peer_rwnd: int
    ) -> None:
        self.base_seq = base_seq
        self.next_seq = next_seq
        self.unacked_packet_count = unacked_packet_count
        self.peer_rwnd = peer_rwnd


class RUDPConn(Structure):
    _fields_ = [
        ("sockfd", c_int),
        ("peer", sockaddr_in),  # Now using proper sockaddr_in structure
        ("send_window", POINTER(SenderWnd)),
        ("connected", c_uint8),
    ]

    @property
    def window(self):
        return self.send_window.contents
