"""
Given a known package, find the 'central' module, the one whose version should sync with the package
"""

from __future__ import annotations

import logging
import os
from pathlib import Path
from typing import List, Optional

from jiggle_version.file_opener import FileOpener
from jiggle_version.module_finder import ModuleFinder

logger = logging.getLogger(__name__)


class CentralModuleFinder:
    """
    Finds modules in a folder. No assumptions about existence of PKG_INFO
    """

    def __init__(self, file_opener: FileOpener) -> None:
        """
        Initialize object
        :param file_opener:
        """
        self.file_opener = file_opener
        self.setup_source: Optional[str] = ""

        self.setup_file_name: Path | None = None
        self.find_setup_file_name()
        self.read_setup_py_source()
        self.package_name = self.parse_package_name()

    def find_setup_file_name(self) -> None:
        """
        Usually setup.py or setup
        """
        for file_path in [
            Path(x)
            for x in os.listdir(".")
            if Path(x).is_file() and x in [Path("setup.py"), Path("setup")]
        ]:
            if self.file_opener.is_python_inside(file_path):
                self.setup_file_name = file_path
                break

    def _read_file(self, file: Path) -> Optional[str]:
        """
        Read any file, deal with encoding.
        """
        if not self.setup_file_name:
            return None
        source = None
        if file.is_file():
            with self.file_opener.open_this(file, "r") as setup_py:
                source = setup_py.read()
        return source

    def read_setup_py_source(self) -> None:
        """
        Read setup.py to string
        :return:
        """
        if not self.setup_file_name:
            self.setup_source = ""
        if not self.setup_source and self.setup_file_name:
            self.setup_source = self._read_file(self.setup_file_name)

    def parse_package_name(self) -> Optional[str]:
        """
        Extract likley module name from setup.py args
        :return:
        """
        if not self.setup_source:
            return None
        for row in self.setup_source.split("\n"):
            simplified_row = row.replace(" ", "").replace("'", '"').strip(" \t\n")
            if "name=" in simplified_row:
                if '"' in simplified_row:
                    return simplified_row.split('"')[1]

        name = ""  # self.execute_setup_name()
        if name:
            return name

        return ""

    def find_central_module(self) -> Optional[Path]:
        """
        Get the module that is the sole module, or the module
        that matches the package name/version
        :return:
        """
        # find modules.
        module_finder = ModuleFinder(self.file_opener)

        candidates = module_finder.find_by_any_method()

        sub_modules = []
        root_modules = []

        for candidate in candidates:
            if str(Path(".")) in str(candidate):
                sub_modules.append(candidate)
            else:
                root_modules.append(candidate)

        candidates = root_modules

        # remove junk. Junk only has meaning in the sense of finding the central module.
        candidates = self.remove_likely_non_central(candidates)
        if len(candidates) == 1:
            return candidates[0]

        # see if there is 1 out of the many with same name pkg_foo, module_foo
        if self.package_name:
            if self.package_name in candidates:
                return Path(self.package_name)

        # I don't understand the _ to - transformations.
        if self.package_name:
            if self.package_name.replace("-", "_") in candidates:
                return Path(str(self.package_name).replace("-", "_"))

        if self.package_name:
            if self.package_name.replace("-", "") in candidates:
                return Path(str(self.package_name).replace("-", ""))

        if self.package_name:
            if self.package_name.replace("_", "") in candidates:
                return Path(str(self.package_name).replace("_", ""))
        # see if there is 1 out of the many with version in sync- pkg_foo v1.2.3, module_bar v1.2.3
        # TODO:

        return None

    def remove_likely_non_central(self, candidates: List[Path]) -> List[Path]:
        """
        Stuff that is likely to be in find_packages(exclude...)
        :param candidates:
        :return:
        """
        if len(candidates) > 1:
            for unlikely_str in [
                "test",
                "tests",
                "example",
                "examples",
                "demo",
                "demos",
                "test_files",
                "doc",
                "docs",
            ]:
                unlikely = Path(unlikely_str)
                if unlikely in candidates:
                    logger.warning(f"Assuming {unlikely} is not the project")
                    candidates.remove(unlikely)
                for candidate in candidates:
                    if candidate.name.startswith(str(unlikely)):
                        logger.warning(f"Assuming {candidate} is not the project")
                        candidates.remove(candidate)
        return candidates
