"""
Smart argument parser with typo suggestions.

This module provides a subclass of `argparse.ArgumentParser` that enhances
the error reporting behavior when users supply invalid choices. If a user
makes a typo in a choice, the parser will suggest the closest matches
based on string similarity.

Example:
    ```python
    import sys

    parser = SmartParser(prog="myapp")
    parser.add_argument("color", choices=["red", "green", "blue"])
    args = parser.parse_args()

    # If the user runs:
    #   myapp gren
    #
    # The output will include:
    #   error: invalid choice: 'gren' (choose from 'red', 'green', 'blue')
    #
    #   Did you mean: green?
    ```
"""

from __future__ import annotations

import argparse
import sys
from difflib import get_close_matches


class SmartParser(argparse.ArgumentParser):
    """Argument parser that suggests similar choices on invalid input.

    This class extends `argparse.ArgumentParser` to provide more helpful
    error messages when the user provides an invalid choice for an argument.
    Instead of only showing the list of valid choices, it also suggests the
    closest matches using fuzzy string matching.

    Example:
        ```python
        parser = SmartParser()
        parser.add_argument("fruit", choices=["apple", "banana", "cherry"])
        args = parser.parse_args()
        ```

    If the user types:
        ```
        myprog bannna
        ```

    The error message will include:
        ```
        Did you mean: banana?
        ```
    """

    def error(self, message: str):
        """Handle parsing errors with suggestions for invalid choices.

        Args:
            message (str): The error message generated by argparse,
                typically when parsing fails (e.g., due to invalid
                choices or syntax errors).

        Side Effects:
            - Prints usage information to `sys.stderr`.
            - Exits the program with status code 2.

        Behavior:
            - If the error message contains an "invalid choice" message,
              attempts to suggest the closest valid alternatives by
              computing string similarity.
            - Otherwise, preserves standard argparse behavior.
        """
        # Detect "invalid choice: 'foo' (choose from ...)"
        if "invalid choice" in message and "choose from" in message:
            bad = message.split("invalid choice:")[1].split("(")[0].strip().strip("'\"")
            choices_str = message.split("choose from")[1]
            choices = [
                c.strip().strip(",)'") for c in choices_str.split() if c.strip(",)")
            ]

            tips = get_close_matches(bad, choices, n=3, cutoff=0.6)
            if tips:
                message += f"\n\nDid you mean: {', '.join(tips)}?"

        self.print_usage(sys.stderr)
        self.exit(2, f"{self.prog}: error: {message}\n")
