param functionAppName string
param rgManagedIdentityName string
param generalKeyVaultName string
param generalKeyVaultRG string
param functionAppOperationalDays string
param functionAppOperationalHours string
param envKeyVaultName string
param envKeyVaultSecretOfficerUserIds array

var location = resourceGroup().location
var appServicePlanName = '${functionAppName}-plan'
var storageAccountName = '${uniqueString(resourceGroup().id)}sa'
var applicationInsightsName = '${functionAppName}-insights'
var tenantId = subscription().tenantId

/////////////////////////////////////////////////////////////////////////
// Role Definitions

var contributorDefinitionId = subscriptionResourceId('Microsoft.Authorization/roleDefinitions', 'b24988ac-6180-42a0-ab88-20f7382dd24c')
var websiteContributorDefinitionId = subscriptionResourceId('Microsoft.Authorization/roleDefinitions', 'de139f84-1756-47ae-9be6-808fbbe84772')
var keyvaultSecretsUserDefinitionId = subscriptionResourceId('Microsoft.Authorization/roleDefinitions', '4633458b-17de-408a-b874-0445c86b69e6')
var keyvaultSecretsOfficerDefinitionId = subscriptionResourceId('Microsoft.Authorization/roleDefinitions', 'b86a8fe4-44ce-4948-aee5-eccb2c155cd7')

/////////////////////////////////////////////////////////////////////////
// Function App

resource storageAccount 'Microsoft.Storage/storageAccounts@2023-05-01' = {
  name: storageAccountName
  location: location
  sku: {
    name: 'Standard_GRS'
  }
  kind: 'StorageV2'
  properties: {
    allowBlobPublicAccess: false
    supportsHttpsTrafficOnly: true
    minimumTlsVersion: 'TLS1_2'
    encryption: {
      services: {
        blob: { enabled: true }
      }
      keySource: 'Microsoft.Storage'
    }
  }
}

resource appServicePlan 'Microsoft.Web/serverfarms@2024-04-01' = {
  name: appServicePlanName
  location: location
  sku: {
    name: 'Y1'
    tier: 'Dynamic'
  }
  properties: {
    reserved: true
  }
}

resource applicationInsights 'Microsoft.Insights/components@2020-02-02' = {
  name: applicationInsightsName
  location: location
  kind: 'web'
  properties: {
    Application_Type: 'web'
    Request_Source: 'rest'
  }
}

resource functionApp 'Microsoft.Web/sites@2024-04-01' = {
  name: functionAppName
  location: location
  kind: 'functionapp,linux'
  identity: {
    type: 'SystemAssigned'
  }
  properties: {
    serverFarmId: appServicePlan.id
    httpsOnly: true
    siteConfig: {
      linuxFxVersion: 'PYTHON|3.11'
      appSettings: [
        {
          name: 'AzureWebJobsStorage'
          value: 'DefaultEndpointsProtocol=https;AccountName=${storageAccount.name};EndpointSuffix=${environment().suffixes.storage};AccountKey=${storageAccount.listKeys().keys[0].value}'
        }
        {
          name: 'FUNCTIONS_EXTENSION_VERSION'
          value: '~4'
        }
        {
          name: 'APPINSIGHTS_INSTRUMENTATIONKEY'
          value: applicationInsights.properties.InstrumentationKey
        }
        {
          name: 'APPLICATIONINSIGHTS_CONNECTION_STRING'
          value: applicationInsights.properties.ConnectionString
        }
        {
          name: 'FUNCTIONS_WORKER_RUNTIME'
          value: 'python'
        }
      ]
    }
  }
  tags: {
    UtilitiesOperationalDays: functionAppOperationalDays
    UtilitiesOperationalHours: functionAppOperationalHours
  }
}

output functionAppPrincipalId string = functionApp.identity.principalId

// Enable remote publishing policy

resource scmCredentialPolicy 'Microsoft.Web/sites/basicPublishingCredentialsPolicies@2022-03-01' = {
  parent: functionApp
  name: 'scm'
  properties: {
    allow: true
  }
}

resource rgManagedIdentity 'Microsoft.ManagedIdentity/userAssignedIdentities@2024-11-30' existing = {
  name: rgManagedIdentityName
}

// Assign 'contributor' to resource group managed identity on function app

resource rgMIRoleAssignment 'Microsoft.Authorization/roleAssignments@2022-04-01' = {
  scope: functionApp
  name: guid(functionApp.id, rgManagedIdentity.id, contributorDefinitionId)
  properties: {
    roleDefinitionId: contributorDefinitionId
    principalId: rgManagedIdentity.properties.principalId
    principalType: 'ServicePrincipal'
  }
}

// Assign 'website contributor' to resource group managed identity on function app

resource rgMIWebsiteContributorRoleAssignment 'Microsoft.Authorization/roleAssignments@2022-04-01' = {
  scope: functionApp
  name: guid(functionApp.id, rgManagedIdentity.id, websiteContributorDefinitionId)
  properties: {
    roleDefinitionId: websiteContributorDefinitionId
    principalId: rgManagedIdentity.properties.principalId
    principalType: 'ServicePrincipal'
  }
}

// Assign 'secret user' to function app mi on general key vault

resource generalKV 'Microsoft.KeyVault/vaults@2023-07-01' existing = {
  name:  generalKeyVaultName
  scope: resourceGroup(generalKeyVaultRG)
}

module keyVaultRoleAssignment './kvRoleAssignment.bicep' = {
  name: guid(generalKV.id, functionApp.id, keyvaultSecretsUserDefinitionId)
  scope: resourceGroup(generalKeyVaultRG)
  params: {
    keyVaultName: generalKeyVaultName
    principalId: functionApp.identity.principalId
    roleDefinitionId: keyvaultSecretsUserDefinitionId
  }
}

/////////////////////////////////////////////////////////////////////////
// Environment Key Vault

resource envKeyVault 'Microsoft.KeyVault/vaults@2023-07-01' = {
  name: envKeyVaultName
  location: location
  properties: {
    tenantId: tenantId
    sku: {
      name: 'standard'
      family: 'A'
    }
    enableRbacAuthorization: true
    networkAcls: {
      defaultAction: 'Allow'
      bypass: 'AzureServices'
    }
  }
}

// Assign 'secrets officer' to function app on environment key vault

resource envKeyVaultRoleAssignment 'Microsoft.Authorization/roleAssignments@2022-04-01' = {
  scope: envKeyVault
  name: guid(envKeyVault.id, functionApp.id, keyvaultSecretsOfficerDefinitionId)
  properties: {
    roleDefinitionId: keyvaultSecretsOfficerDefinitionId
    principalId: functionApp.identity.principalId
    principalType: 'ServicePrincipal'
  }
}

// Assign 'secrets officer' to users on environment key vault

resource envKeyVaultRoleAssignmentUser 'Microsoft.Authorization/roleAssignments@2022-04-01' = [for userId in envKeyVaultSecretOfficerUserIds: {
  scope: envKeyVault
  name: guid(envKeyVault.id, userId, keyvaultSecretsOfficerDefinitionId)
  properties: {
    roleDefinitionId: keyvaultSecretsOfficerDefinitionId
    principalId: userId
  }
}]

