def generate_notebook(prompt, filename=None, image=None, modify_existing_notebook=False, max_number_attempts=3):
    """
    Generates a notebook from a given prompt and saves it to a file.
    If the filename is not specified, a filename will be generated by the language model.
    If the specified file exists already, it will be overwritten.

    Parameters
    ----------
    prompt: str
    filename: str, optional
    image: ndarray, optional
        can be considered for generating the notebook if the used model has vision capabilities.
    modify_existing_notebook: bool, optional
        If True, the notebook will be modified instead of creating a new one.
    max_number_attempts: int, optional
        The maximum number of attempts to generate a valid notebook.

    Returns
    -------
    filename
    """
    from ._utilities import create_system_prompt, generate_response, remove_outer_markdown_annotation
    from ._machinery import Context, init_assistant
    import os
    import json

    if Context.model is None:
        init_assistant()

    system_prompt = create_system_prompt().split("## Todos")[0] + f"""
    You have been asked to generate a notebook in the ipynb format. 
    Start the notebook with an introduction of the task.
    List the steps your are planning to do in the notebook. Do not use Python yet.
    Also labeling is an individual processing step. 
    Import the libraries you are planning to use.
    Write one code cell for each processing step in the analysis. 
    For each individual processing step, write a markdown cell explaining what you are doing.
    By the end of each cell write code for showing the intermediate result:
    * Use stackview.insight() for showing images.
    * Use display() for showing dataframes.
    * Use print() for any other results.
    In the ipynb-json format you write, there is no output in any of the cells you write.
    There are no images displayed in the notebook you write.
    There is no explanatory text around the notebook, only within the notebook.
    Make sure there are line breaks after each code line.
    
    Make sure your output is in valid ipynb/json format.
    """
    
    for attempt in range(1, max_number_attempts + 1):
        if modify_existing_notebook:
            try:
                import ipynbname
                current_notebook_filename = ipynbname.name()
            except:
                current_notebook_filename = "Untitled"

            filename_extraction_prompt = f"""
            Extract the filename of the notebook that should be modified from the following prompt:
    
            ```json
            {prompt}
            ```
            
            If the prompt mentions the current notebook, respond with '{current_notebook_filename}.ipynb'.
    
            Respond with the filename and nothing else.
            """
            filename = generate_response(chat_history=[], image=None, model=Context.model, system_prompt="", user_prompt=filename_extraction_prompt, vision_system_prompt="")

            # Load the notebook as a JSON object
            with open(filename, 'r', encoding='utf-8') as f:
                notebook_content = json.load(f)

            # Iterate through the cells and erase the outputs
            for cell in notebook_content['cells']:
                if cell['cell_type'] == 'code':
                    cell['outputs'] = []
                    cell['execution_count'] = None

            notebook_text = json.dumps(notebook_content, indent=1)

            prompt = prompt + f"""
            The Notebook {filename} is given here:
            ```json
            {notebook_text}
            ```
            """

        full_response = generate_response(chat_history=[], image=image, model=Context.model, system_prompt=system_prompt, user_prompt=prompt, vision_system_prompt="")

        if filename is None:
            filename_prompt = f"""
            What would be a good filename for a notebook that answers the following prompt?
            
            Prompt:
            {prompt}
            
            Respond with the filename and nothing else.
            """

            filename = generate_response(chat_history=[], image=None, model=Context.model, system_prompt="", user_prompt=filename_prompt, vision_system_prompt="")
            filename = filename.strip().strip("`").strip("'").strip()
            if not filename.endswith(".ipynb"):
                filename += ".ipynb"

        # check if file not exists and suffix a number if it does
        filename = not_existing_filename(filename)

        full_response = remove_outer_markdown_annotation(full_response)

        # in case the response is no valid json, we try to convert it.
        try:
            json.loads(full_response)
        except:
            prompt = f"""
            Please take the following jupyter notebook ipynb file content and modify it so that the result is valid json:
            
            {full_response}
            
            Respond in ipynb/json format only, nothing else.
            """

            full_response = generate_response(chat_history=[], image=None, model=Context.model, system_prompt="", user_prompt=prompt, vision_system_prompt="")
            full_response = remove_outer_markdown_annotation(full_response)

        # in case it's still no valid JSON, we try again
        try:
            json_content = json.loads(full_response)
        except:
            print(f"There was an issue. Retrying ({attempt}/{max_number_attempts})...")
            continue

        from ._utilities import version_string
        from bia_bob import __version__ as version
        json_content["cells"] = [{
           "cell_type": "markdown",
           "id": "directed-compensation",
           "metadata": {},
           "source": ["<html><div style=\"font-size:7pt\">This notebook may contain text, code and images generated by artificial intelligence. " + version_string(Context.model, Context.vision_model, Context.endpoint, version) + ". It is good scientific practice to check the code and results it produces carefully. <a href=\"https://github.com/haesleinhuepf/bia-bob\">Read more about code generation using bia-bob</a></div></html>"]
        }] + json_content["cells"]

        # write the response to a file in utf-8 encoding
        with open(filename, "w", encoding="utf-8") as file:
            json.dump(json_content, file, indent=2)
            #file.write(json_content )

        return filename





def generate_file(prompt, filename=None, image=None, max_number_attempts=3):
    """
    Generates a file from a given prompt and saves it.
    If the filename is not specified, a filename will be generated by the language model.
    If the specified file exists already, it will be overwritten.

    Parameters
    ----------
    prompt
    filename, optional
    image: ndarray, optional
        can be considered for generating the file if the used model has vision capabilities.
    max_number_attempts: int, optional
        The maximum number of attempts to generate a valid file.

    Returns
    -------
    filename
    """
    from ._utilities import generate_response, remove_outer_markdown_annotation
    from ._machinery import Context, init_assistant

    if Context.model is None:
        init_assistant()

    separator = "## Requested File"

    system_prompt = f"""
    You will be asked to generate a file in a specified format in the following prompt. 
    Summarize the task first, list potential sub-tasks and then provide the solution in the specified format.
    Write the file after a "{separator}" headline. Only output the file content under this headline. 
    
    Prompt:
    {prompt}
    
    Make sure your output is in the right format as specified by the ending of the file.
    """

    for attempt in range(1, max_number_attempts + 1):

        full_response = generate_response(chat_history=[], image=image, model=Context.model,
                                          system_prompt=system_prompt, user_prompt=prompt, vision_system_prompt="")

        if separator not in full_response:
            print(f"There was an issue. Retrying ({attempt}/{max_number_attempts})...")
            continue

        bla_bla, file_content = full_response.split("## Requested File")
        file_content = remove_outer_markdown_annotation(file_content)

        if filename is None:
            filename_prompt = f"""
            Extract the filename from the given prompt.
    
            Prompt:
            {prompt}
    
            Respond with the filename and nothing else.
            """

            filename = generate_response(chat_history=[], image=None, model=Context.model, system_prompt="",
                                         user_prompt=filename_prompt, vision_system_prompt="")
            filename = filename.strip().strip("`").strip("'").strip()


        # check if file not exists and suffix a number if it does
        filename = not_existing_filename(filename)

        # write the response to a file in utf-8 encoding
        with open(filename, "w", encoding="utf-8") as file:
            file.write(file_content)

        return filename


def not_existing_filename(original_filename):
    """Checks if a given filename exists and if so, adds a number to the filename."""
    import os
    filename = original_filename
    if os.path.exists(filename):
        i = 1
        while os.path.exists(filename):
            ending = filename.split(".")[-1]
            filename = original_filename.replace("." + ending, f"_{i}." + ending)
            i += 1

    return filename
