import re
from django.db import models
from django.core import validators
from django.contrib.auth.models import AbstractBaseUser, PermissionsMixin, UserManager
from django.core.mail import send_mail
from django.utils import timezone

from kobo.django.compat import gettext_lazy as _


MAX_LENGTH = 255

class User(AbstractBaseUser, PermissionsMixin):
    """
    Copy (non-abstract) of AbstractUser with longer username. Removed profile support as it
    is deprecated in django 1.5.

    Username, password and email are required. Other fields are optional.
    """
    username = models.CharField(_('username'), max_length=MAX_LENGTH, unique=True,
        help_text=_('Required. %s characters or fewer. Letters, digits and @.+-_/ only.' % MAX_LENGTH),
        validators=[
            validators.RegexValidator(r'^[\w.@+\-/]+$',
                                      _('Enter a valid username. '
                                        'This value may contain only letters, numbers '
                                        'and @.+-_/ characters.'), 'invalid'),
        ],
        error_messages={
            'unique': _("A user with that username already exists."),
        })
    first_name = models.CharField(_('first name'), max_length=30, blank=True)
    last_name = models.CharField(_('last name'), max_length=30, blank=True)
    email = models.EmailField(_('email address'), blank=True)
    is_staff = models.BooleanField(_('staff status'), default=False,
        help_text=_('Designates whether the user can log into this admin '
                    'site.'))
    is_active = models.BooleanField(_('active'), default=True,
        help_text=_('Designates whether this user should be treated as '
                    'active. Unselect this instead of deleting accounts.'))
    date_joined = models.DateTimeField(_('date joined'), default=timezone.now)

    objects = UserManager()

    USERNAME_FIELD = 'username'
    REQUIRED_FIELDS = ['email']

    class Meta:
        db_table = 'auth_user'
        verbose_name = _('user')
        verbose_name_plural = _('users')

    def get_full_name(self):
        """
        Returns the first_name plus the last_name, with a space in between.
        """
        full_name = '%s %s' % (self.first_name, self.last_name)
        return full_name.strip()

    def get_short_name(self):
        "Returns the short name for the user."
        return self.first_name

    def email_user(self, subject, message, from_email=None, **kwargs):
        """
        Sends an email to this User.
        """
        send_mail(subject, message, from_email, [self.email], **kwargs)
