"""
Viblr API Client - Core functionality for context search and prompt enhancement
"""

import httpx
import asyncio
from typing import Optional, List, Dict, Any, Union
from dataclasses import dataclass
try:
    from .exceptions import ViblrError, ViblrAPIError, ViblrAuthError, ViblrQuotaError
except ImportError:
    from exceptions import ViblrError, ViblrAPIError, ViblrAuthError, ViblrQuotaError


@dataclass
class ViblrResponse:
    """Response object from Viblr API"""
    query: str
    enhanced_prompt: str
    search_results: List[Dict[str, Any]]
    status: str
    textrank_enhanced: bool
    source_summaries: List[str]
    enhanced_topics: Dict[str, Any]
    key_phrases: List[str]
    query_related_phrases: List[str]
    intent_analysis: Dict[str, Any]
    metadata: Dict[str, Any]
    processing: Optional[Dict[str, Any]] = None
    filters_applied: Optional[Dict[str, Any]] = None
    
    @property
    def citations(self) -> List[str]:
        """Get list of citations from search results"""
        citations = []
        for result in self.search_results:
            if 'metadata' in result and 'provider' in result['metadata']:
                provider = result['metadata']['provider']
                external_id = result['metadata'].get('external_id', 'unknown')
                citations.append(f"{provider}:{external_id}")
        return citations


class ViblrClient:
    """Main client for interacting with Viblr API"""
    
    def __init__(
        self, 
        api_key: str, 
        base_url: str = "http://localhost:8000",
        timeout: int = 30
    ):
        self.api_key = api_key
        self.base_url = base_url.rstrip('/')
        self.timeout = timeout
        self._client = httpx.AsyncClient(timeout=timeout)
    
    async def search(
        self,
        query: str,
        max_results: int = 5,
        team_id: Optional[str] = None,
        sources: Optional[List[str]] = None,
        content_types: Optional[List[str]] = None,
        processing: Optional[Dict[str, Any]] = None,
        output_format: str = "enhanced_prompt"
    ) -> ViblrResponse:
        """
        Search for context and enhance prompts
        
        Args:
            query: The search query
            max_results: Maximum number of results to return (default: 5)
            sources: List of sources to filter by (e.g., ['gmail', 'github'])
            content_types: List of content types to filter by (e.g., ['email', 'code'])
            processing: Processing options (sentiment, keywords, summary, textrank)
            output_format: Output format - 'enhanced_prompt', 'raw_results', 'processed_results'
            
        Returns:
            ViblrResponse: Response object with enhanced_prompt, search_results, metadata
            
        Raises:
            ViblrAPIError: If the API request fails
            ViblrAuthError: If authentication fails
        """
        try:
            payload = {
                "query": query,
                "max_results": max_results,
                "output_format": output_format
            }
            
            # Add optional parameters
            if team_id:
                payload["team_id"] = team_id
            if sources:
                payload["sources"] = sources
            if content_types:
                payload["content_types"] = content_types
            if processing:
                payload["processing"] = processing
            
            headers = {
                "X-API-Key": self.api_key,
                "Content-Type": "application/json"
            }
            
            response = await self._client.post(
                f"{self.base_url}/v1/context/search",
                json=payload,
                headers=headers
            )
            
            if response.status_code == 401:
                raise ViblrAuthError("Invalid API key")
            elif response.status_code == 429:
                try:
                    error_data = response.json()
                    if isinstance(error_data, dict) and error_data.get("error") == "quota_exceeded":
                        message = error_data.get("message", "API quota exceeded")
                        retry_after = error_data.get("retry_after", 86400)
                        raise ViblrQuotaError(f"{message}. Retry after {retry_after} seconds.")
                    else:
                        raise ViblrAPIError("API quota exceeded. Please try again later.")
                except:
                    raise ViblrAPIError("API quota exceeded. Please try again later.")
            elif response.status_code != 200:
                error_detail = "Unknown error"
                try:
                    error_data = response.json()
                    error_detail = error_data.get('detail', str(error_data))
                except:
                    error_detail = response.text
                raise ViblrAPIError(f"API request failed: {response.status_code} - {error_detail}")
            
            result = response.json()
            
            return ViblrResponse(
                query=result.get("query", ""),
                enhanced_prompt=result.get("enhanced_prompt", ""),
                search_results=result.get("search_results", []),
                status=result.get("status", "unknown"),
                textrank_enhanced=result.get("textrank_enhanced", False),
                source_summaries=result.get("source_summaries", []),
                enhanced_topics=result.get("enhanced_topics", {}),
                key_phrases=result.get("key_phrases", []),
                query_related_phrases=result.get("query_related_phrases", []),
                intent_analysis=result.get("intent_analysis", {}),
                metadata=result.get("metadata", {}),
                processing=result.get("processing"),
                filters_applied=result.get("filters_applied")
            )
            
        except httpx.RequestError as e:
            raise ViblrAPIError(f"Network error: {str(e)}")
        except Exception as e:
            raise ViblrError(f"Unexpected error: {str(e)}")
    
    def search_sync(
        self,
        query: str,
        max_results: int = 5,
        team_id: Optional[str] = None,
        sources: Optional[List[str]] = None,
        content_types: Optional[List[str]] = None,
        processing: Optional[Dict[str, Any]] = None,
        output_format: str = "enhanced_prompt"
    ) -> ViblrResponse:
        """
        Synchronous version of search
        """
        try:
            loop = asyncio.get_event_loop()
            if loop.is_running():
                raise RuntimeError("Cannot use sync method from within an async context. Use search() instead.")
            return asyncio.run(self.search(query, max_results, team_id, sources, content_types, processing, output_format))
        except RuntimeError:
            return asyncio.run(self.search(query, max_results, team_id, sources, content_types, processing, output_format))
    
    async def health_check(self) -> Dict[str, Any]:
        """
        Check API health status
        
        Returns:
            Dict with health status information
        """
        try:
            response = await self._client.get(f"{self.base_url}/health")
            if response.status_code == 200:
                return response.json()
            else:
                raise ViblrAPIError(f"Health check failed: {response.status_code}")
        except httpx.RequestError as e:
            raise ViblrAPIError(f"Health check network error: {str(e)}")
    
    async def close(self):
        """Close the HTTP client"""
        await self._client.aclose()
    
    def __enter__(self):
        return self
    
    def __exit__(self, exc_type, exc_val, exc_tb):
        asyncio.run(self.close())
    
    # Convenience methods for common use cases
    
    async def enhance_prompt(
        self,
        prompt: str,
        max_results: int = 5,
        sources: Optional[List[str]] = None
    ) -> str:
        """
        Simple method to get enhanced prompt (backward compatibility)
        
        Args:
            prompt: The original prompt
            max_results: Maximum number of results
            sources: Sources to search
            
        Returns:
            Enhanced prompt string
        """
        response = await self.search(
            query=prompt,
            max_results=max_results,
            sources=sources,
            output_format="enhanced_prompt"
        )
        return response.enhanced_prompt
    
    async def get_context(
        self,
        query: str,
        max_results: int = 5,
        sources: Optional[List[str]] = None
    ) -> List[Dict[str, Any]]:
        """
        Get raw context without enhancement
        
        Args:
            query: The search query
            max_results: Maximum number of results
            sources: Sources to search
            
        Returns:
            List of raw search results
        """
        response = await self.search(
            query=query,
            max_results=max_results,
            sources=sources,
            output_format="raw_results"
        )
        return response.search_results

    async def get_tool_schemas(
        self,
        format: str = "openai",
        sources: Optional[str] = None
    ) -> List[Dict[str, Any]]:
        """
        Get tool schemas for AI function calling.

        Args:
            format: Tool format - "openai", "anthropic", "generic"
            sources: Comma-separated list of sources to include

        Returns:
            List of tool schemas in requested format

        Raises:
            ViblrAPIError: If the API request fails
            ViblrAuthError: If authentication fails
        """
        try:
            params = {"format": format}
            if sources:
                params["sources"] = sources

            headers = {
                "X-API-Key": self.api_key,
                "Content-Type": "application/json"
            }

            response = await self._client.get(
                f"{self.base_url}/v1/tools/schemas",
                params=params,
                headers=headers
            )

            if response.status_code == 401:
                raise ViblrAuthError("Invalid API key")
            elif response.status_code == 429:
                raise ViblrQuotaError("API quota exceeded. Please try again later.")
            elif response.status_code != 200:
                error_detail = "Unknown error"
                try:
                    error_data = response.json()
                    error_detail = error_data.get('detail', str(error_data))
                except:
                    error_detail = response.text
                raise ViblrAPIError(f"API request failed: {response.status_code} - {error_detail}")

            result = response.json()
            return result.get("tools", [])

        except httpx.RequestError as e:
            raise ViblrAPIError(f"Network error: {str(e)}")
        except Exception as e:
            raise ViblrError(f"Unexpected error: {str(e)}")

    async def get_context(
        self,
        query: str,
        sources: Optional[List[str]] = None,
        content_types: Optional[List[str]] = None,
        max_results: int = 5,
        time_filter: Optional[str] = None
    ) -> Dict[str, Any]:
        """
        AI tool method for context retrieval during reasoning.

        This is optimized for AI function calling and returns structured data
        for AI models to process.

        Args:
            query: Search query
            sources: List of sources to search
            content_types: List of content types
            max_results: Maximum number of results
            time_filter: Time filter ("today", "week", "month", "quarter")

        Returns:
            Dict with context items optimized for AI consumption

        Raises:
            ViblrAPIError: If the API request fails
            ViblrAuthError: If authentication fails
        """
        try:
            payload = {
                "query": query,
                "max_results": max_results
            }

            # Add optional parameters
            if sources:
                payload["sources"] = sources
            if content_types:
                payload["content_types"] = content_types
            if time_filter:
                payload["time_filter"] = time_filter

            headers = {
                "X-API-Key": self.api_key,
                "Content-Type": "application/json"
            }

            response = await self._client.post(
                f"{self.base_url}/v1/tools/get_context",
                json=payload,
                headers=headers
            )

            if response.status_code == 401:
                raise ViblrAuthError("Invalid API key")
            elif response.status_code == 429:
                raise ViblrQuotaError("API quota exceeded. Please try again later.")
            elif response.status_code != 200:
                error_detail = "Unknown error"
                try:
                    error_data = response.json()
                    error_detail = error_data.get('detail', str(error_data))
                except:
                    error_detail = response.text
                raise ViblrAPIError(f"API request failed: {response.status_code} - {error_detail}")

            return response.json()

        except httpx.RequestError as e:
            raise ViblrAPIError(f"Network error: {str(e)}")
        except Exception as e:
            raise ViblrError(f"Unexpected error: {str(e)}")

    def get_tool_schemas_sync(
        self,
        format: str = "openai",
        sources: Optional[str] = None
    ) -> List[Dict[str, Any]]:
        """
        Synchronous version of get_tool_schemas
        """
        try:
            loop = asyncio.get_event_loop()
            if loop.is_running():
                raise RuntimeError("Cannot use sync method from within an async context. Use get_tool_schemas() instead.")
            return asyncio.run(self.get_tool_schemas(format, sources))
        except RuntimeError:
            return asyncio.run(self.get_tool_schemas(format, sources))

    def get_context_sync(
        self,
        query: str,
        sources: Optional[List[str]] = None,
        content_types: Optional[List[str]] = None,
        max_results: int = 5,
        time_filter: Optional[str] = None
    ) -> Dict[str, Any]:
        """
        Synchronous version of get_context
        """
        try:
            loop = asyncio.get_event_loop()
            if loop.is_running():
                raise RuntimeError("Cannot use sync method from within an async context. Use get_context() instead.")
            return asyncio.run(self.get_context(query, sources, content_types, max_results, time_filter))
        except RuntimeError:
            return asyncio.run(self.get_context(query, sources, content_types, max_results, time_filter))