"""
FastAPI integration for Viblr SDK
"""

try:
    from fastapi import FastAPI, Depends, HTTPException, Request
except ImportError:
    # FastAPI not available, create dummy classes
    class FastAPI:
        pass
    class Depends:
        pass
    class HTTPException:
        pass
    class Request:
        pass
from fastapi.responses import JSONResponse
from typing import Optional, List, Dict, Any
try:
    from .client import ViblrClient
    from .exceptions import ViblrError, ViblrAPIError, ViblrAuthError
except ImportError:
    from client import ViblrClient
    from exceptions import ViblrError, ViblrAPIError, ViblrAuthError


class ViblrMiddleware:
    """FastAPI middleware for automatic prompt enhancement"""
    
    def __init__(self, api_key: str, base_url: str = "http://localhost:8080"):
        self.client = ViblrClient(api_key=api_key, base_url=base_url)
    
    async def __call__(self, request: Request, call_next):
        # Only enhance prompts for specific endpoints
        if request.url.path.startswith("/ai/") and request.method == "POST":
            try:
                body = await request.body()
                # Here you would parse and enhance the prompt
                # This is a simplified example
                pass
            except Exception as e:
                # Log error but don't break the request
                print(f"Viblr enhancement failed: {e}")
        
        response = await call_next(request)
        return response


def create_viblr_dependency(api_key: str, base_url: str = "http://localhost:8080"):
    """Create a FastAPI dependency for Viblr client"""
    
    client = ViblrClient(api_key=api_key, base_url=base_url)
    
    async def get_viblr_client():
        return client
    
    return Depends(get_viblr_client)


def enhance_prompt_dependency(
    prompt: str,
    user_id: str,
    context_sources: Optional[List[str]] = None,
    metadata: Optional[Dict[str, Any]] = None
):
    """
    FastAPI dependency that automatically enhances prompts
    
    Usage:
        @app.post("/chat")
        async def chat(
            request: ChatRequest,
            enhanced_prompt: str = Depends(enhance_prompt_dependency)
        ):
            # enhanced_prompt is already enhanced with context
            pass
    """
    async def _enhance_prompt(client: ViblrClient = Depends(create_viblr_dependency("your_api_key"))):
        try:
            return await client.enhance_prompt(
                prompt=prompt,
                user_id=user_id,
                context_sources=context_sources,
                metadata=metadata
            )
        except ViblrError as e:
            raise HTTPException(status_code=500, detail=str(e))
    
    return _enhance_prompt


# Example FastAPI app integration
def create_example_app(api_key: str):
    """Example FastAPI app showing Viblr integration"""
    
    app = FastAPI(title="AI Chatbot with Viblr Enhancement")
    
    # Add Viblr middleware
    app.add_middleware(ViblrMiddleware, api_key=api_key)
    
    # Create Viblr dependency
    viblr_client = create_viblr_dependency(api_key)
    
    @app.post("/chat")
    async def chat_endpoint(
        request: dict,
        client: ViblrClient = viblr_client
    ):
        """Chat endpoint with automatic prompt enhancement"""
        try:
            # Extract prompt and user info
            prompt = request.get("message", "")
            user_id = request.get("user_id", "anonymous")
            
            # Enhance the prompt with Viblr
            enhanced_prompt = await client.enhance_prompt(
                prompt=prompt,
                user_id=user_id,
                context_sources=["github", "slack", "jira"]
            )
            
            # Here you would send enhanced_prompt to your AI model
            # For demo purposes, we'll just return the enhanced prompt
            return {
                "original_prompt": prompt,
                "enhanced_prompt": enhanced_prompt,
                "message": "Prompt enhanced with Viblr context!"
            }
            
        except ViblrError as e:
            raise HTTPException(status_code=500, detail=str(e))
    
    @app.get("/health")
    async def health_check():
        """Health check endpoint"""
        return {"status": "healthy", "service": "AI Chatbot with Viblr"}
    
    return app
