import os
from dataservice.core.ddsapi import DataServiceApi, DataServiceError, DataServiceAuth
from dataservice.core.util import KindType
from dataservice.core.localstore import HashUtil

FETCH_ALL_USERS_PAGE_SIZE = 25
DOWNLOAD_FILE_CHUNK_SIZE = 20 * 1024 * 1024


class RemoteStore(object):
    """
    Fetches project tree data from remote store.
    """
    def __init__(self, config):
        """
        Setup to allow fetching project tree.
        :param config: ddsc.config.Config settings to use for connecting to the dataservice.
        """
        self.config = config
        auth = DataServiceAuth(self.config)
        self.data_service = DataServiceApi(auth, self.config.url)

    def fetch_remote_project(self, project_name, must_exist=False, include_children=True):
        """
        Retrieve the project via project_name.
        :param project_name: str name of the project to try and download
        :param must_exist: should we error if the project doesn't exist
        :param include_children: should we read children(folders/files)
        :return: RemoteProject project requested or None if not found(and must_exist=False)
        """
        project = self._get_my_project(project_name)
        if project:
            if include_children:
                self._add_project_children(project)
        else:
            if must_exist:
                raise ValueError(u'There is no project with the name {}'.format(project_name).encode('utf-8'))
        return project

    def fetch_remote_project_by_id(self, id):
        """
        Retrieves project from via id
        :param id: str id of project from data service
        :return: RemoteProject we downloaded
        """
        response = self.data_service.get_project_by_id(id).json()
        return RemoteProject(response)

    def _get_my_project(self, project_name):
        """
        Return project tree root for project_name.
        :param project_name: str name of the project to download
        :return: RemoteProject project we found or None
        """
        response = self.data_service.get_projects().json()
        for project in response['results']:
            if project['name'] == project_name:
                return RemoteProject(project)
        return None

    def _add_project_children(self, project):
        """
        Add the rest of the project tree from the remote store to the project object.
        :param project: RemoteProject root of the project tree to add children too
        """
        response = self.data_service.get_project_children(project.id, '').json()
        project_children = RemoteProjectChildren(project.id, response['results'])
        for child in project_children.get_tree():
            project.add_child(child)

    def lookup_user_by_email_or_username(self, email, username):
        if username:
            return self.lookup_user_by_username(username)
        else:
            return self.lookup_user_by_email(email)

    def lookup_user_by_name(self, full_name):
        """
        Query remote store for a single user with the name full_name or raise error.
        :param full_name: str Users full name separated by a space.
        :return: RemoteUser user info for single user with full_name
        """
        res = self.data_service.get_users_by_full_name(full_name)
        json_data = res.json()
        results = json_data['results']
        found_cnt = len(results)
        if found_cnt == 0:
            raise ValueError("User not found:" + full_name)
        elif found_cnt > 1:
            raise ValueError("Multiple users with name:" + full_name)
        user = RemoteUser(results[0])
        if user.full_name.lower() != full_name.lower():
            raise ValueError("User not found:" + full_name)
        return user

    def lookup_user_by_username(self, username):
        """
        Finds the single user who has this username or raises ValueError.
        :param username: str username we are looking for
        :return: RemoteUser user we found
        """
        matches = [user for user in self.fetch_all_users() if user.username == username]
        if not matches:
            raise ValueError('Username not found: {}.'.format(username))
        if len(matches) > 1:
            raise ValueError('Multiple users with same username found: {}.'.format(username))
        return matches[0]

    def lookup_user_by_email(self, email):
        """
        Finds the single user who has this email or raises ValueError.
        :param email: str email we are looking for
        :return: RemoteUser user we found
        """
        matches = [user for user in self.fetch_all_users() if user.email == email]
        if not matches:
            raise ValueError('Email not found: {}.'.format(email))
        if len(matches) > 1:
            raise ValueError('Multiple users with same email found: {}.'.format(email))
        return matches[0]

    def get_current_user(self):
        """
        Fetch info about the current user
        :return: RemoteUser user who we are logged in as(auth determines this).
        """
        response = self.data_service.get_current_user().json()
        return RemoteUser(response)

    def fetch_all_users(self):
        """
        Retrieves all users from data service.
        :return: [RemoteUser] list of all users we downloaded
        """
        page = 1
        per_page = FETCH_ALL_USERS_PAGE_SIZE
        users = []
        while True:
            result = self.data_service.get_users_by_page_and_offset(page, per_page)
            user_list_json = result.json()
            for user_json in user_list_json['results']:
                users.append(RemoteUser(user_json))
            total_pages = int(result.headers["x-total-pages"])
            result_page = int(result.headers["x-page"])
            if result_page == total_pages:
                break;
            page += 1
        return users

    def fetch_user(self, id):
        """
        Retrieves user from data service having a specific id
        :param id: str id of user from data service
        :return: RemoteUser user we downloaded
        """
        response = self.data_service.get_user_by_id(id).json()
        return RemoteUser(response)

    def set_user_project_permission(self, project, user, auth_role):
        """
        Update remote store for user giving auth_role permissions on project.
        :param project: RemoteProject project to give permissions to
        :param user: RemoteUser user who we are giving permissions to
        :param auth_role: str type of authorization to give user(project_admin)
        """
        self.data_service.set_user_project_permission(project.id, user.id, auth_role)

    def revoke_user_project_permission(self, project, user):
        """
        Update remote store for user removing auth_role permissions on project.
        :param project: RemoteProject project to remove permissions from
        :param user: RemoteUser user who we are removing permissions from
        """
        # Server errors out with 500 if a user isn't found.
        try:
            resp = self.data_service.get_user_project_permission(project.id, user.id)
            self.data_service.revoke_user_project_permission(project.id, user.id)
        except DataServiceError as e:
            if e.status_code != 404:
                raise

    def download_file(self, remote_file, path, watcher):
        """
        Download a remote file associated with the remote uuid(file_id) into local path.
        :param remote_file: RemoteFile file to retrieve
        :param path: str file system path to save the contents to.
        :param watcher: object implementing send_item(item, increment_amt) that updates UI
        """
        url_json = self.data_service.get_file_url(remote_file.id).json()
        http_verb = url_json['http_verb']
        host = url_json['host']
        url = url_json['url']
        http_headers = url_json['http_headers']
        response = self.data_service.receive_external(http_verb, host, url, http_headers)
        with open(path, 'wb') as f:
            for chunk in response.iter_content(chunk_size=DOWNLOAD_FILE_CHUNK_SIZE):
                if chunk:  # filter out keep-alive new chunks
                    f.write(chunk)
                    watcher.transferring_item(remote_file, increment_amt=len(chunk))

    def get_project_names(self):
        """
        Return a list of names of the remote projects owned by this user.
        :return: [str]: the list of project names
        """
        names = []
        response = self.data_service.get_projects().json()
        for project in response['results']:
            names.append(project['name'])
        return names

    def delete_project_by_name(self, project_name):
        """
        Find the project named project_name and delete it raise error if not found.
        :param project_name: str: Name of the project we want to be deleted
        """
        project = self._get_my_project(project_name)
        if project:
            self.data_service.delete_project(project.id)
        else:
            raise ValueError("No project named '{}' found.\n".format(project_name))

    def get_active_auth_roles(self, context):
        """
        Retrieve non-deprecated authorization roles based on a context.
        Context should be RemoteAuthRole.PROJECT_CONTEXT or RemoteAuthRole.SYSTEM_CONTEXT.
        :param context: str: context for which auth roles to retrieve
        :return: [RemoteAuthRole]: list of active auth_role objects
        """
        response = self.data_service.get_auth_roles(context).json()
        return self.get_active_auth_roles_from_json(response)

    @staticmethod
    def get_active_auth_roles_from_json(json_data):
        """
        Given a json blob response containing a list of authorization roles return the active ones
        in an array of RemoteAuthRole objects.
        :param json_data: list of dictionaries - data from dds in auth_role format
        :return: [RemoteAuthRole] list of active auth_role objects
        """
        result = []
        for auth_role_properties in json_data['results']:
            auth_role = RemoteAuthRole(auth_role_properties)
            if not auth_role.is_deprecated:
                result.append(auth_role)
        return result


class RemoteProject(object):
    """
    Project data from a remote store projects request.
    Represents the top of a tree.
    Has kind property to allow project tree traversal with ProjectWalker.
    """
    def __init__(self, json_data):
        """
        Set properties based on json_data.
        :param json_data: dict JSON data containing project info
        """
        self.id = json_data['id']
        self.kind = json_data['kind']
        self.name = json_data['name']
        self.description = json_data['description']
        self.is_deleted = json_data['is_deleted']
        self.children = []
        self.remote_path = ''

    def add_child(self, child):
        """
        Add a file or folder to our remote project.
        :param child: RemoteFolder/RemoteFile child to add.
        """
        self.children.append(child)

    def __str__(self):
        return 'project: {} id:{} {}'.format(self.name, self.id, self.children)


class RemoteFolder(object):
    """
    Folder data from a remote store project_id_children or folder_id_children request.
    Represents a leaf or branch in a project tree.
    Has kind property to allow project tree traversal with ProjectWalker.
    """
    def __init__(self, json_data, parent_remote_path):
        """
        Set properties based on json_data.
        :param json_data: dict JSON data containing folder info
        :param parent_remote_path: remote_path path to this folder's parent
        """
        self.id = json_data['id']
        self.kind = json_data['kind']
        self.name = json_data['name']
        self.is_deleted = json_data['is_deleted']
        self.children = []
        self.remote_path = os.path.join(parent_remote_path, self.name)

    def add_child(self, child):
        """
        Add remote file or folder to this folder.
        :param child: RemoteFolder or remoteFile to add.
        """
        self.children.append(child)

    def __str__(self):
        return 'folder: {} id:{} {}'.format(self.name, self.id, self.children)


class RemoteFile(object):
    """
    File data from a remote store project_id_children or folder_id_children request.
    Represents a leaf in a project tree.
    Has kind property to allow project tree traversal with ProjectWalker.
    """
    def __init__(self, json_data, parent_remote_path):
        """
        Set properties based on json_data.
        :param json_data: dict JSON data containing file info
        :param parent_remote_path: remote_path path to this file's parent
        """
        self.id = json_data['id']
        self.kind = json_data['kind']
        self.name = json_data['name']
        self.path = self.name # for compatibilty with ProgressPrinter
        self.is_deleted = json_data['is_deleted']
        upload = RemoteFile.get_upload_from_json(json_data)
        self.size = upload['size']
        self.file_hash = None
        self.hash_alg = None
        hash_data = RemoteFile.get_hash_from_upload(upload)
        if hash_data:
            self.file_hash = hash_data.get('value')
            self.hash_alg = hash_data.get('algorithm')
        self.remote_path = os.path.join(parent_remote_path, self.name)

    def set_hash(self, file_hash, hash_alg):
        """
        Set the hash value and algorithm for the contents of the file.
        :param file_hash: str hash value
        :param hash_alg: str name of the hash algorithm(md5)
        """
        self.file_hash = file_hash
        self.hash_alg = hash_alg

    @staticmethod
    def get_upload_from_json(json_data):
        if 'current_version' in json_data:
            return json_data['current_version']['upload']
        else:
            if 'upload' in json_data:
                return json_data['upload']
            else:
                raise ValueError("Invalid file json data, unable to find upload.")

    @staticmethod
    def get_hash_from_upload(upload, target_algorithm=HashUtil.HASH_NAME):
        """
        Find hash value in upload dictionary.
        Older upload format stores a single hash in 'hash' property.
        New upload format stores multiple under 'hashes' property for this one we look for a particular algorithm.
        :param upload: dictionary: contains hash data in DukeDS upload format.
        :param target_algorithm: str: name of the algorithm to look for if there are more than one hash
        :return: dictionary of hash information, keys: "algorithm" and  "value"
        """
        hash_info = upload.get('hash')
        if hash_info:
            return hash_info
        hashes_array = upload.get('hashes')
        if hashes_array:
            for hash_info in hashes_array:
                algorithm = hash_info.get('algorithm')
                if algorithm == target_algorithm:
                    return hash_info
        return None

    def __str__(self):
        return 'file: {} id:{} size:{}'.format(self.name, self.id, self.size)


class RemoteUser(object):
    """
    User who can download/upload/edit project on remote store.
    """
    def __init__(self, json_data):
        """
        Set properties based on json_data.
        :param json_data: dict JSON data containing file info
        """
        self.id = json_data['id']
        self.username = json_data['username']
        self.full_name = json_data['full_name']
        self.email = json_data['email']

    def __str__(self):
        return 'id:{} username:{} full_name:{}'.format(self.id, self.username, self.full_name)


class RemoteAuthRole(object):
    PROJECT_CONTEXT = "project"
    SYSTEM_CONTEXT = "system"
    """
    Permissions a user can be given on a project.
    """
    def __init__(self, json_data):
        """
        Set properties based on json_data.
        :param json_data: dict JSON data containing auth_role info
        """
        self.id = json_data['id']
        self.name = json_data['name']
        self.description = json_data['description']
        self.is_deprecated = json_data['is_deprecated']

    def __str__(self):
        return 'id:{} name:{} description:{}'.format(self.id, self.name, self.description)


class RemoteProjectChildren(object):
    """
    Creates RemoteFolders and RemoteFiles as tree structure based on DukeDS recursive project children data.
    """
    def __init__(self, project_id, data):
        """
        Specify the project_id and the array of item dictionaries.
        :param project_id: str: uuid of the project
        :param data: [object]: DukeDS recursive project children
        """
        self.project_id = project_id
        self.data = data

    def _get_children_for_parent(self, parent_id):
        """
        Given a parent uuid return a list of dictionaries.
        :param parent_id: str: uuid of the parent
        :return: [dict]: children in this list with parent_id parent
        """
        children = []
        for child in self.data:
            parent = child['parent']
            if parent['id'] == parent_id:
                children.append(child)
        return children

    def get_tree(self):
        """
        Return array of RemoteFolders(with appropriate children)/RemoteFiles based on the values from constructor.
        :return: [RemoteFolder/RemoteFile]
        """
        return self.get_tree_recur(self.project_id, '')

    def get_tree_recur(self, parent_id, parent_path):
        """
        Recursively create array RemoteFolders/RemoteFiles.
        :param parent_id: str: uuid if the parent to find children for
        :param parent_path: str: remote path of parent to build child paths
        :return: [RemoteFolder/RemoteFile]
        """
        children = []
        for child_data in self._get_children_for_parent(parent_id):
            if child_data['kind'] == KindType.folder_str:
                folder = RemoteFolder(child_data, parent_path)
                for grand_child in self.get_tree_recur(child_data['id'], folder.remote_path):
                    folder.add_child(grand_child)
                children.append(folder)
            else:
                file = RemoteFile(child_data, parent_path)
                children.append(file)
        return children
