"""ryo3-std types"""

import datetime as pydt
import ipaddress
import pathlib
import typing as t

from ry._types import (
    Buffer,
    DurationDict,
    FileTypeDict,
    FsPathLike,
    MetadataDict,
)
from ry.protocols import FromStr, RyIterator, ToPy, ToPyTimeDelta, ToString
from ry.ryo3._bytes import Bytes

# =============================================================================
# STD::TIME
# =============================================================================
@t.final
class Duration(ToPy[pydt.timedelta], ToPyTimeDelta, ToString):
    ZERO: t.ClassVar[Duration]
    MIN: t.ClassVar[Duration]
    MAX: t.ClassVar[Duration]
    NANOSECOND: t.ClassVar[Duration]
    MICROSECOND: t.ClassVar[Duration]
    MILLISECOND: t.ClassVar[Duration]
    SECOND: t.ClassVar[Duration]

    def __init__(self, secs: int = 0, nanos: int = 0) -> None: ...
    def __eq__(self, other: object) -> bool: ...
    def __ne__(self, other: object) -> bool: ...
    def __add__(self, other: t.Self | pydt.timedelta) -> t.Self: ...
    def __sub__(self, other: t.Self | pydt.timedelta) -> t.Self: ...
    def __radd__(self, other: t.Self | pydt.timedelta) -> t.Self: ...
    def __rsub__(self, other: t.Self | pydt.timedelta) -> t.Self: ...
    def __lt__(self, other: t.Self) -> bool: ...
    def __le__(self, other: t.Self) -> bool: ...
    def __gt__(self, other: t.Self) -> bool: ...
    def __ge__(self, other: t.Self) -> bool: ...
    def __hash__(self) -> int: ...
    def __richcmp__(self, other: t.Self | pydt.timedelta, op: int) -> bool: ...
    def __bool__(self) -> bool: ...
    def __float__(self) -> float: ...
    def __int__(self) -> int: ...
    @t.overload
    def __truediv__(self, other: t.Self | pydt.timedelta) -> float: ...
    @t.overload
    def __truediv__(self, other: float) -> t.Self: ...
    @t.overload
    def __rtruediv__(self, other: t.Self | pydt.timedelta) -> float: ...
    @t.overload
    def __rtruediv__(self, other: float) -> t.Self: ...
    def __mul__(self, other: float) -> t.Self: ...
    def __rmul__(self, other: float) -> t.Self: ...
    def abs_diff(self, other: t.Self | pydt.timedelta) -> t.Self: ...
    def sleep(self, interval: int = 10) -> None: ...

    # =========================================================================
    # PYTHON_CONVERSIONS
    # =========================================================================
    @classmethod
    def from_pytimedelta(cls, td: pydt.timedelta) -> t.Self: ...
    def to_pytimedelta(self) -> pydt.timedelta: ...
    def to_py(self) -> pydt.timedelta: ...
    def to_dict(self) -> DurationDict: ...
    @classmethod
    def from_dict(cls, d: DurationDict) -> t.Self: ...

    # =========================================================================
    # TO/FROM STRING(s)
    # =========================================================================
    def to_string(self) -> str: ...
    def isoformat(self) -> str: ...
    @classmethod
    def fromisoformat(cls, s: str) -> t.Self: ...
    def friendly(
        self,
        designator: t.Literal[
            "compact", "human", "human-time", "short", "verbose"
        ] = "compact",
    ) -> str: ...
    @classmethod
    def from_str(cls, s: str) -> t.Self: ...
    # =========================================================================
    # PROPERTIES
    # =========================================================================
    @property
    def is_zero(self) -> bool: ...
    @property
    def nanos(self) -> int: ...
    @property
    def nanoseconds(self) -> int:
        """Alias for .nanos"""
    @property
    def ns(self) -> int:
        """Alias for .nanos"""
    @property
    def secs(self) -> int: ...
    @property
    def days(self) -> int: ...
    @property
    def seconds(self) -> int: ...
    @property
    def seconds_remainder(self) -> int: ...
    @property
    def microseconds(self) -> int: ...
    @property
    def subsec_micros(self) -> int: ...
    @property
    def subsec_millis(self) -> int: ...
    @property
    def subsec_nanos(self) -> int: ...

    # =========================================================================
    # CLASSMETHODS
    # =========================================================================
    @classmethod
    def from_hours(cls, hours: int) -> t.Self: ...
    @classmethod
    def from_micros(cls, micros: int) -> t.Self: ...
    @classmethod
    def from_millis(cls, millis: int) -> t.Self: ...
    @classmethod
    def from_mins(cls, mins: int) -> t.Self: ...
    @classmethod
    def from_nanos(cls, nanos: int) -> t.Self: ...
    @classmethod
    def from_secs(cls, secs: int) -> t.Self: ...
    @classmethod
    def from_secs_f32(cls, secs: float) -> t.Self: ...
    @classmethod
    def from_secs_f64(cls, secs: float) -> t.Self: ...
    @classmethod
    def from_days(cls, days: int) -> t.Self: ...
    @classmethod
    def from_weeks(cls, weeks: int) -> t.Self: ...
    def as_micros(self) -> int: ...
    def as_millis(self) -> int: ...
    def as_nanos(self) -> int: ...
    def as_secs(self) -> int: ...
    def as_secs_f32(self) -> float: ...
    def as_secs_f64(self) -> float: ...

    # =========================================================================
    # ARITHMETIC
    # =========================================================================
    def checked_add(self, other: t.Self) -> t.Self | None: ...
    def checked_div(self, other: int) -> t.Self | None: ...
    def checked_mul(self, other: float) -> t.Self | None: ...
    def checked_sub(self, other: t.Self) -> t.Self | None: ...
    def div_duration_f32(self, other: t.Self) -> float: ...
    def div_duration_f64(self, other: t.Self) -> float: ...
    def div_f32(self, other: float) -> t.Self: ...
    def div_f64(self, other: float) -> t.Self: ...
    def mul_f32(self, other: float) -> t.Self: ...
    def mul_f64(self, other: float) -> t.Self: ...
    def saturating_add(self, other: t.Self) -> t.Self: ...
    def saturating_mul(self, other: int) -> t.Self: ...
    def saturating_sub(self, other: t.Self) -> t.Self: ...

@t.final
class Instant:
    def __init__(self) -> None: ...
    @classmethod
    def now(cls) -> Instant: ...
    def __eq__(self, other: object) -> bool: ...
    def __ne__(self, other: object) -> bool: ...
    def __lt__(self, other: Instant) -> bool: ...
    def __le__(self, other: Instant) -> bool: ...
    def __gt__(self, other: Instant) -> bool: ...
    def __ge__(self, other: Instant) -> bool: ...
    def __hash__(self) -> int: ...
    def __add__(self, other: Duration) -> t.Self: ...
    @t.overload
    def __sub__(self, other: Duration) -> t.Self: ...
    @t.overload
    def __sub__(self, other: t.Self) -> Duration: ...

    # =========================================================================
    # INSTANCE METHODS
    # =========================================================================
    def checked_add(self, other: Duration) -> t.Self | None: ...
    def checked_duration_since(self, earlier: t.Self) -> Duration | None: ...
    def checked_sub(self, other: Duration) -> t.Self | None: ...
    def duration_since(self, earlier: t.Self) -> Duration: ...
    def elapsed(self) -> Duration: ...
    def saturating_duration_since(self, earlier: t.Self) -> Duration: ...

def instant() -> Instant: ...
def sleep(seconds: float) -> float: ...

# =============================================================================
# STD::FS
# =============================================================================
@t.final
class FileType(ToPy[t.Literal["file", "dir", "symlink"]]):
    def __init__(
        self, t: t.Literal["f", "file", "d", "dir", "directory", "l", "symlink", "link"]
    ) -> None: ...
    @property
    def is_dir(self) -> bool: ...
    @property
    def is_file(self) -> bool: ...
    @property
    def is_symlink(self) -> bool: ...
    def to_dict(self) -> FileTypeDict: ...
    def to_py(self) -> t.Literal["file", "dir", "symlink"]: ...

@t.final
class Permissions:
    @property
    def readonly(self) -> bool: ...
    def __eq__(self, value: object) -> bool: ...
    def __ne__(self, value: object) -> bool: ...

@t.final
class Metadata:
    def __init__(self) -> t.NoReturn: ...
    @property
    def file_type(self) -> FileType: ...
    @property
    def len(self) -> int: ...
    @property
    def is_empty(self) -> bool: ...
    @property
    def modified(self) -> pydt.datetime: ...
    @property
    def accessed(self) -> pydt.datetime: ...
    @property
    def created(self) -> pydt.datetime: ...
    @property
    def is_dir(self) -> bool: ...
    @property
    def is_file(self) -> bool: ...
    @property
    def is_symlink(self) -> bool: ...
    @property
    def permissions(self) -> Permissions: ...
    @property
    def readonly(self) -> bool: ...
    def to_py(self) -> MetadataDict: ...

@t.final
class DirEntry:
    def __init__(self) -> t.NoReturn: ...
    def __fspath__(self) -> str: ...
    @property
    def path(self) -> pathlib.Path: ...
    @property
    def basename(self) -> str: ...
    @property
    def metadata(self) -> Metadata: ...
    @property
    def file_type(self) -> FileType: ...

@t.final
class ReadDir(RyIterator[DirEntry]):
    def __init__(self) -> t.NoReturn: ...
    def __iter__(self) -> t.Self: ...
    def __next__(self) -> DirEntry: ...
    def collect(self) -> list[DirEntry]: ...
    def take(self, n: int = 1) -> list[DirEntry]: ...

@t.final
class FileReadStream(RyIterator[Bytes]):
    def __init__(
        self,
        path: FsPathLike,
        *,
        chunk_size: int = 65536,
        offset: int = 0,
        buffered: bool = True,
    ) -> None: ...
    def __iter__(self) -> t.Self: ...
    def __next__(self) -> Bytes: ...
    def collect(self) -> list[Bytes]: ...
    def take(self, n: int = 1) -> list[Bytes]: ...

# ============================================================================
# STD::FS ~ functions
# =============================================================================
def canonicalize(path: FsPathLike) -> pathlib.Path: ...
def copy(from_path: FsPathLike, to_path: FsPathLike) -> int: ...
def create_dir(path: FsPathLike) -> None: ...
def create_dir_all(path: FsPathLike) -> None: ...
def exists(path: FsPathLike) -> bool: ...
def hard_link(from_path: FsPathLike, to_path: FsPathLike) -> None: ...
def is_dir(path: FsPathLike) -> bool: ...
def is_file(path: FsPathLike) -> bool: ...
def is_symlink(path: FsPathLike) -> bool: ...
def metadata(path: FsPathLike) -> Metadata: ...
def read(path: FsPathLike) -> Bytes: ...
def read_bytes(path: FsPathLike) -> bytes: ...
def read_dir(path: FsPathLike) -> ReadDir: ...
def read_link(path: FsPathLike) -> pathlib.Path: ...
def read_stream(
    path: FsPathLike, chunk_size: int = 65536, *, offset: int = 0
) -> FileReadStream: ...
def read_text(path: FsPathLike) -> str: ...
def read_str(path: FsPathLike) -> str: ...
def read_to_string(path: FsPathLike) -> str: ...
def remove_dir(path: FsPathLike) -> None: ...
def remove_dir_all(path: FsPathLike) -> None: ...
def remove_file(path: FsPathLike) -> None: ...
def rename(from_path: FsPathLike, to_path: FsPathLike) -> None: ...
def set_permissions(path: FsPathLike, permissions: Permissions) -> None: ...
def soft_link(from_path: FsPathLike, to_path: FsPathLike) -> None: ...
def symlink_metadata(path: FsPathLike) -> Metadata: ...
def write(path: FsPathLike, data: Buffer | str) -> int: ...
def write_bytes(path: FsPathLike, data: bytes) -> int: ...
def write_text(path: FsPathLike, data: str) -> int: ...

# =============================================================================
# STD::NET
# =============================================================================

class _Version4(t.Protocol):
    @property
    def version(self) -> t.Literal[4]: ...

class _Version6(t.Protocol):
    @property
    def version(self) -> t.Literal[6]: ...

class _Version(t.Protocol):
    @property
    def version(self) -> t.Literal[4, 6]: ...

class _Ipv4AddrProperties(t.Protocol):
    @property
    def is_benchmarking(self) -> bool: ...
    @property
    def is_broadcast(self) -> bool: ...
    @property
    def is_documentation(self) -> bool: ...
    @property
    def is_global(self) -> t.NoReturn: ...
    @property
    def is_link_local(self) -> bool: ...
    @property
    def is_loopback(self) -> bool: ...
    @property
    def is_multicast(self) -> bool: ...
    @property
    def is_private(self) -> bool: ...
    @property
    def is_reserved(self) -> bool: ...
    @property
    def is_shared(self) -> bool: ...
    @property
    def is_unspecified(self) -> bool: ...
    @property
    def is_unicast(self) -> bool: ...

_T_ipaddress_co = t.TypeVar(
    "_T_ipaddress_co",
    bound=ipaddress.IPv4Address | ipaddress.IPv6Address,
    covariant=True,
)

class ToPyIpAddress(t.Protocol[_T_ipaddress_co]):
    def to_pyipaddress(self) -> _T_ipaddress_co: ...

@t.final
class Ipv4Addr(
    _Ipv4AddrProperties,
    _Version4,
    FromStr,
    ToPy[ipaddress.IPv4Address],
    ToPyIpAddress[ipaddress.IPv4Address],
):
    BROADCAST: Ipv4Addr
    LOCALHOST: Ipv4Addr
    UNSPECIFIED: Ipv4Addr

    @t.overload
    def __init__(self, a: int, b: int, c: int, d: int) -> None: ...
    @t.overload
    def __init__(
        self, iplike: int | str | bytes | Ipv4Addr | ipaddress.IPv4Address
    ) -> None: ...
    def __eq__(self, other: object) -> bool: ...
    def __ne__(self, other: object) -> bool: ...
    def __lt__(self, other: Ipv4Addr) -> bool: ...
    def __le__(self, other: Ipv4Addr) -> bool: ...
    def __gt__(self, other: Ipv4Addr) -> bool: ...
    def __ge__(self, other: Ipv4Addr) -> bool: ...
    def __hash__(self) -> int: ...
    def to_py(self) -> ipaddress.IPv4Address: ...
    @property
    def version(self) -> t.Literal[4]: ...
    @property
    def is_documentation(self) -> bool: ...

    # ========================================================================
    # CLASSMETHODS
    # ========================================================================
    @classmethod
    def from_str(cls, s: str) -> Ipv4Addr: ...
    @classmethod
    def parse(cls, s: str | bytes) -> Ipv4Addr: ...
    @classmethod
    def from_bits(cls, bits: int) -> Ipv4Addr: ...
    @classmethod
    def from_octets(cls, b: bytes) -> Ipv4Addr: ...

    # =======================================================================
    # METHODS
    # =======================================================================
    def to_ipaddr(self) -> IpAddr: ...
    def to_socketaddr_v4(self, port: int) -> SocketAddrV4: ...
    def to_socketaddr_v6(
        self, port: int, flowinfo: int = 0, scope_id: int = 0
    ) -> SocketAddrV6: ...

class _Ipv6AddrProperties(t.Protocol):
    # ========================================================================
    # PROPERTIES
    # ========================================================================

    @property
    def is_benchmarking(self) -> bool: ...
    @property
    def is_global(self) -> t.NoReturn: ...
    @property
    def is_ipv4_mapped(self) -> bool: ...
    @property
    def is_loopback(self) -> bool: ...
    @property
    def is_multicast(self) -> bool: ...
    @property
    def is_reserved(self) -> bool: ...
    @property
    def is_shared(self) -> bool: ...
    @property
    def is_unicast(self) -> bool: ...
    @property
    def is_unicast_global(self) -> t.NoReturn: ...
    @property
    def is_unicast_link_local(self) -> bool: ...
    @property
    def is_unique_local(self) -> bool: ...
    @property
    def is_unspecified(self) -> bool: ...

@t.final
class Ipv6Addr(
    _Ipv6AddrProperties,
    _Version6,
    FromStr,
    ToPy[ipaddress.IPv6Address],
    ToPyIpAddress[ipaddress.IPv6Address],
):
    LOCALHOST: Ipv6Addr
    UNSPECIFIED: Ipv6Addr

    @t.overload
    def __init__(
        self, a: int, b: int, c: int, d: int, e: int, f: int, g: int, h: int
    ) -> None: ...
    @t.overload
    def __init__(
        self, iplike: int | str | bytes | Ipv6Addr | ipaddress.IPv6Address
    ) -> None: ...
    def __eq__(self, other: object) -> bool: ...
    def __ne__(self, other: object) -> bool: ...
    def __lt__(self, other: Ipv6Addr) -> bool: ...
    def __le__(self, other: Ipv6Addr) -> bool: ...
    def __gt__(self, other: Ipv6Addr) -> bool: ...
    def __ge__(self, other: Ipv6Addr) -> bool: ...
    def __hash__(self) -> int: ...
    def to_py(self) -> ipaddress.IPv6Address: ...
    @property
    def version(self) -> t.Literal[6]: ...
    @property
    def is_documentation(self) -> t.NoReturn: ...

    # ========================================================================
    # CLASSMETHODS
    # ========================================================================
    @classmethod
    def from_str(cls, s: str) -> Ipv6Addr: ...
    @classmethod
    def parse(cls, s: str | bytes) -> Ipv6Addr: ...
    @classmethod
    def from_bits(cls, bits: int) -> IpAddr: ...

    # =======================================================================
    # METHODS
    # =======================================================================
    def to_ipaddr(self) -> IpAddr: ...
    def to_socketaddr_v4(self, port: int) -> SocketAddrV4: ...
    def to_socketaddr_v6(
        self, port: int, flowinfo: int = 0, scope_id: int = 0
    ) -> SocketAddrV6: ...

@t.final
class IpAddr(
    _Ipv4AddrProperties,
    _Ipv6AddrProperties,
    _Version,
    FromStr,
    ToPy[ipaddress.IPv4Address | ipaddress.IPv6Address],
    ToPyIpAddress[ipaddress.IPv4Address | ipaddress.IPv6Address],
):
    BROADCAST: IpAddr
    LOCALHOST_V4: IpAddr
    UNSPECIFIED_V4: IpAddr
    LOCALHOST_V6: IpAddr
    UNSPECIFIED_V6: IpAddr

    def __init__(
        self,
        iplike: int
        | str
        | bytes
        | Ipv4Addr
        | Ipv6Addr
        | ipaddress.IPv4Address
        | ipaddress.IPv6Address,
    ) -> None: ...
    def __eq__(self, other: object) -> bool: ...
    def __ne__(self, other: object) -> bool: ...
    def __lt__(self, other: IpAddr) -> bool: ...
    def __le__(self, other: IpAddr) -> bool: ...
    def __gt__(self, other: IpAddr) -> bool: ...
    def __ge__(self, other: IpAddr) -> bool: ...
    def __hash__(self) -> int: ...
    def to_py(self) -> ipaddress.IPv4Address | ipaddress.IPv6Address: ...
    def to_ipv4(self) -> Ipv4Addr: ...
    def to_ipv6(self) -> Ipv6Addr: ...

    # =========================================================================
    # CLASSMETHODS
    # =========================================================================
    @classmethod
    def from_str(cls, s: str) -> IpAddr: ...
    @classmethod
    def parse(cls, ip: str | bytes) -> IpAddr: ...

    # ========================================================================
    # PROPERTIES
    # ========================================================================

    @property
    def version(self) -> t.Literal[4, 6]: ...
    @property
    def is_benchmarking(self) -> t.NoReturn: ...
    @property
    def is_ipv4(self) -> bool: ...
    @property
    def is_ipv6(self) -> bool: ...
    @property
    def is_broadcast(self) -> bool: ...
    @property
    def is_documentation(self) -> bool: ...
    @property
    def is_loopback(self) -> bool: ...
    @property
    def is_multicast(self) -> bool: ...
    @property
    def is_private(self) -> bool: ...
    @property
    def is_unspecified(self) -> bool: ...

    # =======================================================================
    # METHODS
    # =======================================================================
    def to_canonical(self) -> IpAddr: ...

@t.final
class SocketAddrV4(
    _Ipv4AddrProperties, _Version4, ToPyIpAddress[ipaddress.IPv4Address]
):
    def __init__(
        self,
        ip: IpAddr | Ipv4Addr | ipaddress.IPv4Address | ipaddress.IPv6Address,
        port: int,
    ) -> None: ...
    def __eq__(self, other: object) -> bool: ...
    def __ne__(self, other: object) -> bool: ...
    def __lt__(self, other: SocketAddrV4) -> bool: ...
    def __le__(self, other: SocketAddrV4) -> bool: ...
    def __gt__(self, other: SocketAddrV4) -> bool: ...
    def __ge__(self, other: SocketAddrV4) -> bool: ...
    def __hash__(self) -> int: ...
    def to_ipaddrv4(self) -> Ipv4Addr: ...
    def to_ipaddr(self) -> IpAddr: ...
    @classmethod
    def from_str(cls, s: str) -> t.Self: ...
    @staticmethod
    def parse(s: str | bytes) -> SocketAddrV4: ...
    @property
    def port(self) -> int: ...
    @property
    def ip(self) -> Ipv4Addr: ...

@t.final
class SocketAddrV6(
    _Ipv6AddrProperties, _Version6, ToPyIpAddress[ipaddress.IPv6Address]
):
    def __init__(
        self,
        ip: IpAddr | Ipv6Addr | ipaddress.IPv4Address | ipaddress.IPv6Address,
        port: int,
    ) -> None: ...
    def __eq__(self, other: object) -> bool: ...
    def __ne__(self, other: object) -> bool: ...
    def __lt__(self, other: SocketAddrV6) -> bool: ...
    def __le__(self, other: SocketAddrV6) -> bool: ...
    def __gt__(self, other: SocketAddrV6) -> bool: ...
    def __ge__(self, other: SocketAddrV6) -> bool: ...
    def __hash__(self) -> int: ...
    def to_ipaddrv6(self) -> Ipv6Addr: ...
    def to_ipaddr(self) -> IpAddr: ...
    @classmethod
    def from_str(cls, s: str) -> t.Self: ...
    @staticmethod
    def parse(s: str | bytes) -> SocketAddrV6: ...
    @property
    def port(self) -> int: ...
    @property
    def ip(self) -> Ipv6Addr: ...
    @property
    def is_documentation(self) -> t.NoReturn: ...

@t.final
class SocketAddr(
    _Ipv4AddrProperties,
    _Ipv6AddrProperties,
    _Version,
    FromStr,
    ToPyIpAddress[ipaddress.IPv4Address | ipaddress.IPv6Address],
):
    def __init__(
        self,
        ip: IpAddr
        | Ipv4Addr
        | Ipv6Addr
        | ipaddress.IPv4Address
        | ipaddress.IPv6Address,
        port: int,
    ) -> None: ...
    def __eq__(self, other: object) -> bool: ...
    def __ne__(self, other: object) -> bool: ...
    def __lt__(self, other: SocketAddr) -> bool: ...
    def __le__(self, other: SocketAddr) -> bool: ...
    def __gt__(self, other: SocketAddr) -> bool: ...
    def __ge__(self, other: SocketAddr) -> bool: ...
    def __hash__(self) -> int: ...
    @classmethod
    def from_str(cls, s: str) -> SocketAddr: ...
    @staticmethod
    def parse(s: str | bytes) -> SocketAddr: ...
    def to_ipaddr(self) -> IpAddr: ...
    @property
    def is_ipv4(self) -> bool: ...
    @property
    def is_ipv6(self) -> bool: ...
    @property
    def ip(self) -> IpAddr: ...
    @property
    def port(self) -> int: ...
