import csv
import json
import math
from pathlib import Path
import unittest

ROOT = Path(__file__).resolve().parents[1]
CONFIG_PATH = ROOT / "config.json"
VOCAB_PATH = ROOT / "practicejapanese" / "data" / "Vocab.csv"


class SentenceCacheSettingsTest(unittest.TestCase):
    def setUp(self):
        with CONFIG_PATH.open(encoding="utf-8") as fh:
            self.config = json.load(fh)
        with VOCAB_PATH.open(encoding="utf-8") as fh:
            self.vocab_rows = list(csv.DictReader(fh))

    def test_sentence_prefetch_budget(self):
        settings = self.config.get("sentence_cache_settings", {})
        interval = max(1, int(settings.get("prefetch_interval_seconds", 30)))
        batch = max(1, int(settings.get("batch_fetch_size", 1)))
        api_limit = max(1, int(settings.get("api_limit", 1)))
        min_sentences = max(1, int(settings.get("min_sentences", 1)))

        fill_scores = []
        for row in self.vocab_rows:
            raw = (row.get("FillingScore") or "0").strip()
            fill_scores.append(int(raw) if raw else 0)
        min_score = min(fill_scores)
        backlog = sum(1 for score in fill_scores if score == min_score)

        batches_needed = math.ceil(backlog / batch)
        estimated_minutes = batches_needed * (interval / 60)
        requests_per_minute = batch * (60 / interval)

        self.assertGreaterEqual(
            api_limit,
            min_sentences,
            "api_limit must fetch at least as many sentences as we plan to cache",
        )
        self.assertLessEqual(
            estimated_minutes,
            30,
            f"Prefetch warm-up would take {estimated_minutes:.1f} minutes (>30)",
        )
        self.assertLessEqual(
            requests_per_minute,
            20,
            f"Prefetch uses {requests_per_minute:.1f} API calls per minute (>20)",
        )


if __name__ == "__main__":
    unittest.main()
