"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const lambda = require("@aws-cdk/aws-lambda");
const sns = require("@aws-cdk/aws-sns");
const ec2 = require("@aws-cdk/aws-ec2");
const lib_1 = require("../lib");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
// --------------------------------------------------------------
// Test deployment with new Lambda function
// --------------------------------------------------------------
test('Test deployment with new Lambda function', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    new lib_1.LambdaToSns(stack, 'lambda-to-sns-stack', {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            environment: {
                LAMBDA_NAME: 'deployed-function'
            }
        }
    });
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Assertion 2
    expect(stack).toHaveResourceLike("AWS::Lambda::Function", {
        Environment: {
            Variables: {
                LAMBDA_NAME: 'deployed-function',
                SNS_TOPIC_ARN: {
                    Ref: 'lambdatosnsstackSnsTopic6292A14A'
                },
                SNS_TOPIC_NAME: {
                    'Fn::GetAtt': [
                        'lambdatosnsstackSnsTopic6292A14A',
                        'TopicName'
                    ]
                }
            }
        }
    });
    expect(stack).toHaveResource("AWS::SNS::Topic", {
        KmsMasterKeyId: {
            "Fn::Join": [
                "",
                [
                    "arn:",
                    {
                        Ref: "AWS::Partition"
                    },
                    ":kms:",
                    {
                        Ref: "AWS::Region"
                    },
                    ":",
                    {
                        Ref: "AWS::AccountId"
                    },
                    ":alias/aws/sns"
                ]
            ]
        }
    });
});
// --------------------------------------------------------------
// Test deployment with existing existingTopicObj
// --------------------------------------------------------------
test('Test deployment with existing existingTopicObj', () => {
    // Stack
    const stack = new core_1.Stack();
    const topic = new sns.Topic(stack, 'MyTopic', {
        topicName: "custom-topic"
    });
    // Helper declaration
    new lib_1.LambdaToSns(stack, 'lambda-to-sns-stack', {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            environment: {
                LAMBDA_NAME: 'override-function'
            }
        },
        existingTopicObj: topic
    });
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Assertion 2
    assert_1.expect(stack).to(assert_1.haveResource("AWS::SNS::Topic", {
        TopicName: "custom-topic"
    }));
});
// --------------------------------------------------------------
// Test deployment with imported encryption key
// --------------------------------------------------------------
test('override topicProps', () => {
    const stack = new core_1.Stack();
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler'
        },
        topicProps: {
            topicName: "custom-topic"
        }
    };
    new lib_1.LambdaToSns(stack, 'test-sns-lambda', props);
    assert_1.expect(stack).to(assert_1.haveResource("AWS::SNS::Topic", {
        TopicName: "custom-topic"
    }));
});
// --------------------------------------------------------------
// Test the getter methods
// --------------------------------------------------------------
test('Test the properties', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    const pattern = new lib_1.LambdaToSns(stack, 'lambda-to-sns-stack', {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        }
    });
    // Assertion 1
    const func = pattern.lambdaFunction;
    expect(func).toBeDefined();
    // Assertion 2
    const topic = pattern.snsTopic;
    expect(topic).toBeDefined();
});
// --------------------------------------------------------------
// Test minimal deployment that deploys a VPC without vpcProps
// --------------------------------------------------------------
test("Test minimal deployment that deploys a VPC without vpcProps", () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    new lib_1.LambdaToSns(stack, "lambda-to-sns-stack", {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: "index.handler",
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        deployVpc: true,
    });
    expect(stack).toHaveResource("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdatosnsstackReplaceDefaultSecurityGroupsecuritygroup64D1B1DA",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B",
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055",
                },
            ],
        },
    });
    expect(stack).toHaveResource("AWS::EC2::VPC", {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
    expect(stack).toHaveResource("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Interface",
    });
    expect(stack).toCountResources("AWS::EC2::Subnet", 2);
    expect(stack).toCountResources("AWS::EC2::InternetGateway", 0);
});
// --------------------------------------------------------------
// Test minimal deployment that deploys a VPC w/vpcProps
// --------------------------------------------------------------
test("Test minimal deployment that deploys a VPC w/vpcProps", () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    new lib_1.LambdaToSns(stack, "lambda-to-sns-stack", {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: "index.handler",
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        vpcProps: {
            enableDnsHostnames: false,
            enableDnsSupport: false,
            cidr: "192.68.0.0/16",
        },
        deployVpc: true,
    });
    expect(stack).toHaveResource("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdatosnsstackReplaceDefaultSecurityGroupsecuritygroup64D1B1DA",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B",
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055",
                },
            ],
        },
    });
    expect(stack).toHaveResource("AWS::EC2::VPC", {
        CidrBlock: "192.68.0.0/16",
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
    expect(stack).toHaveResource("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Interface",
    });
    expect(stack).toCountResources("AWS::EC2::Subnet", 2);
    expect(stack).toCountResources("AWS::EC2::InternetGateway", 0);
});
// --------------------------------------------------------------
// Test minimal deployment with an existing VPC
// --------------------------------------------------------------
test("Test minimal deployment with an existing VPC", () => {
    // Stack
    const stack = new core_1.Stack();
    const testVpc = new ec2.Vpc(stack, "test-vpc", {});
    // Helper declaration
    new lib_1.LambdaToSns(stack, "lambda-to-sns-stack", {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: "index.handler",
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        existingVpc: testVpc,
    });
    expect(stack).toHaveResource("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdatosnsstackReplaceDefaultSecurityGroupsecuritygroup64D1B1DA",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "testvpcPrivateSubnet1Subnet865FB50A",
                },
                {
                    Ref: "testvpcPrivateSubnet2Subnet23D3396F",
                },
            ],
        },
    });
    expect(stack).toHaveResource("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Interface",
    });
});
// --------------------------------------------------------------
// Test minimal deployment with an existing VPC and existing Lambda function not in a VPCs
//
// buildLambdaFunction should throw an error if the Lambda function is not
// attached to a VPC
// --------------------------------------------------------------
test("Test minimal deployment with an existing VPC and existing Lambda function not in a VPC", () => {
    // Stack
    const stack = new core_1.Stack();
    const testLambdaFunction = new lambda.Function(stack, 'test-lamba', {
        runtime: lambda.Runtime.NODEJS_10_X,
        handler: "index.handler",
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    });
    const testVpc = new ec2.Vpc(stack, "test-vpc", {});
    // Helper declaration
    const app = () => {
        // Helper declaration
        new lib_1.LambdaToSns(stack, "lambda-to-sns-stack", {
            existingLambdaObj: testLambdaFunction,
            existingVpc: testVpc,
        });
    };
    // Assertion
    expect(app).toThrowError();
});
// --------------------------------------------------------------
// Test bad call with existingVpc and deployVpc
// --------------------------------------------------------------
test("Test bad call with existingVpc and deployVpc", () => {
    // Stack
    const stack = new core_1.Stack();
    const testVpc = new ec2.Vpc(stack, "test-vpc", {});
    const app = () => {
        // Helper declaration
        new lib_1.LambdaToSns(stack, "lambda-to-sns-stack", {
            lambdaFunctionProps: {
                runtime: lambda.Runtime.NODEJS_10_X,
                handler: "index.handler",
                code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            },
            existingVpc: testVpc,
            deployVpc: true,
        });
    };
    // Assertion
    expect(app).toThrowError();
});
// --------------------------------------------------------------
// Test lambda function custom environment variable
// --------------------------------------------------------------
test('Test lambda function custom environment variable', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    new lib_1.LambdaToSns(stack, 'lambda-to-sns-stack', {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        topicArnEnvironmentVariableName: 'CUSTOM_TOPIC_ARN',
        topicNameEnvironmentVariableName: 'CUSTOM_TOPIC_NAME'
    });
    // Assertion
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Handler: 'index.handler',
        Runtime: 'nodejs14.x',
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
                CUSTOM_TOPIC_ARN: {
                    Ref: 'lambdatosnsstackSnsTopic6292A14A'
                },
                CUSTOM_TOPIC_NAME: {
                    'Fn::GetAtt': [
                        'lambdatosnsstackSnsTopic6292A14A',
                        'TopicName'
                    ]
                }
            }
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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