from django.apps import apps
from django.db import models
from pathlib import Path
from hyperx.logger.hx_logger import *
_logger = load_logger("htmx_backend_dashboard")
_logger.info("htmx_backend_dashboard initialized")
import os, sys, re, json, hmac, hashlib
from datetime import datetime

FIELD_TYPE_MAP = {
    models.CharField: "text",
    models.TextField: "textarea",
    models.EmailField: "email",
    models.IntegerField: "number",
    models.FloatField: "number",
    models.BooleanField: "checkbox",
    models.DateField: "date",
    models.DateTimeField: "datetime-local",
}

def generate_dashboard(app_label: str, output: Path, silent: bool = False):
    """Generate HyperX <hx:panel> CRUD dashboard for all models in an app."""
    models_in_app = apps.get_app_config(app_label).get_models()
    dashboards = []

    for model in models_in_app:
        dashboards.append(build_dashboard(model))

    html = "\n\n".join(dashboards)
    Path(output).write_text(html)
    if not silent:
        print(f"✅ Dashboard written → {output}")

def build_dashboard(model):
    model_name = model.__name__
    endpoint = model_name.lower() + "s"
    fields = [f for f in model._meta.fields if not f.primary_key]

    form_html = "\n".join(build_field_tag(f) for f in fields)
    table_fields = ",".join(f.name for f in fields)

    return f"""# Auto-generated by HyperX build CLI
# Timestamp: {datetime.now().isoformat()}

{{% hx %}}
<hx:panel id="{endpoint}-dashboard" class="container mt-4">
  <hx:crud model="{model_name}" endpoint="{endpoint}" target="#crud-zone">
      <hx:form post="{endpoint}:create" target="#crud-zone" toast="{model_name} saved!">
          {form_html}
          <button type="submit" class="btn btn-primary mt-2">Save</button>
      </hx:form>
      <hx:tablecrud source="{endpoint}" fields="{table_fields}"
                    actions="edit,delete,export" per-page="10"
                    paginate="true" target="#crud-zone" />
  </hx:crud>
</hx:panel>
{{% endhx %}}
""".strip()

def build_field_tag(field):
    label = field.verbose_name.title()
    name = field.name
    field_type = next((t for k, t in FIELD_TYPE_MAP.items() if isinstance(field, k)), "text")
    required = "true" if not field.blank else "false"
    return f'<hx:field label="{label}" name="{name}" type="{field_type}" required="{required}" />'
